{
  FSCANNER.PAS   v2.02
  Copyright (C) 1998, 1999  Alexander Burlakov
            mailto:hunterml@fnmail.com
            http://www.chat.ru/~hunterml

         TBaseScanner - base class for file scanners creation. Scans
                        all or specified drives for files selected by
                        file masks. Performs thread based search for
                        save system resources. You acn tune Priority
                        of the searching process manually.

      TContextScanner - based on TBaseScanner and inherits all it's
                        properies and methods, but also performs
                        Context search in each file appropriated specified
                        file mask. Context can be simple text string or
                        array of any symbols [0..255].

  Changed 24 Jul 1999
     - Unit Masks.pas no longer needed. TMask class now integrated
       in this unit.

  Changed 16 Jul 1999
     - Added ContextAsArray property and context parsing algorithm;
     - Added BufferSizeKb property for user defined buffer size;
     - Added FirstEntryOnly property. If this property is set to True
       Scanner scans every file for first context entry only.
     - Improved context search speed (algorithm completly rewritten);
     - Fixed bug with ScanThread start.

  If anyone using my components with Windows NT please, inform me how it
  works. I really need this information since I still unable to test this
  unit with NT.

  Please, email me your comments or/and suggestions.
  Thanx.
}

unit FScanner;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TScanStrategy = (ssGlobal, ssSpecifiedDisk, ssSpecifiedDir);
  TScanDrive   = (sdHard, sdCDROM, sdRAM, sdNetwork, sdFloppy);
  TScanDrives = set of TScanDrive;
  TFileAttribute = (faReadOnly, faHidden, faSysFile, faVolumeID,
                    faDirectory, faArchive);
  TFileAttributes = set of TFileAttribute;

  TDoFileFound = procedure(AName, APath : string; AAttr : integer;
                           ASize : longint; ADateTime : TDateTime) of object;
  TDoContextFound = procedure(AFileName : string; ASize, APos : longint) of object;
  TDoProcess = procedure(ACount, ASize : cardinal; ACurrent : string;
                         ACurrentSize, ACurrentPos : cardinal) of object;
  TDoProcessDone = procedure(ACount, ASize : cardinal) of object;

type
  TMask = class
  private
    FMask: Pointer;
    FSize: Integer;
  public
    constructor Create(const MaskValue: string);
    destructor Destroy; override;
    function Matches(const Filename: string): Boolean;
  end;


  TScanThread = class(TThread)
  private
    FDoFileFound : TDoFileFound;
    FDoProcess : TDoProcess;
    FRecursive : boolean;
    FScanStrategy : TScanStrategy;
    FScanDrives :TScanDrives;
    FDir : string;
    FAttr : integer;
    FFiles : string;
    FFindData : TWin32FindData;
    FStatus : integer;
    procedure DoScan; virtual;
    procedure DoProcess; virtual;
    procedure DoFileFound; virtual;
    procedure BeforeFileFound; virtual;
    procedure ScanDirectory(const ADir : string; const AAttr : integer;
                            AMasks : TCollection);
  protected
    procedure Execute; override;
  public
    FCount : cardinal;
    FSize, FTotalSize, FScannedSize : cardinal;
    FCurrent : string;
    constructor Create(ADoFileFound : TDoFileFound; ADoProcess : TDoProcess;
                       ARecursive: boolean; AScanStrategy : TScanStrategy;
                       AScanDrives: TScanDrives; ADir, AFiles : string; AAttr : integer);
    function GetStatus  : Integer;
  end;

  TBaseScanner = class(TComponent)
  private
    { Private declarations }
    FFiles,
    FDir : string;
    FAttr : TFileAttributes;
    FScanStrategy : TScanStrategy;
    FScanDrives : TScanDrives;
    FRecursive : boolean;
    FPriority : TThreadPriority;
    FOnProcess : TDoProcess;
    FOnFileFound : TDoFileFound;
    FOnProcessDone : TDoProcessDone;
    FScanThread : TScanThread;
    function GetIntegerAttr : integer;
    procedure DoScanThdDone(Sender: TObject);
    function GetInProcess : boolean;
    function PrepareThread : TScanThread; virtual;
  protected
    { Protected declarations }
    procedure DefOnProcess(ACount, ASize : cardinal; ACurrent : string;
                           ACurrentSize, ACurrentPos : cardinal);  virtual;
    procedure DefOnFileFound(AName, APath : string; AAttr : integer;
                           ASize : longint; ADateTime : TDateTime); virtual;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    procedure Scan; virtual;
    procedure Stop; virtual;
    property InProcess : boolean read GetInProcess;
  published
    { Published declarations }
    property Directory : string read FDir write FDir;
    property Files : string read FFiles write FFiles;
    property ScanStrategy : TScanStrategy read FScanStrategy write FScanStrategy;
    property ScanDrives : TScanDrives read FScanDrives write FScanDrives;
    property Attributes : TFileAttributes read FAttr write FAttr stored true;
    property Recursive : boolean read FRecursive write FRecursive;
    property Priority : TThreadPriority read FPriority write FPriority;
    property OnProcess : TDoProcess read FOnProcess write FOnProcess;
    property OnProcessDone : TDoProcessDone read FOnProcessDone write FOnProcessDone;
    property OnFileFound : TDoFileFound read FOnFileFound write FOnFileFound;
  end;

const
// Context length limit
  MAX_CONTEXT_LEN       = 255;
// Symbols lower MAX_SYMBOL_TO_REPLACE will be replaced by
// ReplaceChar in ContextAsString
  MAX_SYMBOL_TO_REPLACE = #32;

// Buffer size limits for context search
  MAX_BUFFER_SIZE       = 1024*8; // Max buffer size 8Mb
  MIN_BUFFER_SIZE       = 1;      // Min buffer size 1Kb

type
  TContextArray = array[1..MAX_CONTEXT_LEN] of char;
  PContextArray = ^TContextArray;

  TContext = class(TPersistent)
  private
    FContext : TContextArray;
    FUpperCasedContext : TContextArray;
    FCaseSensitive : boolean;
    FRChar : char;
    FContextLen : integer;
    FBufferSize : integer;
    FFirstEntry : boolean;
    Busy : boolean; // Used to lock TContext while scanning for files
    procedure SetArrContext(Value : PContextArray);
    procedure SetStrContext(Value : string);
    procedure SetBufferSize(Value : integer);
    procedure SetContextLen(Value : integer);
    procedure SetCaseSensitive(Value : boolean);
    procedure SetRChar(Value : Char);
    procedure SetFirstEntry(Value : boolean);
    function GetArrContext : PContextArray;
    function GetStrContext : string;
    function GetEmpty : boolean;
    procedure FillUppercased;
  public
    constructor Create;
    procedure Assign(Source: TPersistent); override;
    function CompareContexts(AContext : PContextArray; ALen : integer) : boolean;
    property ContextAsArray : PContextArray read GetArrContext write SetArrContext;
    property Empty : boolean read GetEmpty;
    property ContextLength : integer read FContextLen write SetContextLen;
  published
    property ReplaceChar : char read FRChar write SetRChar;
    property ContextAsString : string read GetStrContext write SetStrContext;
    property CaseSensitive : boolean read FCaseSensitive write SetCaseSensitive;
    property BufferSizeKb : integer read FBufferSize write SetBufferSize;
    property FirstEntryOnly : boolean read FFirstEntry write SetFirstEntry;
  end;

  TContextScanThread = class(TScanThread)
  private
    FContext : TContext;
    FDoContextFound : TDoContextFound;
    FBuf : pointer;
    FFilePos : longint;
    FFileSize : longint;
    procedure DoContextFound; virtual;
    procedure DoProcess; override;
    procedure BeforeFileFound; override;
    procedure ScanFile(AFile : string);
  public
    constructor Create(AContext : TContext; ADoContextFound : TDoContextFound;
                       ADoFileFound : TDoFileFound; ADoProcess : TDoProcess;
                       ARecursive: boolean; AScanStrategy : TScanStrategy;
                       AScanDrives: TScanDrives; ADir, AFiles : string; AAttr : integer);
    destructor Destroy; override;
  end;

  TContextScanner = class(TBaseScanner)
  private
    { Private declarations }
    FContext : TContext;
    FOnContextFound : TDoContextFound;
    function GetContext : TContext;
    procedure SetContext(Value : TContext);
    function PrepareThread : TScanThread; override;
  protected
    { Protected declarations }
    procedure DefOnContextFound(AFileName : string;
                       ASize, APos : longint); virtual;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property Context : TContext read GetContext write SetContext;
    property OnContextFound : TDoContextFound read FOnContextFound
             write FOnContextFound;
  end;

procedure Register;

function SlashSep(const Path, S: String): String;
function MatchesMask(const Filename, Mask: string): Boolean;


implementation
uses ShellAPI, FileCtrl;

type
  TMultiMask = class(TCollectionItem)
    FMask : TMask;
  public
    destructor Destroy; override;
  end;

const
  MaxCards = 30;
  tsOk               =  0;
  tsError            = -1;

function SlashSep(const Path, S: String): String;
begin
  Result := '';
  if Path = '' then
  begin
    Result := S;
    exit;
  end;
  if AnsiLastChar(Path)^ <> '\' then
    Result := Path + '\' + S
  else
    Result := Path + S;
end;

function ConvertStringToArray(AContext : PContextArray;
                              AStr : string): integer;
var
  i : integer;
begin
  Result := Length(AStr);
  if Result < 1 then exit;
  for i := 1 to Result do
      AContext^[i] := Astr[i];
end;

function ConvertArrayToString(AContext : PContextArray;
                              AContextLen : integer;
                              ARChar : char) : string;
var
  i : integer;
begin
  Result := '';
  if (AContextLen < 1) or (AContext = nil) then exit;
  for i := 1 to AContextLen do
    if AContext^[i] > MAX_SYMBOL_TO_REPLACE then
       Result := Result + Char(AContext^[i]) else
       Result := Result + ARChar;
end;

{procedure ConvertToUpperCase(AContext : PContextArray; ALen : integer);

begin
  for i := 1 to ALen do
      AContext^[i] := UpChar(AContext^[i]);
end;
}

{ TContext }

constructor TContext.Create;
begin
  inherited Create;
  FillChar(FContext, MAX_CONTEXT_LEN, $00);
  FContextLen := 0;
  FCaseSensitive := false;
  FRChar := ' ';
  FBufferSize := 64; { 64k buffer by default }
  Busy := false;
end;

function TContext.CompareContexts(AContext : PContextArray; ALen : integer) : boolean;
var i : integer;
begin
  Result := false;
  if ALen <> FContextLen then exit;
  if not FCaseSensitive then
  begin
    for i := 1 to FContextLen do
       if FUppercasedContext[i] <> UpCase(AContext^[i]) then exit
  end else
  begin
    for i := 1 to FContextLen do
       if FContext[i] <> AContext^[i] then exit;
  end;
  Result := true;
end;

procedure TContext.Assign;
begin
  if (Source is TContext) and not Busy then
  begin
    if Empty then
    begin
      FillChar(FContext, MAX_CONTEXT_LEN, $00);
      FContextLen := 0;
    end else
    begin
      Move(TContext(Source).ContextAsArray^,
           FContext, TContext(Source).ContextLength);
      FContextLen := TContext(Source).ContextLength;
    end;
    FCaseSensitive := TContext(Source).CaseSensitive;
    FRChar := TContext(Source).ReplaceChar;
  end;
  inherited Assign(Source);
end;

procedure TContext.FillUppercased;
var i : integer;
begin
  for i := 1 to MAX_CONTEXT_LEN do
     FUppercasedContext[i] := UpCase(FContext[i]);
end;

procedure TContext.SetArrContext(Value : PContextArray);
begin
  if (Value <> nil) and (Value <> @FContext) and (not Busy) then
  begin
    Move(Value^, FContext, MAX_CONTEXT_LEN);
    FillUppercased;
  end;
end;

procedure TContext.SetStrContext(Value : string);
begin
  if not Busy then
  begin
    FContextLen := ConvertStringToArray(@FContext, Value);
    FillUppercased;
  end;
end;

procedure TContext.SetBufferSize(Value : integer);
begin
  if (Value >= MIN_BUFFER_SIZE) and
     (Value <= MAX_BUFFER_SIZE) and not Busy then
     FBufferSize := Value else
       raise Exception.Create(
            Format('Buffer size must be integer between %d and %d',
                   [MIN_BUFFER_SIZE, MAX_BUFFER_SIZE]));
end;

procedure TContext.SetContextLen(Value : integer);
begin
  if (Value >= 0) and not Busy then
     FContextLen := Value;
end;

procedure TContext.SetCaseSensitive(Value : boolean);
begin
  if not Busy then FCaseSensitive := Value;
end;

procedure TContext.SetRChar(Value : Char);
begin
  if (Value > MAX_SYMBOL_TO_REPLACE) and (not Busy) then
      FRChar := Value;
end;

procedure TContext.SetFirstEntry(Value : boolean);
begin
  if not Busy then FFirstEntry := Value;
end;

function TContext.GetArrContext : PContextArray;
begin
  if Empty then Result := nil
     else Result := @FContext;
end;

function TContext.GetStrContext : string;
begin
  if Empty then Result := '<Empty>' else
     Result := ConvertArrayToString(@FContext, FContextLen, FRChar);
end;

function TContext.GetEmpty : boolean;
begin
  Result := (FContextLen = 0);
end;


{ TContextScanner }

constructor TContextScanner.Create;
begin
  inherited Create(AOwner);
  FContext := TContext.Create;
  FOnContextFound := DefOnContextFound;
end;

destructor TContextScanner.Destroy;
begin
  FContext.Free;
  inherited Destroy;
end;

function TContextScanner.PrepareThread : TScanThread;
begin
  Result := TContextScanThread.Create(FContext, FOnContextFound, FOnFileFound, FOnProcess, FRecursive,
                                    FScanStrategy, FScanDrives, FDir, FFiles, GetIntegerAttr);
end;

function TContextScanner.GetContext : TContext;
begin
  Result := FContext;
end;

procedure TContextScanner.SetContext(Value : TContext);
begin
  FContext.Assign(Value);
end;

procedure TContextScanner.DefOnContextFound;
begin
  { do nothing }
end;


procedure CreateMasks(ACol : TCollection; AStr : string);
var
  P : TMultiMask;
  i : integer;
  S : string;
begin
  i := 1; S := '';
  while i <= Length(AStr) do
  begin
    case AnsiChar(AStr[i]) of
     ' ' : begin
             if S = '' then Inc(i) else
             begin
               S := S + AStr[i];
               if i = Length(AStr) then
               begin
                 while AnsiLastChar(S)^ = ' ' do
                     S := Copy(S, 1, Length(S)-1);
                 P := TMultiMask(ACol.Add);
                 P.FMask := TMask.Create(S);
                 Exit;
               end else  Inc(i);
             end;
           end;
     ';' : begin
             while AnsiLastChar(S)^ = ' ' do
                  S := Copy(S, 1, Length(S)-1);
             P := TMultiMask(ACol.Add);
             P.FMask := TMask.Create(S);
             S := '';
             if i = Length(AStr) then Exit else Inc(i);
           end;
    else   begin
             S := S + AStr[i];
             if i = Length(AStr) then
             begin
               while AnsiLastChar(S)^ = ' ' do
                    S := Copy(S, 1, Length(S)-1);
               P := TMultiMask(ACol.Add);
               P.FMask := TMask.Create(S);
               Exit;
             end else  Inc(i);
           end;
    end;
  end;
end;

{ TMultiMask }

destructor TMultiMask.Destroy;
begin
  if Assigned(FMask) then FMask.Free;
  inherited;
end;

{ TBaseScanner }

constructor TBaseScanner.Create;
begin
  inherited Create(AOwner);
  FFiles := '*';
  FDir := 'C:\';
  FAttr := [faArchive];
  FScanDrives := [sdHard, sdCDROM, sdRAM, sdNetwork];
  FScanStrategy := ssGlobal;
  FRecursive := true;
  FPriority := tpTimeCritical;
  FOnProcess := DefOnProcess;
  FOnFileFound := DefOnFileFound;
  FScanThread := nil;
end;

function TBaseScanner.PrepareThread : TScanThread;
begin
  Result := TScanThread.Create(FOnFileFound, FOnProcess, FRecursive,
                                    FScanStrategy, FScanDrives, FDir, FFiles, GetIntegerAttr);
end;

procedure TBaseScanner.Scan;
begin
  if FScanThread <> nil then exit; { safety }
  FScanThread := PrepareThread;
  if FScanThread = nil then exit;
  with FScanThread do
  begin
    OnTerminate := DoScanThdDone;
    FreeOnTerminate := true;
    Priority := FPriority;
    Resume;
  end;
end;

function TBaseScanner.GetInProcess : boolean;
begin
  Result := (FScanThread <> nil);
end;

procedure TBaseScanner.Stop;
begin
  if not InProcess then exit; { safety }
  FScanThread.Terminate;
end;

procedure TBaseScanner.DoScanThdDone(Sender: TObject);
begin
  with Sender as TScanThread do
  begin
    if Assigned(FOnProcessDone) then
       FOnProcessDone(FCount, FSize);
  end;
  FScanThread := nil;
end;

function TBaseScanner.GetIntegerAttr : integer;
begin
  Result := 0;
  if faReadOnly in FAttr then Result := Result + FILE_ATTRIBUTE_READONLY;
  if faHidden in FAttr then Result := Result + FILE_ATTRIBUTE_HIDDEN;
  if faSysFile in FAttr then Result := Result + FILE_ATTRIBUTE_SYSTEM;
  if faVolumeID in FAttr then Result := Result + SysUtils.faVolumeID;
  if faDirectory in FAttr then Result := Result + FILE_ATTRIBUTE_DIRECTORY;
  if faArchive in FAttr then Result := Result + FILE_ATTRIBUTE_ARCHIVE;
end;

procedure TBaseScanner.DefOnProcess;
begin
  { do nothing }
end;

procedure TBaseScanner.DefOnFileFound;
begin
  { do nothing }
end;

{ TContextScanThread }

constructor TContextScanThread.Create(AContext : TContext; ADoContextFound : TDoContextFound;
                       ADoFileFound : TDoFileFound; ADoProcess : TDoProcess;
                       ARecursive: boolean; AScanStrategy : TScanStrategy;
                       AScanDrives: TScanDrives; ADir, AFiles : string; AAttr : integer);
begin
  FContext := AContext;
  FContext.Busy := true;
  FDoContextFound := ADoContextFound;
  GetMem(FBuf, FContext.BufferSizeKb*1024);
  inherited Create(ADoFileFound, ADoProcess, ARecursive, AScanStrategy,
                   AScanDrives, ADir, AFiles, AAttr);
end;

destructor TContextScanThread.Destroy;
begin
  FreeMem(FBuf, FContext.BufferSizeKb*1024);
  FContext.Busy := false;
  inherited;
end;

procedure TContextScanThread.DoContextFound;
begin
  if Assigned(FDoContextFound) then
     FDoContextFound(SlashSep(FCurrent, FFindData.cFileName),
                     FFileSize, FFilePos);
end;

procedure TContextScanThread.DoProcess;
begin
  if Assigned(FDoProcess) then
     FDoProcess(FCount, FSize, FCurrent, FFileSize, FFilePos);
end;

procedure TContextScanThread.BeforeFileFound;
begin
  inherited;
  if not FContext.Empty then
     ScanFile(SlashSep(FCurrent, FFindData.cFileName));
end;

procedure TContextScanThread.ScanFile(AFile : string);
var
  Range : longint;
  FS, RS : TStream;
  Skip : boolean;
  FirstCh : char;

  procedure ScanBuffer;
  var i : longint;

    procedure CheckName;
    var
      ChArray : PContextArray;
      AChArray : TContextArray;
    begin
      if (i+FContext.FContextLen) > Pred(Range) then
      begin
        RS.Seek(FS.Position-Range+i, soFromBeginning);
        if RS.Read(AChArray, FContext.FContextLen) <>
                   FContext.FContextLen then exit;
        ChArray := @AChArray
      end else
        ChArray := Pointer(Longint(FBuf)+i);
      if FContext.CompareContexts(ChArray, FContext.FContextLen) then
      begin
        FFilePos := FS.Position-Range+i;
        Synchronize(DoContextFound);
        if FContext.FFirstEntry then Skip := true;
      end;
    end;

  begin
    if FContext.FCaseSensitive then
      for i := 0 to Pred(Range) do
      begin
        if Char(Pointer(Longint(FBuf)+i)^) = FirstCh then CheckName;
        if Skip or Terminated then exit;
      end else
      for i := 0 to Pred(Range) do
      begin
        if UpCase(Char(Pointer(Longint(FBuf)+i)^)) = FirstCh then CheckName;
        if Skip or Terminated then exit;
      end;
  end;

begin
  try
   FS := TFileStream.Create(AFile, fmOpenRead or fmShareDenyWrite);
   RS := TFileStream.Create(AFile, fmOpenRead or fmShareDenyWrite);
  except
   exit;
  end;
  FFileSize := FS.Size; Skip := false;
  if FContext.FCaseSensitive then
     FirstCh := FContext.FContext[1] else
     FirstCh := FContext.FUppercasedContext[1];
  try
    with FS do
     while (Position < pred(Size)) and (not Terminated)
           and (not Skip) do
     begin
       Range := Read(FBuf^, FContext.BufferSizeKb*1024);
       ScanBuffer;
       FFilePos := Position;
       Synchronize(DoProcess);
       // Update progress here !
     end;
  finally
    RS.Free;
    FS.Free;
  end;
end;


{ TScanThread }

constructor TScanThread.Create;
begin
  FDoFileFound := ADoFileFound;
  FDoProcess := ADoProcess;
  FDir := ADir;
  FFiles := AFiles;
  FAttr := AAttr;
  FRecursive := ARecursive;
  FScanStrategy := AScanStrategy;
  FScanDrives := AScanDrives;
  inherited Create(true);
end;

function TScanThread.GetStatus : Integer;
begin Result := fStatus; end;

procedure TScanThread.ScanDirectory;
var
  Handle : THandle;
  FindData : TWin32FindData;
  zBuffer : array[0..MAX_PATH] of char;
  LastOk : boolean;

  function MultiMatches(N : string) : boolean;
  var i : integer;
  begin
    MultiMatches := true;
    for i := 0 to Pred(AMasks.Count) do
        if TMultiMask(AMasks.Items[i]).FMask.Matches(N) then exit;
    MultiMatches := false;
  end;

begin
  if FStatus <> tsOk then exit;
  FCurrent := ADir; LastOk := true;
  Synchronize(DoProcess);
  Handle := FindFirstFile(StrPCopy(zBuffer, SlashSep(ADir, '*.*')), FindData);
  if Handle = INVALID_HANDLE_VALUE then exit; { safety }
  try
    while (Handle <> INVALID_HANDLE_VALUE) and (not Terminated) and LastOk do
    begin
      if (PChar(@FindData.cFileName[0]) <> '.') and
         (PChar(@FindData.cFileName[0]) <> '..') then
      begin
        if (FindData.dwFileAttributes and AAttr = AAttr) and
           MultiMatches(FindData.cFileName) then
        begin
          FFindData := FindData;   FCurrent := ADir;
          if FStatus = tsOk then BeforeFileFound;
        end;
        if FRecursive and (FindData.dwFileAttributes and
           FILE_ATTRIBUTE_DIRECTORY = FILE_ATTRIBUTE_DIRECTORY) then
             ScanDirectory(SlashSep(ADir, FindData.cFileName), AAttr, AMasks);
      end;
      FillChar(FindData, SizeOf(FindData), $00);
      LastOk := FindNextFile(Handle, FindData); { make sure FindClose }
    end;
  finally
    Windows.FindClose(Handle);
  end;
end;

procedure TScanThread.DoScan;
var Masks : TCollection;

  procedure GlobalScan;
  var
    DriveNum: Integer;
    DriveChar: Char;
    DriveType: TDriveType;
    DriveBits: set of 0..25;
  begin
    Integer(DriveBits) := GetLogicalDrives;
    for DriveNum := 0 to 25 do
    begin
      if not (DriveNum in DriveBits) then Continue;
      DriveChar := Char(DriveNum + Ord('a'));
      DriveType := TDriveType(GetDriveType(PChar(DriveChar + ':\')));
      DriveChar := Upcase(DriveChar);
      case DriveType of
        dtFloppy: if sdFloppy in FScanDrives then ScanDirectory(DriveChar+':', FAttr, Masks);
        dtFixed: if sdHard in FScanDrives then ScanDirectory(DriveChar+':', FAttr, Masks);
        dtNetwork: if sdNetwork in FScanDrives then ScanDirectory(DriveChar+':', FAttr, Masks);
        dtCDROM: if sdCDROM in FScanDrives then ScanDirectory(DriveChar+':', FAttr, Masks);
        dtRAM: if sdRAM in FScanDrives then ScanDirectory(DriveChar+':', FAttr, Masks);
      end;
    end;
  end;

begin
  Masks := TCollection.Create(TMultiMask);
  try
    CreateMasks(Masks, FFiles);
    case FScanStrategy of
      ssGlobal : GlobalScan;
      ssSpecifiedDisk : ScanDirectory(Copy(FDir, 1, 2), FAttr, Masks);
      ssSpecifiedDir : ScanDirectory(FDir, FAttr, Masks);
    end; { CASE }
  finally
    Masks.Free; { check if this not free :-))) }
  end;
end;

procedure TScanThread.BeforeFileFound;
begin
  Synchronize(DoFileFound);
end;

procedure TScanThread.DoFileFound;
var
  FFSize : cardinal;
  FFDate : TDateTime;
  LocalFileTime : TFileTime;
  DosTime : integer;
begin
  if Assigned(FDoFileFound) then
  begin
    FFSize := (FFindData.nFileSizeHigh*MAXDWORD)+FFindData.nFileSizeLow;
    Inc(FCount); Inc(FSize, FFSize);

    FileTimeToLocalFileTime(FFindData.ftLastWriteTime, LocalFileTime);
    if not FileTimeToDosDateTime(LocalFileTime, LongRec(DosTime).Hi,
        LongRec(DosTime).Lo) then FStatus := tsError;

    FFDate := FileDateToDateTime(DosTime);
    FDoFileFound(FFindData.cFileName,
               FCurrent,
               FFindData.dwFileAttributes,
               FFSize,
               FFDate);
  end;
end;

procedure TScanThread.DoProcess;
begin
  if Assigned(FDoProcess) then FDoProcess(FCount, FSize, FCurrent, 0, 0);
end;

procedure TScanThread.Execute;
begin
  FStatus := tsOk;
  try
    DoScan;
  except
    on E:Exception do
      begin
        FCurrent := E.Message;
        FStatus := tsError;
      end;
  end;
end;

{ Mask tools }

type
  PMaskSet = ^TMaskSet;
  TMaskSet = set of Char;
  TMaskStates = (msLiteral, msAny, msSet, msMBCSLiteral);
  TMaskState = record
    SkipTo: Boolean;
    case State: TMaskStates of
      msLiteral: (Literal: Char);
      msAny: ();
      msSet: (
        Negate: Boolean;
        CharSet: PMaskSet);
      msMBCSLiteral: (LeadByte, TrailByte: Char);
  end;
  PMaskStateArray = ^TMaskStateArray;
  TMaskStateArray = array[0..128] of TMaskState;

function InitMaskStates(const Mask: string;
  var MaskStates: array of TMaskState): Integer;
var
  I: Integer;
  SkipTo: Boolean;
  Literal: Char;
  LeadByte, TrailByte: Char;
  P: PChar;
  Negate: Boolean;
  CharSet: TMaskSet;
  Cards: Integer;

  procedure InvalidMask;
  begin
    raise Exception.CreateFmt('''%s'' is an invalid mask at (%d)', [Mask,
      P - PChar(Mask) + 1]);
  end;

  procedure Reset;
  begin
    SkipTo := False;
    Negate := False;
    CharSet := [];
  end;

  procedure WriteScan(MaskState: TMaskStates);
  begin
    if I <= High(MaskStates) then
    begin
      if SkipTo then
      begin
        Inc(Cards);
        if Cards > MaxCards then InvalidMask;
      end;
      MaskStates[I].SkipTo := SkipTo;
      MaskStates[I].State := MaskState;
      case MaskState of
        msLiteral: MaskStates[I].Literal := UpCase(Literal);
        msSet:
          begin
            MaskStates[I].Negate := Negate;
            New(MaskStates[I].CharSet);
            MaskStates[I].CharSet^ := CharSet;
          end;
        msMBCSLiteral:
          begin
            MaskStates[I].LeadByte := LeadByte;
            MaskStates[I].TrailByte := TrailByte;
          end;
      end;
    end;
    Inc(I);
    Reset;
  end;

  procedure ScanSet;
  var
    LastChar: Char;
    C: Char;
  begin
    Inc(P);
    if P^ = '!' then
    begin
      Negate := True;
      Inc(P);
    end;
    LastChar := #0;
    while not (P^ in [#0, ']']) do
    begin
      // MBCS characters not supported in msSet!
      if P^ in LeadBytes then
         Inc(P)
      else
      case P^ of
        '-':
          if LastChar = #0 then InvalidMask
          else
          begin
            Inc(P);
            for C := LastChar to UpCase(P^) do Include(CharSet, C);
          end;
      else
        LastChar := UpCase(P^);
        Include(CharSet, LastChar);
      end;
      Inc(P);
    end;
    if (P^ <> ']') or (CharSet = []) then InvalidMask;
    WriteScan(msSet);
  end;

begin
  P := PChar(Mask);
  I := 0;
  Cards := 0;
  Reset;
  while P^ <> #0 do
  begin
    case P^ of
      '*': SkipTo := True;
      '?': if not SkipTo then WriteScan(msAny);
      '[':  ScanSet;
    else
      if P^ in LeadBytes then
      begin
        LeadByte := P^;
        Inc(P);
        TrailByte := P^;
        WriteScan(msMBCSLiteral);
      end
      else
      begin
        Literal := P^;
        WriteScan(msLiteral);
      end;
    end;
    Inc(P);
  end;
  Literal := #0;
  WriteScan(msLiteral);
  Result := I;
end;

function MatchesMaskStates(const Filename: string;
  MaskStates: array of TMaskState): Boolean;
type
  TStackRec = record
    sP: PChar;
    sI: Integer;
  end;
var
  T: Integer;
  S: array[0..MaxCards - 1] of TStackRec;
  I: Integer;
  P: PChar;

  procedure Push(P: PChar; I: Integer);
  begin
    with S[T] do
    begin
      sP := P;
      sI := I;
    end;
    Inc(T);
  end;

  function Pop(var P: PChar; var I: Integer): Boolean;
  begin
    if T = 0 then
      Result := False
    else
    begin
      Dec(T);
      with S[T] do
      begin
        P := sP;
        I := sI;
      end;
      Result := True;
    end;
  end;

  function Matches(P: PChar; Start: Integer): Boolean;
  var
    I: Integer;
  begin
    Result := False;
    for I := Start to High(MaskStates) do
      with MaskStates[I] do
      begin
        if SkipTo then
        begin
          case State of
            msLiteral:
              while (P^ <> #0) and (UpperCase(P^) <> Literal) do Inc(P);
            msSet:
              while (P^ <> #0) and not (Negate xor (UpCase(P^) in CharSet^)) do Inc(P);
            msMBCSLiteral:
              while (P^ <> #0) do
              begin
                if (P^ <> LeadByte) then Inc(P, 2)
                else
                begin
                  Inc(P);
                  if (P^ = TrailByte) then Break;
                  Inc(P);
                end;
              end;
          end;
          if P^ <> #0 then Push(@P[1], I);
        end;
        case State of
          msLiteral: if UpperCase(P^) <> Literal then Exit;
          msSet: if not (Negate xor (UpCase(P^) in CharSet^)) then Exit;
          msMBCSLiteral:
            begin
              if P^ <> LeadByte then Exit;
              Inc(P);
              if P^ <> TrailByte then Exit;
            end;
        end;
        Inc(P);
      end;
    Result := True;
  end;

begin
  Result := True;
  T := 0;
  P := PChar(Filename);
  I := Low(MaskStates);
  repeat
    if Matches(P, I) then Exit;
  until not Pop(P, I);
  Result := False;
end;

procedure DoneMaskStates(var MaskStates: array of TMaskState);
var
  I: Integer;
begin
  for I := Low(MaskStates) to High(MaskStates) do
    if MaskStates[I].State = msSet then Dispose(MaskStates[I].CharSet);
end;

{ TMask }

constructor TMask.Create(const MaskValue: string);
var
  A: array[0..0] of TMaskState;
begin
  FSize := InitMaskStates(MaskValue, A);
  FMask := AllocMem(FSize * SizeOf(TMaskState));
  InitMaskStates(MaskValue, Slice(PMaskStateArray(FMask)^, FSize));
end;

destructor TMask.Destroy;
begin
  if FMask <> nil then
  begin
    DoneMaskStates(Slice(PMaskStateArray(FMask)^, FSize));
    FreeMem(FMask, FSize * SizeOf(TMaskState));
  end;
end;

function TMask.Matches(const Filename: string): Boolean;
begin
  Result := MatchesMaskStates(Filename, Slice(PMaskStateArray(FMask)^, FSize));
end;

function MatchesMask(const Filename, Mask: string): Boolean;
var
  CMask: TMask;
begin
  CMask := TMask.Create(Mask);
  try
    Result := CMask.Matches(Filename);
  finally
    CMask.Free;
  end;
end;


procedure Register;
begin
  RegisterComponents('Technisoft', [TBaseScanner,
                                    TContextScanner]);
end;

end.
