/* daytime.c - set local time from remote daytime or time server
 *
 * for IBM TCP/IP 2.0 for OS/2
 *
 * Author:  Kai Uwe Rommel <rommel@ars.muc.de>
 * Created: Sun Apr 10 1994
 *
 * This code is in the public domain. 
 * Let the author know if you make improvements or fixes, though.
 */
 
static char *rcsid =
"$Id: daytime.c,v 1.8 1996/12/08 11:18:04 rommel Exp rommel $";
static char *rcsrev = "$Revision: 1.8 $";

/*
 * $Log: daytime.c,v $
 * Revision 1.8  1996/12/08 11:18:04  rommel
 * added get_date prototype
 *
 * Revision 1.7  1996/12/08 11:15:07  rommel
 * don't change time within 1 second margin
 * from Ewen McNeill <ewen@naos.co.nz>
 *
 * Revision 1.6  1996/11/29 15:45:00  rommel
 * changes (maxadj) from Ewen McNeill <ewen@naos.co.nz>
 *
 * Revision 1.5  1995/08/20 08:15:10  rommel
 * updated for new emx socket library, IBM compiler
 * fixed minor bugs
 *
 * Revision 1.4  1994/07/17 21:13:26  rommel
 * fixed usage() display again
 *
 * Revision 1.3  1994/07/17 21:10:23  rommel
 * fixed usage() display
 *
 * Revision 1.2  1994/07/17 21:07:57  rommel
 * added daemon mode, continuously updating local time
 *
 * Revision 1.1  1994/07/17 20:45:55  rommel
 * Initial revision
 */

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <time.h>
#include <string.h>

#include "daytime.h"
#include "os.h"

struct options
{
  int interval;
  int dont;
  int maxadj;
  time_t offset;
  service serv;
  protocol proto;
  int portnum;
  char host[64];
};

static struct options opts;

int get_and_set_time(int port, char *node, service serv, protocol proto, 
		     int dont, time_t offs, time_t maxadj)
{
  int sock, bytes, size;
  struct sockaddr_in server;
  struct hostent *host;
  struct linger linger;
  time_t now, here;
  char buffer[64], *request;
  sntp *message;

  switch (serv)
  {

  case DAYTIME:
  case TIME:

    bytes = 1;
    buffer[0] = 0;

    break;

  case SNTP:

    bytes = sizeof(sntp);
    memset(buffer, 0, bytes);

    message = (sntp *) buffer;

    message->flags.version = 1;
    message->flags.mode = 3;

    break;

  }

  if ((host = gethostbyname(node)) == NULL)
    return psock_errno("gethostbyname()"), 1;

  server.sin_family = AF_INET;
  server.sin_port = port;
  server.sin_addr = * (struct in_addr *) (host -> h_addr);

  if (proto == TCP)
  {
    if ((sock = socket(PF_INET, SOCK_STREAM, 0)) < 0)
      return psock_errno("socket(tcp)"), 1;

    if (connect(sock, (struct sockaddr *) &server, sizeof(server)) < 0)
      return psock_errno("connect()"), 1;

    if (serv == SNTP)
    {
      if (send(sock, buffer, bytes, 0) < 0)
	return psock_errno("send()"), 1;
    }

    if ((bytes = recv(sock, buffer, sizeof(buffer), 0)) <= 0)
      return psock_errno("recv()"), 1;
  }
  else
  {
    if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0)
      return psock_errno("socket(udp)"), 1;

    if (sendto(sock, buffer, bytes, 0,
               (struct sockaddr *) &server, sizeof(server)) < 0)
      return psock_errno("sendto()"), 1;

    size = sizeof(server);
    if ((bytes = recvfrom(sock, buffer, sizeof(buffer), 0,
                          (struct sockaddr *) &server, &size)) < 0)
      psock_errno("recvfrom()");
  }

  soclose(sock);

  switch (serv)
  {

  case DAYTIME:

    buffer[bytes] = 0;
    now = get_date(buffer, NULL);

    break;

  case TIME:

    if (bytes != sizeof(now))
      return printf("invalid time value received\n"), 1;

    now = * (time_t *) buffer;
    now = ntohl(now);
    /*
     * The time service returns seconds since 1900/01/01 00:00:00 GMT, not
     * since 1970 as assumed by Unix time_t values.  We need to subtract
     * 70 years' worth of seconds.  Fortunately, RFC 868 tells us that the 
     * time value 2,208,988,800 corresponds to 00:00 1 Jan 1970 GMT.
     */
    now -= 2208988800UL;

    break;

  case SNTP:

    if (message->flags.version < 1 || 3 < message->flags.version)
      return printf("incompatible protocol version on server\n"), 1;
    if (message->flags.leap_ind == 3)
      return printf("server not synchronized\n"), 1;
    if (message->stratum == 0)
      return printf("server not synchronized\n"), 1;

    now = (time_t) message->transmit_time.integer;
    now = ntohl(now);

    if (now == 0)
      return printf("server not synchronized\n"), 1;

    /* same here */
    now -= 2208988800UL;

    break;

  }

  if (now <= 0)
    return printf("invalid time\n"), 1;

  now += offs;
  time(&here);

  if (maxadj)
    if (now < (here - maxadj) || (here + maxadj) < now)
      dont = 1;               /* Too far off: just report it */

  if (abs(now - here) <= 1)
    dont = 1;
  
  if (!dont && stime(&now))
    return printf("invalid time set request\n"), 1;

  printf("time %s %s: (%+ld) %s", dont ? "at" : "set from", 
	 host -> h_name, now - here, ctime(&now));
  fflush(stdout);

  return 0;
}

int usage(void)
{
  printf("\nSet time from remote host, %s"
	 "\n(C) 1997 Kai-Uwe Rommel\n", rcsrev);

#ifdef WIN32
  printf("\nUsage: daytime [-IUndtsu] [-m secs] [-c int] [-p port] host\n"
         "\n  -I       install as a service"
         "\n  -U       uninstall service\n"
#else
  printf("\nUsage: daytime [-ndtsu] [-m secs] [-c int] [-p port] host\n"
#endif
	 "\n  -c int   run continuously, perform action every 'int' seconds"
         "\n  -n       do not set the local clock, display the remote time only"
         "\n  -o offs  adjust the retrieved time by offs seconds before"
	 "\n           setting the local clock"
	 "\n  -m secs  Maximum number of seconds to adjust by\n"
         "\n  -d       use the DAYTIME service (port 13)"
         "\n  -t       use the TIME service (port 37, this is the default)"
         "\n  -s       use the SNTP service (port 123), usually requires -u too"
	 "\n  -u       use the UDP protocol (default is to use TCP)"
	 "\n  -p port  use nonstandard port number\n");
  return 1;
}

int endless(void)
{
  int rc = 0;

  for(;;)
  {
    rc = get_and_set_time(opts.portnum, opts.host, opts.serv, opts.proto, 
			  opts.dont, opts.offset, opts.maxadj);

    if (rc)
      return rc;

    sleep(opts.interval);
  }
}

int main(int argc, char **argv)
{
  int opt, rc, isrv = 0;
  struct servent *port;

  tzset();

  if (sock_init())
    return psock_errno("sock_init()"), 1;

#ifdef WIN32
#define SERVICE_NAME "TimeClient"
  if (started_as_service())
  {
    restore_options(SERVICE_NAME, &opts, sizeof(opts));
    run_as_service(SERVICE_NAME, endless);
  }
#endif

  opts.serv = TIME;
  opts.proto = TCP;
  opts.portnum = -1;

  while ((opt = getopt(argc, argv, "?IUndtsuo:p:c:m:")) != EOF)
    switch (opt)
    {
    case 'n':
      opts.dont = 1;
      break;
    case 'd':
      opts.serv = DAYTIME;
      break;
    case 't':
      opts.serv = TIME;
      break;
    case 's':
      opts.serv = SNTP;
      break;
    case 'u':
      opts.proto = UDP;
      break;
    case 'o':
      opts.offset = atol(optarg);
      break;
    case 'p':
      opts.portnum = htons(atoi(optarg));
      break;
    case 'c':
      opts.interval = atol(optarg);
      break;
    case 'm':
      opts.maxadj = atol(optarg);
      break;
#ifdef WIN32
    case 'I':
      isrv = 1;
      break;
    case 'U':
      if ((rc = install_as_service(SERVICE_NAME, 0)) != 0)
	printf("deinstallation of service failed\n");
      return rc;
#endif
    default:
      return usage();
    }

  if (optind == argc)
    return usage();
  else
    strcpy(opts.host, argv[optind]);

  if (opts.portnum == -1)
  {
    char *s_service = opts.serv == SNTP ? "ntp" :
                      opts.serv == DAYTIME ? "daytime" : "time";
    char *s_proto = opts.proto == TCP ? "tcp" : "udp";

    if ((port = getservbyname(s_service, s_proto)) == NULL)
      return psock_errno("getservbyname()"), 1;

    opts.portnum = port -> s_port;
  }

#ifdef WIN32
  if (isrv)
  {
    if ((rc = install_as_service(SERVICE_NAME, 1)) != 0)
      return printf("installation as service failed\n"), rc;
    else
      return save_options(SERVICE_NAME, &opts, sizeof(opts));
  }
#endif

  if (opts.interval == 0)
    return get_and_set_time(opts.portnum, opts.host, opts.serv, opts.proto, 
			    opts.dont, opts.offset, opts.maxadj);
  else
    return endless();
}

/* end of daytime.c */
