/* daytime.c - set local time from remote daytime or time server
 *
 * for IBM TCP/IP 2.0 for OS/2
 *
 * Author:  Kai Uwe Rommel <rommel@ars.muc.de>
 * Created: Sun Apr 10 1994
 *
 * This code is in the public domain. 
 * Let the author know if you make improvements or fixes, though.
 */
 
static char *rcsid =
"$Id$";
static char *rcsrev = "$Revision$";

/* $Log$ */

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <time.h>
#include <os2.h>

#define OS2
#include "types.h"
#include "utils.h"
#include "sys/socket.h"
#include "netdb.h"

typedef enum {TIME = 0, DAYTIME = 1} service;
typedef enum {UDP = 0, TCP = 1} protocol;

int stime(time_t *newtime)
{
  struct tm *newtm = localtime(newtime);
  DATETIME dt;

  dt.hours   = newtm -> tm_hour;
  dt.minutes = newtm -> tm_min;
  dt.seconds = newtm -> tm_sec;
  dt.hundredths = 0;

  dt.day     = newtm -> tm_mday;
  dt.month   = newtm -> tm_mon + 1;
  dt.year    = newtm -> tm_year + 1900;
  dt.weekday = newtm -> tm_wday;

  dt.timezone = -1;

  return DosSetDateTime(&dt) != 0;
}

int get_and_set_time(int dont, int port, char *node, 
		     service serv, protocol proto, time_t offs)
{
  int sock, bytes, size;
  struct sockaddr_in server;
  struct hostent *host;
  struct linger linger;
  time_t now;
  char buffer[64];

  if ((host = gethostbyname(node)) == NULL)
    return psock_errno("gethostbyname()"), 1;

  server.sin_family = AF_INET;
  server.sin_port = port;
  server.sin_addr = * (struct in_addr *) (host -> h_addr);

  if (proto == TCP)
  {
    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
      return psock_errno("socket(tcp)"), 1;

    if (connect(sock, (struct sockaddr *) &server, sizeof(server)) < 0)
      return psock_errno("connect()"), 1;

    if ((bytes = recv(sock, buffer, sizeof(buffer), 0)) <= 0)
      return psock_errno("recv()"), 1;
  }
  else
  {
    if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
      return psock_errno("socket(udp)"), 1;

    if (sendto(sock, "", 1, 0,
               (struct sockaddr *) &server, sizeof(server)) < 0)
      return psock_errno("sendto()"), 1;

    size = sizeof(server);
    if ((bytes = recvfrom(sock, buffer, sizeof(buffer), 0,
                          (struct sockaddr *) &server, &size)) < 0)
      psock_errno("recvfrom()");
  }

  soclose(sock);

  if (serv == DAYTIME)
  {
    buffer[bytes] = 0;
    now = get_date(buffer, NULL);
  }
  else
  {
    if (bytes != sizeof(now))
      return printf("invalid time value received\n"), 1;
    else
    {
      now = * (time_t *) buffer;
      now = ntohl(now);
    }
  }

  if (now <= 0)
    return printf("invalid time\n"), 1;

  now += offs;

  if (!dont && stime(&now))
    return printf("invalid time set request\n"), 1;

  printf("time %s %s: %s", dont ? "at" : "set from", 
	 host -> h_name, ctime(&now));

  return 0;
}

int usage(void)
{
  printf("\nUsage: daytime [-ndtTU] [-p port] host\n"
         "\n  -n       do not set the local clock, display the remote time only"
         "\n  -o offs  adjust the retrieved time by offs seconds before"
	 "\n           setting the local clock"
         "\n  -d       use the 'daytime' service (port 13)"
         "\n  -t       use the 'time' service (port 37, default)"
	 "\n  -T       use the 'tcp' protocol (default)"
	 "\n  -U       use the 'udp' protocol"
	 "\n  -p port  use nonstandard port number\n");
  return 1;
}

int main(int argc, char **argv)
{
  int opt, portnum = -1, daytime = 0, dont = 0;
  char *host = NULL;
  struct servent *port;
  service serv = TIME;
  protocol proto = TCP;
  time_t offset = 0;

  if (sock_init())
    return psock_errno("sock_init()"), 1;

  while ((opt = getopt(argc, argv, "?ndtTUo:p:")) != EOF)
    switch (opt)
    {
    case 'n':
      dont = 1;
      break;
    case 'd':
      serv = DAYTIME;
      break;
    case 't':
      serv = TIME;
      break;
    case 'T':
      proto = TCP;
      break;
    case 'U':
      proto = UDP;
      break;
    case 'o':
      offset = atol(optarg);
      break;
    case 'p':
      portnum = htons(atoi(optarg));
      break;
    default:
      return usage();
    }

  if (optind == argc)
    return usage();
  else
    host = argv[optind];

  if (portnum == -1)
  {
    if ((port = getservbyname(serv == DAYTIME ? "daytime" : "time", 
			      proto == TCP ? "tcp" : "udp")) == NULL)
      return psock_errno("getservbyname()"), 1;

    portnum = port -> s_port;
  }

  return get_and_set_time(dont, portnum, host, serv, proto, offset);
}

/* end of daytime.c */
