/*

    File: fat.c

    Copyright (C) 1998-2000  Christophe Grenier <grenier@nef.esiea.fr>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#include <stdio.h>
#include <string.h>
#include "types.h"
#include "common.h"
#include "fat.h"
#include "lang.h"
#include "hdaccess.h"
#include "fnctdsk.h"
#include "testdisk.h"
//#define DEV
#define MAX_TBFAT 10
extern t_param_disk disk_car;
uint sector_size(const struct fat_boot_sector *fat_header);
uint get_dir_entries(const struct fat_boot_sector *fat_header);
uint sectors(const struct fat_boot_sector *fat_header);
t_fat_test start_fat[MAX_TBFAT];

int check_FAT(t_diskext partition,int debug)
{
  t_sector buffer;
#ifdef DEV
  recover_FAT(partition,debug);
#endif
  if(hd_read2(1, &buffer, partition))
  {
    doprintf(m_CHKFAT_RERR); return 1;
  }
  return check_FAT_aux((struct fat_boot_sector *)buffer,partition,debug);
}

#ifdef DEV
/* The idea is to be able to rebuild boot sector and first FAT
 * when they have been erased by CIH virus */
int get_frag(uchar *buffer,int fat_type)
{
  int nb_frag=1;
  switch(fat_type)
  {
    case 12:
      nb_frag=0x200;
    case 16:
      {
	int i;
	word *p16=(word*)buffer;
	for(i=1; i<0x200/2-1; i++)
	  if(p16[i]!=0xFFFF && p16[i]!=i+1)
	    nb_frag++;
      }
      break;
    case 32:
      {
	int i;
	dword *p32=(dword*)buffer;
      	for(i=1; i<0x200/4-1; i++)
      	  if(p32[i]!=(unsigned)0xFFFFFFFF && p32[i]!=i+1)
	    nb_frag++;
      }
      break;
  }
  return nb_frag;
}
/*
int dump_fat(const struct fat_boot_sector*fat_header,int part_type)
{
  doprintf("\nsector_size %u", sector_size(fat_header));
  doprintf("\ncluster_size %u", fat_header->cluster_size);
  doprintf("\nreserved %u", fat_header->reserved);
  doprintf("\nfats %u", fat_header->fats);
  doprintf("\ndir_entries %u", get_dir_entries(fat_header));
  doprintf("\nsectors %u", sectors(fat_header));
  doprintf("\nmedia %02X", fat_header->media);
  doprintf("\nfat_length %u", fat_header->fat_length);
  doprintf("\nsecs_track %u", fat_header->secs_track);
  doprintf("\nheads %u", fat_header->heads);
  doprintf("\nhidden %u", fat_header->hidden);
  doprintf("\ntotal_sect %u", fat_header->total_sect);
  if(part_type==P_32FAT_1)
  {
	doprintf("\nfat32_length %u", fat_header->fat32_length);
	doprintf("\nflags %04X", fat_header->flags);
	doprintf("\nversion %u %u", fat_header->version[0], fat_header->version[1]);
	doprintf("\nroot_cluster %u", fat_header->root_cluster);
	doprintf("\ninfo_sector %u", fat_header->info_sector);
	doprintf("\nbackup_boot %u", fat_header->backup_boot);
  }
  return 0;
}
*/

int dump_2fat(const struct fat_boot_sector*fh1, const struct fat_boot_sector*fh2, int part_type)
{
  doprintf("\nsector_size  %u %u", sector_size(fh1),sector_size(fh2));
  doprintf("\ncluster_size %u %u", fh1->cluster_size,fh2->cluster_size);
  doprintf("\nreserved     %u %u", fh1->reserved,fh2->reserved);
  doprintf("\nfats         %u %u", fh1->fats,fh2->fats);
  doprintf("\ndir_entries  %u %u", get_dir_entries(fh1),get_dir_entries(fh2));
  doprintf("\nsectors      %u %u", sectors(fh1),sectors(fh2));
  doprintf("\nmedia        %02X %02X", fh1->media,fh2->media);
  doprintf("\nfat_length   %u %u", fh1->fat_length,fh2->fat_length);
  doprintf("\nsecs_track   %u %u", fh1->secs_track,fh2->secs_track);
  doprintf("\nheads        %u %u", fh1->heads,fh2->heads);
  doprintf("\nhidden       %u %u", fh1->hidden,fh2->hidden);
  doprintf("\ntotal_sect   %u %u", fh1->total_sect,fh2->total_sect);
  if(part_type==P_32FAT_1)
  {
	doprintf("\nfat32_length %u %u", fh1->fat32_length,fh2->fat32_length);
	doprintf("\nflags        %04X %04X", fh1->flags,fh2->flags);
	doprintf("\nversion      %u.%u  %u.%u", fh1->version[0], fh1->version[1],fh2->version[0], fh2->version[1]);
	doprintf("\nroot_cluster %u %u", fh1->root_cluster,fh2->root_cluster);
	doprintf("\ninfo_sector  %u %u", fh1->info_sector,fh2->info_sector);
	doprintf("\nbackup_boot  %u %u", fh1->backup_boot,fh2->backup_boot);
  }
  return 0;
}

int check_entree(uchar *entree)
{ /* 1 si non nulle */
  int i;
  for(i=0;i<0x20;i++)
	if(*(entree+i))
	  return 1;
  return 0;
}

dword get_subdirectory(t_disk_cst pos_disk,dword i)
{
  uchar buffer[0x200];
  if(hd_read(1, &buffer, pos_disk))
  {
	doprintf(m_ROOT_CLUSTER_RERR); return 1;
  }
//  dump(&buffer,0x200);
//  doprintf("\n1");
  /*                     12345678123*/
  if(strncmp(&buffer[0],".          ",8+3)!=0)
	return 1;
//  doprintf("2");
//  doprintf("\n%d %d",(buffer[0x1B]<<8)+buffer[0x1A],i);
  if((buffer[0x15]<<24)+(buffer[0x14]<<16)+(buffer[0x1B]<<8)+buffer[0x1A]!=i)
	return 1;
//  doprintf("3");
  /*                        12345678123*/
  if(strncmp(&buffer[0x20],"..         ",8+3)!=0)
	return 1;
//  doprintf("4");
//  doprintf("Ok");
  return (buffer[0x35]<<24)+(buffer[0x34]<<16)+(buffer[0x3B]<<8)+buffer[0x3A];
}

dword get_next_cluster(t_diskext_cst partition, int part_type,const int offset, const dword cluster)
{
  /* Offset can be offset to FAT1 or to FAT2 */
  uchar buffer[0x200];
  t_disk pos_disk=(t_disk)MALLOC(sizeof(*pos_disk));
  pos_disk->disk=partition->disk;
  switch(part_type)
  {
	case P_16FAT:
	  {
		/* secteur 1: cluster X 2-256 */
		/* secteur 2: cluster 257-513 */
		word *p16=(word*)&buffer;
		dword offset_s,offset_o;
		offset_s=(cluster-1)/256;
		set_pos(pos_disk,get_LBA_part(partition)+offset+offset_s);
		offset_o=cluster-1-offset_s*256;
		if(hd_read(1, &buffer, pos_disk))
		{
		  doprintf("\nget_next_cluster error"); FREE(pos_disk); return 0;
		}
//		doprintf("cluster=%u offset_s=%u offset_o=%u",cluster,offset_s,offset_o);
		return p16[offset_o];
	  }
	case P_32FAT_1:
	  {
		dword *p32=(dword*)&buffer;
		dword offset_s,offset_o;
		offset_s=(cluster-1)/128;
		set_pos(pos_disk,get_LBA_part(partition)+offset+offset_s);
		offset_o=cluster-1-offset_s*128;
		if(hd_read(1, &buffer, pos_disk))
		{
		  doprintf("\nget_next_cluster error"); FREE(pos_disk); return 0;
		}
//		doprintf("cluster=%u offset_s=%u offset_o=%u",cluster,offset_s,offset_o);
		return p32[offset_o];
	  }
	  break;
  }
  FREE(pos_disk);
  return 0;
}

dword get_prev_cluster(t_diskext_cst partition, int part_type,const int offset, const dword cluster, const dword no_of_cluster)
{
  dword prev_cluster;
  for(prev_cluster=2;prev_cluster<=no_of_cluster;prev_cluster++)
  {
	if(get_next_cluster(partition,part_type,offset, prev_cluster)==cluster)
	  return prev_cluster;
  }
  return 0;
}

int get_root_cluster(t_diskext_cst partition,const int cluster_size, const dword no_of_cluster,const int offset)
{
  dword root_cluster;
  uchar buffer[cluster_size*0x200];
  t_disk pos_disk=(t_disk)MALLOC(sizeof(*pos_disk));
  t_disk pos_disk2=(t_disk)MALLOC(sizeof(*pos_disk));
  pos_disk->disk=partition->disk;
  pos_disk2->disk=partition->disk;
  for(root_cluster=2;root_cluster<no_of_cluster;root_cluster++)
  {
	set_pos(pos_disk,get_LBA_part(partition)+offset+(root_cluster-2)*cluster_size);
	if(hd_read(cluster_size, &buffer, pos_disk))
	{
	  doprintf(m_ROOT_CLUSTER_RERR); return 0;
	}
	if(strncmp(&buffer[0],".          ",8+3)!=0)
	{
	  int i,found=1;
	  int etat=0,nb_subdir=0;
	  for(i=0;found && (i<16*cluster_size);i++)
	  {
		if(check_entree(&buffer[i*0x20])==0)
		{
		  if(etat==0)
			etat=1;
		}
		else
		  if(etat==1)
		  {
			etat=2;
			found=0;
		  }
		if(buffer[i*0x20+0xB]==0x10) /* Test directory */
		{
		  dword cluster=(buffer[i*0x20+0x15]<<24)+(buffer[i*0x20+0x14]<<16)+
			(buffer[i*0x20+0x1B]<<8)+buffer[i*0x20+0x1A];
		  nb_subdir++;
		  set_pos(pos_disk2,get_LBA_part(partition)+offset+(cluster-2)*cluster_size);
//		  doprintf("\n%8s cluster %d back to %d",&buffer[i*0x20],cluster,get_subdirectory(pos_disk2,cluster));
		  if((cluster>no_of_cluster)||(get_subdirectory(pos_disk2,cluster)!=0))
			found=0;
		}
	  }
	  if(nb_subdir<3)
		found=0;
//	  doprintf("\ncluster %d, etat=%d, found=%d,nb_subdir=%d",root_cluster,etat,found,nb_subdir);
	  if(found)
	  {
		dword tmp;
		FREE(pos_disk);
		FREE(pos_disk2);
		/* Il faut ajouter un parcours arriere de la FAT 
		 * car on localise le dernier cluster du root_cluster
		 * */
		tmp=get_prev_cluster(partition,partition->part_type,offset,root_cluster,no_of_cluster);
		while(tmp)
		{
		  root_cluster=tmp;
		tmp=get_prev_cluster(partition,partition->part_type,offset,root_cluster,no_of_cluster);
		}
		return root_cluster;
	  }
	}
  }
  FREE(pos_disk);
  FREE(pos_disk2);
  return 0;
}
  /*
  cluster=681;
  doprintf("\n%d",cluster);
  set_pos(pos_disk,get_LBA_part(partition)+offset+(cluster-2)*cluster_size);
  cluster=get_subdirectory(pos_disk,cluster);
  doprintf("\n%d",cluster);
  set_pos(pos_disk,get_LBA_part(partition)+offset+(cluster-2)*cluster_size);
  cluster=get_subdirectory(pos_disk,cluster);
  doprintf("\n%d",cluster);
  set_pos(pos_disk,get_LBA_part(partition)+offset+(cluster-2)*cluster_size);
  cluster=get_subdirectory(pos_disk,cluster);

  return 0;
  */

int analyse_dir_entries(t_diskext_cst partition, int offset)
{
  uchar buffer[0x200];
  int i,j;
  int etat=0;
  t_disk pos_disk=(t_disk)MALLOC(sizeof(*pos_disk));
  pos_disk->disk=partition->disk;
  set_pos(pos_disk,get_LBA_part(partition)+offset);
  for(i=0;(i<200)&&(etat!=2);i++)
  {
	if(hd_read(1, &buffer, pos_disk))
	{
	  doprintf(m_ANALYSE_DIR_FAT_RERR); return 0;
	}
	for(j=0;j<16;j++)
	{
	  if(check_entree(&buffer[j*0x20])==0)
	  {
		if(etat==0)
		{
		  doprintf("\ndir_entries 0->1 %d",i*16+j);
		  etat=1;
		}
	  }
	  else
		if(etat==1)
		{
		  doprintf("\ndir_entries 1->2 %d",i*16+j);
		  etat=2;
		}
	}
	disk_apres(pos_disk);
  }
  FREE(pos_disk);
  return (i-1)*16;
}

int create_boot_sector(int part_type,t_diskext_cst partition, int reserved, int debug,int dir_entries,dword root_cluster,int cluster_size,int fat_length)
{
  uchar newboot[0x200];
  struct fat_boot_sector *fat_header=(struct fat_boot_sector *)&newboot;
  memset(&newboot,0,0x200);
  fat_header->sector_size[0]=0x200 & 0xFF;
  fat_header->sector_size[1]=0x200>>8;
  fat_header->fats=2;
  fat_header->media=0xF8;
  fat_header->secs_track=disk_car->sector;
  fat_header->heads=disk_car->head+1;
  fat_header->marker=0xAA55;
  fat_header->hidden=get_LBA_part(partition);
  fat_header->cluster_size=cluster_size;
  fat_header->reserved=reserved;
  if(partition->part_size>0xFFFF)
  {
	fat_header->sectors[0]=0;
	fat_header->sectors[1]=0;
	fat_header->total_sect=partition->part_size;
  }
  else
  {
  	fat_header->sectors[0]=partition->part_size&0xFF;
	fat_header->sectors[1]=(partition->part_size>>8)&0xFF;
	fat_header->total_sect=0;
  }
  switch(part_type)
  {
	case P_12FAT:
	  fat_header->fat_length=fat_length;
	  fat_header->dir_entries[0]=dir_entries & 0xFF;
	  fat_header->dir_entries[1]=(dir_entries >> 8) & 0xFF;
	  break;
	case P_16FAT:
	  fat_header->fat_length=fat_length;
	  fat_header->dir_entries[0]=dir_entries & 0xFF;
	  fat_header->dir_entries[1]=(dir_entries >> 8) & 0xFF;
	  break;
	case P_32FAT_1:
	  fat_header->fat32_length=fat_length;
	  fat_header->info_sector=1;
	  fat_header->backup_boot=6;
	  fat_header->root_cluster=root_cluster;
	  break;
  }
  if(debug)
  {
	uchar orgboot[0x200];
	struct fat_boot_sector *org_fat_header=(struct fat_boot_sector *)&orgboot;
	if(hd_read2(1, &orgboot, partition))
	{
	  doprintf(m_CHKFAT_RERR); return 1;
	}
	doprintf("\nInformation from new boot sector");
	dump_2fat(fat_header,org_fat_header,part_type);
  }
  return 0;
}

int calcul_cluster_size(int part_type,dword data_size,int fat_length)
{
  switch(part_type)
  {
	case P_12FAT:
	  return up2power(data_size/((fat_length-1)*512/1.5-1));
	case P_16FAT:
	  return up2power(data_size/((fat_length-1)*512/2-1));
	case P_32FAT_1:
	  return up2power(data_size/((fat_length-1)*512/4-1));
  }
  return 0;
}

int search_fat_aux(t_diskext_cst partition,dword max_offset,int p_fat12,int p_fat16,int p_fat32,int *f_fat12,int*f_fat16,int*f_fat32,int debug,int dump_ind)
{
  t_disk pos_fat=(t_disk)MALLOC(sizeof(*pos_fat));
  int inc_ind;
  dword offset;
  uchar buffer[0x200];
  int pos_fat_test=0;
  pos_fat->disk=partition->disk;
  set_pos(pos_fat,get_LBA_part(partition));
  for(offset=0;offset<max_offset;offset++)
  {
	inc_ind=0;
	if(hd_read(1, &buffer, pos_fat))
	  break;
	if(buffer[0]==0xF8 && buffer[1]==0xFF)
	{
	  start_fat[pos_fat_test].offset=offset;
	  start_fat[pos_fat_test].f_fat12=0;
	  start_fat[pos_fat_test].f_fat16=0;
	  start_fat[pos_fat_test].f_fat32=0;
	  if(dump_ind)
	  {
		doprintf("%u/%u/%u\n",pos_fat->cylinder,pos_fat->head,pos_fat->sector);
		dump(buffer,0x200);
	  }
	  if(p_fat32==1 && buffer[2]==0xFF && buffer[3]==0xFF)
	  {
		start_fat[pos_fat_test].f_fat32=get_frag(buffer,32);
		if(*f_fat32==0 || start_fat[pos_fat_test].f_fat32<*f_fat32)
		  *f_fat32=start_fat[pos_fat_test].f_fat32;
		inc_ind=1;
	  }
	  if(p_fat16==1 && buffer[2]==0xFF)
	  {
		start_fat[pos_fat_test].f_fat16=get_frag(buffer,16);
		if(*f_fat16==0 || start_fat[pos_fat_test].f_fat16<*f_fat16)
		  *f_fat16=start_fat[pos_fat_test].f_fat16;
		inc_ind=1;
	  }
	  if(p_fat12==1)
	  {
		start_fat[pos_fat_test].f_fat12=get_frag(buffer,12);
		if(*f_fat12==0 || start_fat[pos_fat_test].f_fat12<*f_fat12)
		  *f_fat12=start_fat[pos_fat_test].f_fat12;
		inc_ind=1;
	  }
	  if(inc_ind)
	  {
		if(debug)
		  doprintf("\n%5d  12:%d 16:%d 32:%d",start_fat[pos_fat_test].offset,start_fat[pos_fat_test].f_fat12,start_fat[pos_fat_test].f_fat16,start_fat[pos_fat_test].f_fat32);
		pos_fat_test++;
	  }
	}
	disk_apres(pos_fat);
  }
  FREE(pos_fat);
  return pos_fat_test;
}

int get_part_type(int f_fat12,int f_fat16,int f_fat32)
{
  /* Election du type de FAT */
  int part_type=0;
  if(f_fat12)
  {
	if(f_fat16)
	{
	  if(f_fat12<f_fat16)
		part_type=P_12FAT;
	  else
		part_type=P_16FAT;
	}
	else
	  part_type=P_12FAT;
  }
  else
  {
	if(f_fat16)
	{
	  if(f_fat32)
	  {
		if(f_fat16<f_fat32)
		  part_type=P_16FAT;
		else
		  part_type=P_32FAT_1;
	  }
	  else
		part_type=P_16FAT;
	}
	else
	  if(f_fat32) part_type=P_32FAT_1;
  }
  return part_type;
}

int select_fat(int pos_fat_test, int part_type, int f_fat12, int f_fat16, int f_fat32, int debug)
{
  int i,pos=0;
  switch(part_type)
  {
	case P_12FAT:
	  for(i=0;i<pos_fat_test;i++)
	  {
		if(start_fat[i].f_fat12==f_fat12)
		{
		  start_fat[pos].offset=start_fat[i].offset;
		  start_fat[pos].f_fat12=start_fat[i].f_fat12;
		  if(debug)
			doprintf("\nFAT12 at %u",start_fat[pos].offset);
		  pos++;
		}
	  }
	  break;
	case P_16FAT:
	  for(i=0;i<pos_fat_test;i++)
	  {
		if(start_fat[i].f_fat16==f_fat16)
		{
		  start_fat[pos].offset=start_fat[i].offset;
		  start_fat[pos].f_fat16=start_fat[i].f_fat16;
		  if(debug)
			doprintf("\nFAT16 at %u",start_fat[pos].offset);
		  pos++;
		}
	  }
	  break;
	case P_32FAT_1:
	  for(i=0;i<pos_fat_test;i++)
	  {
		if(start_fat[i].f_fat32==f_fat32)
		{
		  start_fat[pos].offset=start_fat[i].offset;
		  start_fat[pos].f_fat32=start_fat[i].f_fat32;
		  if(debug)
			doprintf("\nFAT32 at %u",start_fat[pos].offset);
		  pos++;
		}
	  }
	  break;
  }
  return pos;
}

int set_fat_offset(int*reserved, int*fat_length, t_diskext_cst partition,dword max_offset,int p_fat12,int p_fat16,int p_fat32,int debug,int dump_ind)
{
  int pos_fat_test=0;
  int f_fat12=0, f_fat16=0, f_fat32=0;
  int part_type;
  pos_fat_test=search_fat_aux(partition,max_offset,p_fat12,p_fat16,p_fat32,&f_fat12,&f_fat16,&f_fat32,debug,dump_ind);
  part_type=get_part_type(1.5*f_fat12, 2*f_fat16, 4*f_fat32);
  pos_fat_test=select_fat(pos_fat_test,part_type,f_fat12,f_fat16,f_fat32,debug);

  switch(part_type)
  {
	case P_12FAT:
	case P_16FAT:
	  *reserved=1;
	  break;
	case P_32FAT_1:
	  *reserved=32;
	  break;
  }
  switch(pos_fat_test)
  {
	case 0:
	  *fat_length=0;
	  break;
	case 1:
	  *fat_length=start_fat[0].offset-*reserved;
	  break;
	default:
	  *reserved=start_fat[0].offset;
	  *fat_length=start_fat[1].offset-*reserved;
	  break;
  }
  return part_type;
}

int search_fat16(t_diskext_cst partition,dword max_offset,int p_fat12,int p_fat16,int p_fat32,int debug,int dump_ind)
{
  int part_type;
  int dir_entries=0;
  int cluster_size;
  dword root_cluster=0;
  dword start_data,data_size;
  int reserved;
  int fat_length;
  part_type=set_fat_offset(&reserved, &fat_length, partition,max_offset,p_fat12,p_fat16,p_fat32,debug,dump_ind);
  start_data=reserved+2*fat_length+(part_type==P_32FAT_1?0:((dir_entries-1)/16+1));
  data_size=partition->part_size-start_data;
  cluster_size=calcul_cluster_size(part_type,data_size,fat_length);
  switch(part_type)
  {
	case P_12FAT:
	case P_16FAT:
	  dir_entries=analyse_dir_entries(partition,reserved+2*fat_length);
	  break;
	case P_32FAT_1:
	  root_cluster=get_root_cluster(partition,cluster_size,data_size/cluster_size,start_data);
	  break;
  }
  create_boot_sector(part_type,partition, reserved, debug,dir_entries,root_cluster,cluster_size,fat_length);
  {
	int i;
	for(i=250;i<260;i++)
	  doprintf("\n%u=>%u",i,get_next_cluster(partition, part_type,reserved, i));
  }
  return 0;
}

int recover_FAT(t_diskext partition,int debug)
{
  dword fat_length_max;
//  dword start_rootdir,start_data,no_of_cluster,
//  int nb_entree_tbfat;
//  dword start_fat[MAX_TBFAT];
  int p_fat12,p_fat16,p_fat32;
  int cluster_size_min=1;
  if(partition->part_size>2*1024*1024*2+63)
  {
    p_fat32=1;
    p_fat16=0;
    p_fat12=0;
  }
  else
/* 1<<12 clusters * 8 secteurs/clusters= 32768 secteurs 
   fat_length=((1<<12+1)*1.5/512)+1=13; */
    if(partition->part_size>=(1+2*13+32768+63))
    {
      p_fat32=1;
      p_fat16=1;
      p_fat12=0;
    }
    else
    {
      p_fat32=0;
      p_fat16=1;
      p_fat12=1;
    }
  if(debug)
    doprintf("\nrecover_FAT p_fat12 %d, p_fat16 %d, p_fat32 %d\n", p_fat12,p_fat16,p_fat32);
  /* Set fat_length_max */
  if(p_fat32)
  {
    fat_length_max=partition->part_size*4/512;
  }
  else
  if(p_fat16)
  {
    while(partition->part_size/cluster_size_min>(1<<16))
      cluster_size_min=(cluster_size_min<<1);
    fat_length_max=partition->part_size/cluster_size_min*2/512;
  }
  else
  {
    while(partition->part_size/cluster_size_min>(1<<12))
      cluster_size_min=(cluster_size_min<<1);
    fat_length_max=partition->part_size/cluster_size_min*1.5/512;
  }
  if(debug)
    doprintf("fat_length_max %d\n", fat_length_max);
  search_fat16(partition,64+fat_length_max,p_fat12,p_fat16,p_fat32,debug,0);
  return 1; /* Echec */
}
#endif

int check_FAT_aux(struct fat_boot_sector *fat_header, t_diskext partition,int debug)
{
  dword start_fat1,start_fat2,start_rootdir,start_data,no_of_cluster,fat_length,fat_length_calc,part_size,end_data;
  if(fat_header->marker!=0xAA55)
  { doprintf("check_FAT : " m_BAD_NMARK); return 1; }
  if(!((fat_header->ignored[0]==0xeb&&fat_header->ignored[2]==0x90)||fat_header->ignored[0]==0xe9))
  { doprintf(m_CHKFAT_BAD_JUMP); return 1; }
  if(sector_size(fat_header)!=512)
  { doprintf(m_CHKFAT_BYTESECT); return 1; }
  switch(fat_header->cluster_size)
  { /* should depend of FAT12/16/32 */
    case 1:case 2:case 4:case 8:case 16:case 32:case 64:case 128: break;
    default: doprintf(m_CHKFAT_SECT_CLUSTER); return 1;
  }
  if(fat_header->secs_track!=disk_car->sector)
  { doprintf(m_CHKFAT_SECTTRACK); return 1; }
  if(fat_header->heads!=disk_car->head+1)
  { doprintf(m_CHKFAT_SIDE); return 1; }
  if(fat_header->fats!=2)
  { doprintf(m_CHKFAT_NBRFAT); return 1; }
  if(get_dir_entries(fat_header)%16)        /* Improved it */
  { doprintf(m_CHKFAT_ENTRY); return 1; }
  if(fat_header->media!=0xF8)
  { doprintf(m_CHKFAT_MEDIA); return 1; }
  switch(partition->part_type)
  {
    case P_12FAT:
	  part_size=sectors(fat_header);
      if(part_size!=partition->part_size)
      { doprintf(m_CHKFAT_SIZE); return 1; }
      if(fat_header->reserved!=1)
      { doprintf(m_CHKFAT_RESERV); return 1; }
      if(get_dir_entries(fat_header)==0)
      { doprintf(m_CHKFAT_ENTRY); return 1; }
      if(fat_header->fat_length>256)
      { doprintf(m_CHKFAT_SECTPFAT); return 1; }
	  fat_length=fat_header->fat_length;
	  start_fat1=fat_header->reserved;
      start_fat2=start_fat1+fat_length;
      start_rootdir=start_fat2+fat_length;
      start_data=start_rootdir+(get_dir_entries(fat_header)-1)/16+1;
	  no_of_cluster=(part_size-start_data)/fat_header->cluster_size;
	  fat_length_calc=((no_of_cluster+1)*1.5/512)+1;
	  break;
    case P_16FAT:
    case P_16FATH:
    case P_16FATBD:
    case P_16FATBDH:
	  part_size=fat_header->total_sect;
      if((sectors(fat_header)!=0)||(part_size!=partition->part_size))
      { doprintf(m_CHKFAT_SIZE); return 1; }
   	  fat_length=fat_header->fat_length;
	  start_fat1=fat_header->reserved;
      start_fat2=start_fat1+fat_length;
      start_rootdir=start_fat2+fat_length;
      start_data=start_rootdir+(get_dir_entries(fat_header)-1)/16+1;
	  no_of_cluster=(fat_header->total_sect-start_data)/fat_header->cluster_size;
	  fat_length_calc=((no_of_cluster+1)*2/512)+1;
	  break;
    case P_32FAT_1:
    case P_32FAT_2:
    case P_32FAT_1H:
    case P_32FAT_2H:
	  part_size=fat_header->total_sect;
      if((sectors(fat_header)!=0)||
	  (part_size!=partition->part_size))
      { doprintf(m_CHKFAT_SIZE); return 1; }
      fat_length=fat_header->fat_length?fat_header->fat_length:fat_header->fat32_length;
      start_fat1=fat_header->reserved;
      start_fat2=start_fat1+fat_length;
      start_data=start_fat2+fat_length;
      start_rootdir=start_data+fat_header->root_cluster*fat_header->cluster_size;
      no_of_cluster=(part_size-start_data)/fat_header->cluster_size;
	  fat_length_calc=((no_of_cluster+1)*4/512)+1;
	  break;
	default:
	  return 1;
  }
  end_data=start_data+no_of_cluster*fat_header->cluster_size;
  if(debug)
  {
  	doprintf("\nFAT1 : %u-%u",start_fat1,start_fat1+fat_length-1);
	doprintf("\nFAT2 : %u-%u",start_fat2,start_fat2+fat_length-1);
	doprintf("\nstart_rootdir : %u",start_rootdir);
	doprintf("\nData : %u-%u",start_data,end_data);
	doprintf("\nsectors : %u",part_size);
	doprintf("\ncluster_size : %u",fat_header->cluster_size);
	doprintf("\nno_of_cluster : %u (2 - %u)", no_of_cluster,no_of_cluster+1);
	doprintf("\nfat_length %u calculated %u\n",fat_length,fat_length_calc);
  }
  if(fat_length<fat_length_calc)
  { doprintf(m_CHKFAT_SECTPFAT); return 1; }
  if(fat_header->hidden!=get_LBA_part(partition))
  { doprintf(m_CHKFAT_BAD_HIDDEN); return 1; }
  return comp_FAT(partition,fat_length,fat_header->reserved);
}

int comp_FAT(t_diskext partition,dword taille,const dword sect_res)
{
  uint read_size;
  uint i;
  uchar buffer[NBR_SECT*0x200],buffer2[NBR_SECT*0x200];
  t_disk pos_fat=(t_disk)MALLOC(sizeof(*pos_fat));
  t_disk pos_fat2=(t_disk)MALLOC(sizeof(*pos_fat2));
  fflush(stdout);
  pos_fat->disk=partition->disk;
  pos_fat2->disk=partition->disk;
  set_pos(pos_fat,get_LBA_part(partition)+sect_res);
  set_pos(pos_fat2,get_LBA_part(partition)+sect_res+taille);
  while(taille)
  {
    read_size=taille>NBR_SECT?NBR_SECT:taille;
    taille-=read_size;
    if(hd_read(read_size, &buffer, pos_fat))
    { doprintf(m_CHKFAT_RERR);
      FREE(pos_fat);
      FREE(pos_fat2);
      return 1;}
    if(hd_read(read_size, &buffer2, pos_fat2))
    { doprintf(m_CHKFAT_RERR); 
      FREE(pos_fat);
      FREE(pos_fat2);
      return 1;}
    if(memcmp(buffer,buffer2,0x200*read_size)!=0)
    { doprintf("\nFAT differs"); 
      FREE(pos_fat);
      FREE(pos_fat2);
      return 1;
    }
    for(i=0;i<read_size;i++)
    {
      disk_apres(pos_fat);
      disk_apres(pos_fat2);
    }
  }
  FREE(pos_fat);
  FREE(pos_fat2);
  return 0;
}

uint sector_size(const struct fat_boot_sector *fat_header)
{ return (fat_header->sector_size[1]<<8)+fat_header->sector_size[0]; }

uint get_dir_entries(const struct fat_boot_sector *fat_header)
{ return (fat_header->dir_entries[1]<<8)+fat_header->dir_entries[0]; }

uint sectors(const struct fat_boot_sector *fat_header)
{ return (fat_header->sectors[1]<<8)+fat_header->sectors[0]; }

int analyse_FAT(struct fat_boot_sector*fat_header, t_diskext partition,int debug, int dump_ind)
{
  char*buffer=(char*)fat_header;
  if(fat_header->marker==0xAA55)
  {
	if(debug||dump_ind)
	{
	  doprintf("\nMarker (0xAA55) at %u/%u/%u\n", partition->scylinder,partition->shead,partition->ssector);
	}
	if(dump_ind)
	  dump(buffer,0x200);
    if(memcmp(buffer+FAT_NAME1,"FAT12",5)==0) /* 2 Mo max */
    {
      partition->part_size=sectors(fat_header);
      partition->part_type=P_12FAT;
      return 0;
    }
    else if(memcmp(buffer+FAT_NAME1,"FAT16",5)==0)
    {
      if(sectors(fat_header)!=0)
      {
	partition->part_size=sectors(fat_header);
	partition->part_type=P_16FAT;
	return 0;
      }
      partition->part_size=fat_header->total_sect;
      partition->part_type=P_16FATBD;
    }
    else if(memcmp(buffer+FAT_NAME2,"FAT32",5)==0)
    {
      if(sectors(fat_header)!=0)
	partition->part_size=sectors(fat_header);
      else
	partition->part_size=fat_header->total_sect;
      partition->part_type=P_32FAT_1;
      return 0;
    }
    else if(memcmp(buffer+OS2_NAME,"IBM",3)==0)
    {   /* D'apres une analyse de OS2 sur systeme FAT...
	FAT_NAME1=FAT
	*/
      if(sectors(fat_header)!=0)
	partition->part_size=sectors(fat_header);
      else
	partition->part_size=fat_header->total_sect;
      partition->part_type=P_HPFS;
      return 0;
    }
    else if(memcmp(buffer+FAT_NAME1,"FAT",3)==0)
    {   /* Attention a detecter apres OS2 */
      partition->part_size=(disk_car->head+1) * disk_car->sector; /* 1 cylinder */
      partition->part_type=P_OS2MB;
      return 0;
    }
    else if(memcmp(buffer+NTFS_NAME,"NTFS",4)==0)
    {
      partition->part_size=NT_sect(buffer)+1;
      partition->part_type=P_NTFS;
      return 0;
    }
  }     /* fin marqueur de fin :)) */
  return 1;
}
