/*************************************************************
 ** ListBox.cpp  --  part of the Falcon DLL Extension v1.50
 **
 ** Abstract: This file contains the FALC_listbox command
 **           and related functions.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux  mailto:davygrvy@bigfoot.com
 ************************************************************/

#include "falcon.h"
#include "util.h"
#include "resource.h"
#include <winuser.h>

extern dyn_AppendResult Tcl_AppendResult;
//extern dyn_AppendElement Tcl_AppendElement;
//extern dyn_CreateCommand Tcl_CreateCommand;
extern dyn_ResetResult Tcl_ResetResult;
extern dyn_GlobalEval Tcl_GlobalEval;
extern dyn_DoOneEvent Tcl_DoOneEvent;
extern BOOL InXiRCON;
extern HINSTANCE hInstance;


/* 
 *  Even though the CUSTDATA structure is OK to 
 *  compile, it still gives this warning:
 *
 *    nonstandard extension used : zero-sized array in struct/union
 */
#pragma warning(disable: 4200)
typedef struct {
  int retVal;
  BOOL done;
  char *result;
  BOOL UseEdit;
  BOOL xUI;
  char *title;
  char *msg;
  int count;
  char *string[];    // unsized array.
} CUSTDATA,*LPCUSTDATA;
#pragma warning(default: 4200)

/* Local Prototypes */
static BOOL CALLBACK DialogProc ( HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam );
static BOOL InitList ( HWND hWndCmbo, LPCUSTDATA data );


/*
 *--------------------------------------------------------------
 *
 * CmdDialog --
 *
 *  Displays a list pick dialog box.
 *
 * Results:
 *  The selected string.
 *
 * Side effects:
 *  Creates a thread to manage the dialog box while it exists. 
 *
 *--------------------------------------------------------------
 */
extern int CmdListBox(ClientData clientData,
                     Tcl_Interp *interp,int argc,char *argv[]) {
 LPCUSTDATA info;
 int j,size,parenthesized,result;
 char *p,*element,*next;
 LPSTR dlgbox;
 HWND hwndOwner;

  info = (LPCUSTDATA) xmalloc (sizeof(CUSTDATA));

  info->xUI = FALSE;
  info->UseEdit = FALSE;
  info->done = FALSE;
  info->result = NULL;
  hwndOwner = Falc_GetXiRChWnd(interp);
  dlgbox = MAKEINTRESOURCE(IDD_DIALOG1);


  if (argc > 1) {
    for (j=1;j<(argc-1);j++) {
      if (argv[j][0] == '-') {
        switch (argv[j][1]) {
        case 't': /* title */
          info->title=argv[++j]; 
          break;
        case 'x': /* extended UI */
          info->xUI = TRUE;
          break;
        case 'e': /* use a dropdown combo instead of a dropdown list */
          dlgbox = MAKEINTRESOURCE(IDD_DIALOG2);
          break;
        case 'm': /* set message */
          info->msg = argv[++j];
          break;
        default:
          Tcl_AppendResult(interp,argv[0],": ",argv[j]," is an unknown option.",NULL);
          return TCL_ERROR;
        }
      }
    }
  } else {
Usage:
    Tcl_AppendResult(interp,"Usage: ",argv[0]," [-t <title>] [-m <message>] [-x] [-e] <list>.",NULL);
    return TCL_ERROR;
  }

  p=argv[argc-1]; info->count=0;

  do {
    int newmemsize;
    result = Falc_FindElement(NULL, p, &element, &next, &size, &parenthesized);
    if (result != TCL_OK) goto Usage;
    if (*element == NULL) break;

    newmemsize = sizeof(CUSTDATA)+(sizeof(char *)*(info->count+1));
    info = (LPCUSTDATA) xrealloc(info,newmemsize);
    info->string[info->count] = (char *) xmalloc (size+1);

    if (parenthesized) {
      memcpy((void *) info->string[info->count], (void *) element, (size_t) size);
      info->string[info->count++][size] = '\0';
    } else {
      Falc_CopyAndCollapse(size, element, info->string[info->count++]);
    }
    p=next;

  } while (*p != NULL);


  info->retVal = DialogBoxParam( hInstance, dlgbox, hwndOwner, (DLGPROC)DialogProc, (LPARAM) info);



  Tcl_AppendResult(interp, info->result, NULL);

  if (info->result) xfree(info->result);
  xfree(info);

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * DialogProc --
 *
 *  callback routine for the dialog.
 *
 * Results:
 *  depends
 *
 * Side Effects:
 *----------------------------------------------------------------------
 */
static BOOL CALLBACK 
DialogProc( HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam ) {
 auto BOOL bRet = FALSE;
 auto LPCUSTDATA cust;

  switch (uMsg) {

  case WM_INITDIALOG: {
     auto LPCUSTDATA cust = (LPCUSTDATA) lParam;
     auto HWND combo = GetDlgItem(hDlg, IDC_COMBO1);

      /* place the pointer to our data structure 
       * in the dialog's user space */
      SetWindowLong(hDlg, DWL_USER, (LONG)cust);

      if (cust->title == NULL) {
        SetWindowText(hDlg,DEFTITLE);
      } else {
        SetWindowText(hDlg,cust->title);
      }

      if (cust->msg == NULL) {
        SetWindowText(GetDlgItem(hDlg, IDC_STATIC1), "Select an item:");
      } else {
        SetWindowText(GetDlgItem(hDlg, IDC_STATIC1), cust->msg);
      }

      /* Turns on 'extended UI' for the combobox */
      if (cust->xUI) SendDlgItemMessage(hDlg, IDC_COMBO1, CB_SETEXTENDEDUI, 1, 0);

      if(!InitList(combo, cust)) EndDialog(hDlg, 0);

      bRet = TRUE;
    }
    break;

  case WM_ENTERIDLE:
    while (Tcl_DoOneEvent(TCL_DONT_WAIT));
    break;

  case WM_COMMAND:

    cust = (LPCUSTDATA) GetWindowLong(hDlg, DWL_USER);

    switch (LOWORD(wParam)) {
    case IDOK:
      cust->result = (char *) xmalloc(256);
      GetDlgItemText( hDlg, IDC_COMBO1, cust->result, 256 );
      /* fall through */

    case IDCANCEL:
      EndDialog(hDlg, uMsg);
    }
    break;

  }

  return (bRet);
}  

/*
 *----------------------------------------------------------------------
 * InitList --
 *
 *  fills the combobox when the dialog is created.
 *
 * Results:
 *  TRUE
 *
 * Side Effects:
 *  deallocates some memory.
 *
 *----------------------------------------------------------------------
 */
static BOOL
InitList (HWND hWndCmbo, LPCUSTDATA data) {
 int i;

  if (hWndCmbo == NULL) return FALSE;

  SendMessage(hWndCmbo, CB_RESETCONTENT, 0, 0);
  for (i=0;i<data->count;i++) {
    SendMessage(hWndCmbo, CB_ADDSTRING, 0, (LPARAM) data->string[i]);
    xfree(data->string[i]);
  };
  SendMessage(hWndCmbo, CB_SETCURSEL , 0, 0);
  return TRUE;
}

