/*************************************************************
 ** FileOpen.cpp  --  part of the Falcon DLL Extension v1.4
 **
 ** Abstract:  This file contains the FALC_fileopen command.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux  mailto:davygrvy@bigfoot.com
 ************************************************************/

#include "falcon.h"
#include "util.h"

#include <commdlg.h>
#include <cderr.h>
#include <dlgs.h>

extern dyn_AppendElement Tcl_AppendElement;
extern dyn_AppendResult Tcl_AppendResult;
extern dyn_GlobalEval Tcl_GlobalEval;
extern dyn_ResetResult Tcl_ResetResult;
extern dyn_DoOneEvent Tcl_DoOneEvent;
extern BOOL InXiRCON;

typedef struct {
    char *hlpfilename;
    int hlpindex;
} CMDLGCUSTDATA, *LPCMDLGCUSTDATA;

typedef struct _ComdlgFileInfo {
    OPENFILENAME ofn;
    Tcl_Interp *interp;
    BOOL DoSave;
    int done;
    int retVal;
} CMDLGFILEINFO, *LPCMDLGFILEINFO;

static DWORD WINAPI FileOpenThread(LPVOID arg);

/*
 *----------------------------------------------------------------------
 * FileOpenDlgHook --
 *
 *  Hook function that is used to invoke help.
 *----------------------------------------------------------------------
 */
BOOL CALLBACK  
FileOpenDlgHook(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam) {

  switch (msg) {
  case WM_INITDIALOG:
    /* Save off the long pointer to the OPENFILENAME structure. */
    SetWindowLong(hDlg, DWL_USER, lParam);
    SetForegroundWindow(hDlg);
    SetActiveWindow(hDlg);
    break;

  case WM_COMMAND:
    if (LOWORD(wParam) == pshHelp) {
      auto LPOPENFILENAME lpOFN = (LPOPENFILENAME) GetWindowLong(hDlg, DWL_USER);
      auto LPCMDLGCUSTDATA cust = (LPCMDLGCUSTDATA) lpOFN->lCustData;
      
      WinHelp(hDlg, cust->hlpfilename, HELP_CONTEXT, cust->hlpindex);
      break;
    }
    /* yes, we let this fall through. */

  default:
    return FALSE;   /* let the internal Dlg handler take care of this message */
  }

  return TRUE;   /* tell the internal Dlg handler we already handled it. */
}

/*
 *----------------------------------------------------------------------
 *
 * Cmdfile_open --
 *  opens a 'fileopen common dialogbox'
 *
 * Results:
 *  returns the fullpath of the file(s) selected.
 *
 * Side effects:
 *  none know.
 *
 *----------------------------------------------------------------------
 */
extern int Cmdfile_open(ClientData clientData,
                        Tcl_Interp *interp,int argc,char *argv[]) {
 auto char *lpFile;
 auto int i,j=1;
 auto char *lpFilter=NULL,*lpDir=NULL;
 auto HANDLE thread;
 auto DWORD threadId;
 auto char buff2[33];
 auto int ret;
 auto CMDLGCUSTDATA cust;
 auto LPCMDLGFILEINFO info;
 auto BOOL NoExist = FALSE;


  if(!(info=(LPCMDLGFILEINFO)xmalloc(sizeof(CMDLGFILEINFO)))) {
    Tcl_AppendResult(interp,"HeapAlloc on line ",itoa(__LINE__,buff2,10)," of file ",__FILE__,NULL);
    return TCL_ERROR;
  };

  /* filename buffer */ 
  if(!(lpFile=(char *)xmalloc(MAX_PATH * 20))) {
    Tcl_AppendResult(interp,"HeapAlloc on line ",itoa(__LINE__,buff2,10)," of file ",__FILE__,NULL);
    return TCL_ERROR;
  };
  info->ofn.nMaxFile = (MAX_PATH * 20);

  info->interp = interp;

  /* Set the members of the OPENFILENAME structure. */ 
  info->ofn.lStructSize = sizeof(OPENFILENAME);
  info->ofn.lpstrFileTitle = NULL; 
  info->ofn.lpstrDefExt = NULL;
  info->ofn.lpstrCustomFilter = NULL;
  info->ofn.lCustData = (LPARAM) &cust;
  info->ofn.lpfnHook = (LPOFNHOOKPROC) FileOpenDlgHook;
  info->ofn.lpTemplateName = NULL;
  info->ofn.hwndOwner = Falc_GetXiRChWnd(interp);

  /* preset options for below */
  info->ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_ENABLEHOOK | 
        OFN_NONETWORKBUTTON | OFN_NOCHANGEDIR | OFN_HIDEREADONLY | 
        OFN_LONGNAMES | OFN_EXPLORER;
  info->ofn.lpstrTitle=DEFTITLE;
  info->ofn.hwndOwner=NULL;
  info->ofn.lpstrFilter=NULL; 
  info->ofn.nFilterIndex=0;
  info->ofn.lpstrInitialDir=NULL;
  info->DoSave=FALSE;

  if (argc > 1) for (j=1;j<argc;j++) {
    if (!strncmp(argv[j],"-",1)) {
      switch (*(argv[j]+1)) {
      case 't': /* title */
        info->ofn.lpstrTitle=argv[++j]; 
        break;
      case 'm':  /* Modal dialog */
        // do nothing.  option has been removed
        break;
      case 'f':  /* Add a filter array */
        /* Allocate new memory */
        if(!(lpFilter=(char *)xmalloc(strlen(argv[++j])+2))) {
          Tcl_AppendResult(interp,"HeapAlloc on line ",itoa(__LINE__,buff2,10)," of file ",__FILE__,NULL);
          return TCL_ERROR;
        };
        /* Load the filter string and replace '|'s with NULLs */
        for(i=0;argv[j][i]!='\0';i++) {
          if(argv[j][i]=='|') *(lpFilter+i)='\0';
          else *(lpFilter+i)=argv[j][i];
        }
        *(lpFilter+i)='\0';
        /* must be double-null terminated */
        if(*(lpFilter+i-1)!='\0') *(lpFilter+i+1)='\0';
        info->ofn.lpstrFilter=lpFilter;
        info->ofn.nFilterIndex=1;
        break;
      case 'd':  /* Add a default directory to open the dialog in */
        if ((j+1)<argc) {
          if(!(lpDir=Falc_DoFileName(argv[++j]))) {
            Tcl_AppendResult(interp,"HeapAlloc on line ",itoa(__LINE__,buff2,10)," of file ",__FILE__,NULL);
            return TCL_ERROR;
          }
          info->ofn.lpstrInitialDir=lpDir;
        } else {
          Tcl_AppendResult(interp,argv[0],": -d usage is '-d <defaultDir>'",NULL);
          return TCL_ERROR;
        }
        break;
      case 'h':  /* Add the help button and a helpfile to call */
        if ((j+2)<argc) {
          info->ofn.Flags |= OFN_SHOWHELP;
          cust.hlpfilename = argv[++j];
          if (Falc_GetInt(interp, argv[++j], &(cust.hlpindex)) != TCL_OK) {
            return TCL_ERROR;
          }
        } else {
          Tcl_AppendResult(interp,argv[0],": -h usage is '-h <helpfile> <index>'",NULL);
          return TCL_ERROR;
        }
        break;
      case 'o':  /* Use old style dialog */
        info->ofn.Flags &= ~(OFN_EXPLORER);
        break;
      case 'M':  /* Use multiselect */
        info->ofn.Flags |= OFN_ALLOWMULTISELECT;
        break;
      case 'n':  /* Add a network button */
        info->ofn.Flags &= ~(OFN_NONETWORKBUTTON);
        break;
      case 's':  /* changes dialog to a filesave instead of a fileopen */
        info->DoSave = TRUE;
        NoExist = TRUE;
        break;
      case 'e':
        NoExist = TRUE;
        break;
      default:
        Tcl_AppendResult(interp,argv[0],": ",argv[j]," is an unknown option.",NULL);
        return TCL_ERROR;
      }
    }
  }


  if (NoExist == TRUE) {
    info->ofn.Flags &= ~(OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST);
  };

  /* Ensure we don't return multiple names for a save */
  if (info->DoSave == TRUE) {
    info->ofn.Flags &= ~(OFN_ALLOWMULTISELECT);
  }


  /* Place the terminating null character in szFile. */ 
  lpFile[0] = '\0';
  info->ofn.lpstrFile = lpFile;

  /* add the default filter for the old-style dialog if none were specified. */
  if (!(info->ofn.Flags & OFN_EXPLORER) && !lpFilter) {
    info->ofn.lpstrFilter = "All Files\0*.*\0";
    info->ofn.nFilterIndex=1;
  }

/*
Tcl_AppendResult(interp,
      "ofn.lpstrTitle = ",info->ofn.lpstrTitle?info->ofn.lpstrTitle:"",
      "\nofn.hwndOwner = ",ultoa((long)info->ofn.hwndOwner,buff2,10),
      "\nofn.lpstrFilter = ",info->ofn.lpstrFilter?info->ofn.lpstrFilter:"",
      "\nofn.lpstrInitialDir = ",info->ofn.lpstrInitialDir?info->ofn.lpstrInitialDir:"",
      NULL);
Tcl_AppendResult(interp,"\nofn.Flags = ",ultoa(info->ofn.Flags,buff2,16),NULL);
return TCL_OK;
*/

  thread = CreateThread(NULL,0,(LPTHREAD_START_ROUTINE)FileOpenThread,info,0,&threadId);

  /* check for error */
  if (!thread) {
    Tcl_AppendResult(interp,argv[0],": error starting non-blocking thread!",NULL);
    return TCL_ERROR;
  } else {
    /* dereference handle */
    CloseHandle( thread );
  }

  /*  
   *  Loop, but yield to Tcl. Actually in Beta2 of XiRC this 
   *  was needed. But for beta4, Tcl functions aren't blocking
   *  to IRC events :) I don't know how this command will
   *  behave for re-entracy (like, what if its called a second 
   *  time while the first is still waiting on a return). 
   *  I should test this. 
   */
  while (info->done == 0) {
    Tcl_DoOneEvent(TCL_DONT_WAIT);
  }

  /* free dynamic memory we created earlier */
  if(info->ofn.lpstrInitialDir) xfree(lpDir);
  if(info->ofn.lpstrFilter) xfree(lpFilter);
  ret = info->retVal;
  xfree(info);
  xfree(lpFile);

  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * FileOpenThread --
 *	opens the common dialog and handles the return.
 *----------------------------------------------------------------------
 */
static DWORD WINAPI 
FileOpenThread (LPVOID arg) {
 auto DWORD er;
 auto char buff[225]={'\0'},buff2[33];
 auto LPCMDLGFILEINFO info = (LPCMDLGFILEINFO) arg;
 auto int rtn;

  if (info->DoSave) {
    rtn = GetSaveFileName(&info->ofn);
  } else {
    rtn = GetOpenFileName(&info->ofn);
  };

  /* Display the Open dialog box. */ 
  if (rtn) {
    auto int i,pos=0;

    /* fix slashes in the directory name. */
    for(i=0;*(info->ofn.lpstrFile+i)!='\0';i++)
	    if(*(info->ofn.lpstrFile+i)=='\\')*(info->ofn.lpstrFile+i)='/';
    pos = i + 1;

	/* check for multiselect flag */
    if(info->ofn.Flags & OFN_ALLOWMULTISELECT) {
      auto int DirLen;
      auto char newFile[MAX_PATH];
      auto int j;

      /* check if we are using the older dialog. */
      if(!(info->ofn.Flags & OFN_EXPLORER)){
        /* change spaces into <EndOfStringMarker> */
        for(i=0;*(info->ofn.lpstrFile+i)!='\0';i++)
            if(*(info->ofn.lpstrFile+i)==' ')*(info->ofn.lpstrFile+i)='\0';
      };

      /* check if we just have one entry */
      if(*(info->ofn.lpstrFile+pos)=='\0') {
        goto justone;
      }

      /* copy DirName into newFile. */
      for(i=0;*(info->ofn.lpstrFile+i)!='\0';i++)
        *(newFile+i)=*(info->ofn.lpstrFile+i);
      /* append a directory seperator */
      *(newFile+(i++))='/';
      /* terminate string */
      *(newFile+i)='\0';
      /* save its length. */
      DirLen = strlen(newFile);

      do {
        for(i=pos,j=0;*(info->ofn.lpstrFile+i)!='\0';i++,j++)
            *(newFile+DirLen+j)=*(info->ofn.lpstrFile+i);
        /* terminate the new string */
        *(newFile+DirLen+j)='\0';
        Tcl_AppendElement(info->interp,newFile);
        pos+=j+1;
      } while (*(info->ofn.lpstrFile+pos)!='\0');

      goto theend;
    };

justone:
    Tcl_AppendResult(info->interp,info->ofn.lpstrFile,NULL);

theend:
    info->retVal = TCL_OK;
    info->done = 1;
    return 0;
  } else if((er=CommDlgExtendedError())==CDERR_GENERALCODES){
    /* the user pressed cancel. Just return silently. */
    info->retVal = TCL_OK;
    info->done = 1;
    return 0;
  } else {
    switch (er) {
    case CDERR_STRUCTSIZE : 
      strcpy(buff,"CDERR_STRUCTSIZE: The lStructSize member of the structure for the corresponding common dialog box is invalid.");break;
    case CDERR_INITIALIZATION :
      strcpy(buff,"CDERR_INITIALIZATION: The common dialog box procedure failed during initialization. This error often occurs when insufficient memory is available.");break;
    case CDERR_NOTEMPLATE :
      strcpy(buff,"CDERR_NOTEMPLATE: The ENABLETEMPLATE flag was specified in the Flags member of a structure for the corresponding common dialog box, but the application failed to provide a corresponding template.");break;
    case CDERR_NOHINSTANCE :
      strcpy(buff,"CDERR_NOHINSTANCE: The ENABLETEMPLATE flag was specified in the Flags member of a structure for the corresponding common dialog box, but the application failed to provide a corresponding instance handle.");break;
    case CDERR_LOADSTRFAILURE :
      strcpy(buff,"CDERR_LOADSTRFAILURE: The common dialog box procedure failed to load a specified string.");break;
    case CDERR_FINDRESFAILURE :
      strcpy(buff,"CDERR_FINDRESFAILURE: The common dialog box procedure failed to find a specified resource!");break;
    case CDERR_LOADRESFAILURE :
      strcpy(buff,"CDERR_LOADRESFAILURE: The common dialog box procedure failed to load a specified resource.");break;
    case CDERR_LOCKRESFAILURE :
      strcpy(buff,"CDERR_LOCKRESFAILURE: The common dialog box procedure failed to lock a specified resource.");break;
    case CDERR_MEMALLOCFAILURE :
      strcpy(buff,"CDERR_MEMALLOCFAILURE: The common dialog box procedure was unable to allocate memory for internal structures.");break;
    case CDERR_MEMLOCKFAILURE :
      strcpy(buff,"CDERR_MEMLOCKFAILURE: The common dialog box procedure was unable to lock the memory associated with a handle.");break;
    case CDERR_NOHOOK :
      strcpy(buff,"CDERR_NOHOOK: The ENABLEHOOK flag was specified in the Flags member of a structure for the corresponding common dialog box, but the application failed to provide a pointer to a corresponding hook function.");break;
    case FNERR_FILENAMECODES :
      strcpy(buff,"FNERR_FILENAMECODES: Error codes for the Open and Save As common dialog boxes. these errors are in the range 0x3000 through 0x3FFF.");break;
    case FNERR_SUBCLASSFAILURE :
      strcpy(buff,"FNERR_SUBCLASSFAILURE: An attempt to subclass a list box failed because insufficient memory was available.");break;
    case FNERR_INVALIDFILENAME :
      strcpy(buff,"FNERR_INVALIDFILENAME: A filename is invalid.");break;
    case FNERR_BUFFERTOOSMALL :
      strcpy(buff,"FNERR_BUFFERTOOSMALL: The buffer for a filename is too small. (This buffer is pointed to by the lpstrFile member of the structure for a common dialog box.)");break;
    default : 
      strcpy(buff,"Unknown error code!");
    }
    Tcl_AppendResult(info->interp,"CommonDlgError# 0x",ltoa(er,buff2,16)," ",buff,NULL);
    info->retVal = TCL_ERROR;
    info->done = 1;
    return 0;
  };
}

