/*************************************************************
 ** Dialog.cpp  --  part of the Falcon DLL Extension v1.50
 **
 ** Abstract: This file contains the FALC_dialog command
 **           and related functions.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux  mailto:davygrvy@bigfoot.com
 ************************************************************/

#include "falcon.h"
#include "util.h"
#include <winuser.h>

/* Globals */
extern dyn_AppendResult Tcl_AppendResult;
extern dyn_AppendElement Tcl_AppendElement;
extern dyn_ResetResult Tcl_ResetResult;
extern dyn_GlobalEval Tcl_GlobalEval;
extern dyn_DoOneEvent Tcl_DoOneEvent;
extern HINSTANCE hInstance;

WORD ItemIDCounter;


/* Structures */
#include <pshpack2.h>

typedef struct {
     WORD   signature;         // MS says dlgVer is first and signature is
     WORD   dlgVer;            // second, but through trail and error,
     DWORD  helpID;            // I found this was wrong!
     DWORD  exStyle;
     DWORD  style;
     WORD   cdit;
     short  x;
     short  y;
     short  cx; 
     short  cy;
} DLGTEMPLATEEX, *LPDLGTEMPLATEEX;
//     sz_Or_Ord menu;         // name or ordinal of a menu resource
//     sz_Or_Ord windowClass;  // name or ordinal of a window class
//     WCHAR  title[titleLen]; // title string of the dialog box
//     short  pointsize;       // only if DS_SETFONT flag is set
//     short  weight;          // only if DS_SETFONT flag is set
//     short  bItalic;         // only if DS_SETFONT flag is set
//     WCHAR  font[fontLen];   // typeface name, if DS_SETFONT is set

typedef struct {
     DWORD  helpID;
     DWORD  exStyle;
     DWORD  style;
     short  x;
     short  y;
     short  cx;
     short  cy;
     WORD   id;
} DLGITEMTEMPLATEEX, *LPDLGITEMTEMPLATEEX;
//     sz_Or_Ord windowClass; // name or ordinal of a window class
//     sz_Or_Ord title;       // title string or ordinal of a resource
//     WORD   extraCount;     // bytes of following creation data

#include <poppack.h> /* Resume normal packing */

typedef enum {
  Button = 0x0080,
  Edit = 0x0081,
  Static = 0x0082,
  ListBox = 0x0083,
  ScrollBar = 0x0084,
  ComboBox = 0x0085
} ICLASS;

typedef enum { combo, edit, check, label } CTRLTYPE;

typedef struct {
  int length;
  LPWORD mem;
} DLGITEM;

#pragma warning(disable: 4200)
typedef struct {
  int count;
  DLGITEM item[];   //unsized array

} DLGITEMS, *LPDLGITEMS;

typedef struct {
  ICLASS Type;
  WORD id;
  char *TextID;
  union {
    struct { 
      BOOL xUI;
      short RetSel;
      short Sel;
      int count;
      char *string[];    // unsized array.
    } cmbo;
    struct {
      int val;
    } chk;
  };

} CTRLINFO, *LPCTRLINFO;

typedef struct {
  int retVal;
  LPDLGTEMPLATE Dlgtemplate;
  Tcl_Interp *interp;
  int CtrlCount;
  LPCTRLINFO CtrlInfo[];   //unsized array

} DLGDATA, *LPDLGDATA;
#pragma warning(default: 4200)


/* Local Prototypes */
static LPWORD lpwAlign ( LPWORD lpIn );
static int
AddDlgItem (LPWORD lpw, ICLASS cls, DWORD flags, WORD id, DWORD helpID, 
            short x, short y, short w, short h, char *text);
static int AddDlgHeader(LPWORD lpw, WORD cntrls, char *title, short w, short h);
static BOOL CALLBACK DialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
static int
DecodeItemCmdString(Tcl_Interp *interp, char *cmd, DLGITEMS **dlgstuff, 
                    DLGDATA **info);
static BOOL InitList (HWND hDlg, WORD id, LPCTRLINFO data);
static BOOL StuffCmboList( char *list, CTRLINFO **data );
static BOOL StuffCheckBox( char *string, CTRLINFO **data );



/*
 *----------------------------------------------------------------------
 * lpwAlign --
 *
 *  Aligns a pointer to a DWORD boundry.
 *
 * Results:
 *  the aligned pointer.
 *
 * Side Effects:
 *  none.
 *
 *----------------------------------------------------------------------
 */
static LPWORD lpwAlign ( LPWORD lpIn ){
    ULONG ul;
    ul = (ULONG) lpIn;
    ul +=3;
    ul >>=2;
    ul <<=2;
    return (LPWORD) ul;
}

/*
 *----------------------------------------------------------------------
 * AddDlgItem --
 *
 *  Builds a proper Win32 extended dialog item template.
 *
 * Results:
 *  The memory length used in sizeof(WORD).
 *
 * Side Effects:
 *  Could over-run the given memory block if it wasn't 
 *  large enough.
 *
 *----------------------------------------------------------------------
 */
static int
AddDlgItem (LPWORD lpw, ICLASS sty, DWORD flags, WORD id, DWORD helpID, short x,
            short y, short w, short h, char *text) {
 LPWORD lpstart;
 LPDLGITEMTEMPLATEEX lpditex;
 LPWSTR lpwsz;
 int nchar;

  lpstart = lpw;

  lpditex = (LPDLGITEMTEMPLATEEX) lpw;
  lpditex->helpID = helpID;
  lpditex->exStyle = 0;
  lpditex->style = WS_CHILD | WS_VISIBLE | flags;
  lpditex->x  = x;
  lpditex->y  = y;
  lpditex->cx = w;
  lpditex->cy = h;
  lpditex->id = id;
  // align class data on a DWORD boundary.
  // MS says creation data needs the alignment,
  // but I found this was wrong and alignment 
  // goes here, instead!
  lpw = lpwAlign((LPWORD)(lpditex + 1));

  *(lpw++) = 0xFFFF;
  *(lpw++) = sty;

  if (text != NULL && *text != '\0') {
    lpwsz = (LPWSTR) lpw;
    nchar = MultiByteToWideChar(CP_ACP, 0, text, -1, lpwsz, 50);
    lpw += nchar;
  } else {
    *(lpw++) = 0;         // no title
  }

  *(lpw++) = 0;           // creation data
  lpw = lpwAlign (lpw);   // align DLGITEMTEMPLATEEX on DWORD boundary

  // return memory length used in sizeof(WORD).
  return (lpw - lpstart);
}

/*
 *----------------------------------------------------------------------
 * AddDlgHeader --
 *
 *  Builds a proper Win32 extended dialog header template.
 *
 * Results:
 *  the memory length used in sizeof(WORD).
 *
 * Side Effects:
 *  Could over-run the given memory block if it wasn't 
 *  large enough.
 *
 *----------------------------------------------------------------------
 */
static int
AddDlgHeader(LPWORD lpw, WORD cntrls, char *title, short w, short h) {
 LPWORD lpstart;
 LPWSTR lpwsz;
 int nchar;
 LPDLGTEMPLATEEX lpdt;

  lpstart = lpw;

  lpdt = (LPDLGTEMPLATEEX) lpw; // Define a dialog box.
  lpdt->signature = 1;          // this is always 1.
  lpdt->dlgVer = 0xFFFF;        // dialog template is an extended type.
  lpdt->helpID = 0;             // help context identifier.
  lpdt->exStyle = 0;            // extended styles
    // normal styles
  lpdt->style = WS_VISIBLE | WS_POPUP | WS_SYSMENU | WS_CAPTION | DS_CENTER | 
      DS_MODALFRAME | DS_SETFOREGROUND | DS_NOFAILCREATE | DS_SETFONT | 
      DS_CONTEXTHELP;
  lpdt->cdit = cntrls;          // number of controls
  lpdt->x  = 0;
  lpdt->y  = 0;
  lpdt->cx = w;
  lpdt->cy = h;
  lpw = (LPWORD) (lpdt + 1);    // an alignment might be needed here.
                                //  we'll see.
  *(lpw++) = 0;                 // menu
  *(lpw++) = 0;                 // dialog box class
  lpwsz = (LPWSTR) lpw;         // title
  nchar = MultiByteToWideChar(CP_ACP, 0, title, -1, lpwsz, 50);
  lpw   += nchar;
  *(lpw++) = 8;                 // set font pitch
  *(lpw++) = FW_NORMAL;         // set font weight
  *(lpw++) = FALSE;             // set italic
  lpwsz = (LPWSTR) lpw;         // set font name
  nchar = MultiByteToWideChar(CP_ACP, 0, "MS Sans Serif", -1, lpwsz, 25);
  lpw   += nchar;
  lpw = lpwAlign (lpw);  // align next DLGITEMTEMPLATEEX on a DWORD boundary

  //return memory length used by the header in sizeof(WORD).
  return (lpw - lpstart);
}

/*
 *----------------------------------------------------------------------
 * DialogProc --
 *
 *  callback routine for the dialog.
 *
 * Results:
 *  depends
 *
 * Side Effects:
 *----------------------------------------------------------------------
 */
static BOOL CALLBACK 
DialogProc( HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam ) {
 auto BOOL bRet = FALSE;

  switch (uMsg) {

  case WM_INITDIALOG: {
     auto LPDLGDATA cust = (LPDLGDATA) lParam;
     auto int i;

      /* place the pointer to our data structure 
       * in the dialog's user space */
      SetWindowLong(hDlg, DWL_USER, (LONG) cust);

      for (i=0;i<cust->CtrlCount;i++) {
        switch (cust->CtrlInfo[i]->Type) {
        case ComboBox:
          // Turns on 'extended UI' for the combobox.
          if (cust->CtrlInfo[i]->cmbo.xUI) 
              SendDlgItemMessage(hDlg, cust->CtrlInfo[i]->id, CB_SETEXTENDEDUI, 1, 0);
          // Stuffs the combobox with entries.
          if(!InitList(hDlg, cust->CtrlInfo[i]->id, cust->CtrlInfo[i])) 
              EndDialog(hDlg, 0);
          break;
        case Button:
          if (cust->CtrlInfo[i]->chk.val == 1)
              CheckDlgButton(hDlg, cust->CtrlInfo[i]->id, BST_CHECKED);
          else CheckDlgButton(hDlg, cust->CtrlInfo[i]->id, BST_UNCHECKED);
          break;
        }
      }
      bRet = TRUE;
    }
    break;

  case WM_ENTERIDLE:
    // let XiRC redraw and do pending events, if any.
    while (Tcl_DoOneEvent(TCL_DONT_WAIT));
    bRet = FALSE;
    break;

  case WM_COMMAND: {
     auto LPDLGDATA cust = (LPDLGDATA) GetWindowLong(hDlg, DWL_USER);
     auto char *chTemp[2], *out;
     auto BOOL freeflag = FALSE;
     auto int i,s;

      switch (LOWORD(wParam)) {
      case IDOK:
      for (i=0;i<cust->CtrlCount;i++) {
        switch (cust->CtrlInfo[i]->Type) {
        case ComboBox:
          if (cust->CtrlInfo[i]->cmbo.RetSel != 0) {
            chTemp[1] = (char *) xmalloc(33);
            freeflag = TRUE;
            s = SendDlgItemMessage(hDlg, cust->CtrlInfo[i]->id, CB_GETCURSEL , 0, 0);
            _itoa(s,chTemp[1],10);
            break;
          }
        case Edit:
          chTemp[1] = (char *) xmalloc(256);
          GetDlgItemText( hDlg, cust->CtrlInfo[i]->id, chTemp[1], 256 );
          freeflag = TRUE;
          break;
        case Button:
          if (IsDlgButtonChecked(hDlg, cust->CtrlInfo[i]->id) == BST_CHECKED)
              chTemp[1] = "1";
          else chTemp[1] = "0";
          break;
        default:
          continue;
        }
        chTemp[0] = cust->CtrlInfo[i]->TextID;
        out = Falc_Merge(2,chTemp);
        xfree(cust->CtrlInfo[i]->TextID);
        Tcl_AppendElement(cust->interp,out);
        if (freeflag) { freeflag = FALSE; xfree(chTemp[1]); }
        xfree(out);
      }

      /* fall through */

      case IDCANCEL:
        EndDialog(hDlg, uMsg);
      }  
    }
    break;

  }

  return (bRet);
}  

extern int CmdDialog(ClientData clientData,
                     Tcl_Interp *interp,int argc,char *argv[]) {
 char *title, *helpfile;
 int i,j,size,parenthesized,result,dlgitems=0;
 char *p,*element,*next;
 short dlgWidth, dlgHeight;
 LPDLGITEMS dlgstuff;
 LPWORD lpw;
 int newmemsize,headerlength;
 LRESULT retVal;
 int ret = TCL_OK;
 LPDLGDATA info;
 LPVOID lpMsgBuf;
 HWND hwndOwner;


  ItemIDCounter = IDHELP + 2;

  title = DEFTITLE;
  helpfile = NULL;
  dlgstuff = (LPDLGITEMS) xmalloc(sizeof(DLGITEMS));
  dlgstuff->count=0;

  info = (LPDLGDATA) xmalloc(sizeof(DLGDATA));
  info->interp = interp;
  info->CtrlCount=0;
  hwndOwner = Falc_GetXiRChWnd(interp);

  if (argc > 3) {
   BOOL ExitLoopFlag = FALSE;

    for (j=1;!ExitLoopFlag && j<(argc-3);j++) {
      if (argv[j][0] == '-') {
        switch (argv[j][1]) {
        case 't': /* title */
          title = argv[++j]; 
          break;
        case 'h': /* helpfile */
          helpfile = Falc_DoFileName(argv[++j]);
          break;
        default:
          Tcl_AppendResult(interp,argv[0],": ",argv[j]," is an unknown option.",NULL);
          return TCL_ERROR;
        }
      } else {
        ExitLoopFlag = TRUE;
        --j;
      }
    }
    if (j+3 != argc) goto Usage;
  } else {
Usage:
    Tcl_AppendResult(interp,"Usage: ",argv[0]," [-t <Title>] [-h <helpfile>] <width> <height> <DialogItemsList>.",NULL);
    if (helpfile) xfree(helpfile);
    return TCL_ERROR;
  }

  // set the width
  if (Falc_GetShort(interp, argv[j++], &dlgWidth) != TCL_OK) {
    if (helpfile) xfree(helpfile);
    return TCL_ERROR;
  }
  // ensure we have the space for the OK and Cancel buttons
  if (dlgWidth < 116) dlgWidth = 116;

  // set the height
  if (Falc_GetShort(interp, argv[j++], &dlgHeight) != TCL_OK) {
    if (helpfile) xfree(helpfile);
    return TCL_ERROR;
  }
  dlgHeight += 16;  //add the space for the OK and Cancel buttons

  p=argv[argc-1];

  // foreach item from the itemlist, build it. increment item count.
  do {
    char *DlgItemCmdString;
    result = Falc_FindElement(NULL, p, &element, &next, &size, &parenthesized);
    if (result != TCL_OK) goto Usage;
    if (*element == NULL) break;

    DlgItemCmdString = (char *) xmalloc (size+1);

    if (parenthesized) {
      memcpy((VOID *) DlgItemCmdString, (VOID *) element, (size_t) size);
      DlgItemCmdString[size] = '\0';
    } else {
      Falc_CopyAndCollapse(size, element, DlgItemCmdString);
    }
    p=next;

    // decode item commandline
    if ( DecodeItemCmdString(interp,DlgItemCmdString,&dlgstuff,&info) == TCL_ERROR) {
      xfree(DlgItemCmdString);
      if (helpfile) xfree(helpfile);
      return TCL_ERROR;
    }
    dlgitems++;
    xfree(DlgItemCmdString);

  } while (*p != NULL);


  // build the dialog template header in the memory we'll
  // use for the whole template
  lpw = (LPWORD) xmalloc (512);
  headerlength =
      AddDlgHeader(lpw, (dlgitems+2), title, dlgWidth, dlgHeight);

  // calculate the proper memory needs for our complete template
  newmemsize = headerlength;
  for (i=0;i<dlgstuff->count;i++) {
    newmemsize += dlgstuff->item[i].length;
  }
  newmemsize += 40;  // the size of the OK and cancel buttons
  lpw = (LPWORD) xrealloc (lpw, (newmemsize * sizeof(WORD)));
  info->Dlgtemplate = (LPDLGTEMPLATE) lpw;
  lpw += headerlength;

  // append to the header all the item templates.
  for (i=0;i<dlgstuff->count;i++) {
    memcpy((void *) lpw, (void *) dlgstuff->item[i].mem , 
        (size_t) (dlgstuff->item[i].length * sizeof(WORD)));
    lpw += dlgstuff->item[i].length;
    xfree(dlgstuff->item[i].mem);
  }
  xfree(dlgstuff);
  lpw += AddDlgItem(lpw, Button, BS_DEFPUSHBUTTON|WS_TABSTOP, IDOK, 0,
      ((dlgWidth / 2) - 52), (dlgHeight - 16), 50, 14, "OK");
  lpw += AddDlgItem(lpw, Button, BS_PUSHBUTTON|WS_TABSTOP, IDCANCEL, 0,
      ((dlgWidth / 2) + 2), (dlgHeight - 16), 50, 14, "Cancel");


  retVal = DialogBoxIndirectParam(hInstance, (LPDLGTEMPLATE) info->Dlgtemplate, 
      hwndOwner, (DLGPROC) DialogProc, (LPARAM) info);
  if (retVal == -1) {
    // Retrieve the error message.
    FormatMessage(
      FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, NULL,
      GetLastError(), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
      (LPTSTR) &lpMsgBuf, 0, NULL );
    Tcl_AppendResult(interp,"DialogBoxIndirectParam() failed!  GetLastError() returned: ",lpMsgBuf,NULL);
    // Free the buffer.
    LocalFree(lpMsgBuf);
    ret = TCL_ERROR;
  }


CleanUpAndLeave:
  xfree(info->Dlgtemplate);
  if (helpfile) xfree(helpfile);
  xfree(info);
  return ret;

}

/*
/tcl echo [FALC_dialog 150 44 [list [list combo -x cmb1 15 15 104 67 [list "entry1" "entry2" "entry3"]]]]
/tcl echo [FALC_dialog -t "POP3 Email Checker" 150 84 [list [list label lbl1 5 15 20 10 "Host:"] [list edit host 25 13 104 12 "mail.rutgers.edu"] [list label lbl1 5 30 20 10 "User:"] [list edit user 25 28 104 12 "sammy1"] [list label lbl1 5 45 20 10 "Pass:"] [list edit pass 25 43 104 12 "asdasd"] [list check -v no autochk 14 62 70 10 "autocheck every:"] [list edit timer 84 60 14 12 "5"] [list label lbl1 100 63 30 10 "minutes"]]]
*/

static int
DecodeItemCmdString (Tcl_Interp *interp, char *itemString, DLGITEMS **dlgstuff, DLGDATA **info) {
 int newmemsize;
 int j,size,parenthesized,result,count=0;
 char *element,*next, *cmd,
   *cmdarray[12];  // guess 5 options to be the max
 BOOL ExitLoopFlag = FALSE;
 DWORD helpID = 0;
 ICLASS cls;
 CTRLTYPE CtrlType;
 DWORD flags;
 int temp;
 short x,y,w,h;
 LPWORD lpw;
 WORD cntrlID;
 BOOL xUI = FALSE;
 enum { yes = 1, not } HaveChkVal = not;
 short cmboSel = -1;

  cmd = itemString;

  do {
    char *CmdString;
    result = Falc_FindElement(NULL, cmd, &element, &next, &size, &parenthesized);
    if (result != TCL_OK) goto Usage;
    if (*element == NULL) break;
    CmdString = (char *) xmalloc (size+1);
    if (parenthesized) {
      memcpy((void *) CmdString, (void *) element, (size_t) size);
      CmdString[size] = '\0';
    } else {
      Falc_CopyAndCollapse(size, element, CmdString);
    }
    cmd=next;
    cmdarray[count++] = CmdString;
  } while (*cmd != NULL);



  if (count > 5) {
    if (!strncmp(cmdarray[0],"combo",5)) {
      CtrlType = combo;
      cls = ComboBox;
      flags = CBS_DROPDOWNLIST | CBS_DISABLENOSCROLL | CBS_AUTOHSCROLL | WS_VSCROLL | 
          CBS_HASSTRINGS | WS_TABSTOP;
    }
    else if (!strncmp(cmdarray[0],"label",5)) {
      CtrlType = label;
      cls = Static;
      flags = SS_NOPREFIX;
    }
    else if (!strncmp(cmdarray[0],"check",5)) {
      CtrlType = check;
      cls = Button;
      flags = BS_AUTOCHECKBOX | WS_TABSTOP;
    }
    else if (!strncmp(cmdarray[0],"edit",4)) {
      CtrlType = edit;
      cls = Edit;
      flags = ES_AUTOHSCROLL | WS_BORDER | WS_TABSTOP;
    }
    else goto Usage;

    xfree(cmdarray[0]);

    for (j=1;!ExitLoopFlag && j<(count-6);j++) {
      if (cmdarray[j][0] == '-') {
        switch (cmdarray[j][1]) {
        case 'x': /* extended UI */
          if (CtrlType != combo) goto Usage;
          xUI = TRUE;
          break;
        case 'h': /* context ID */
          if (Falc_GetInt(interp, cmdarray[j++], &temp) != TCL_OK) {
            return TCL_ERROR;
          }
          helpID = (DWORD) temp;
          break;
        case 'e':
          if (CtrlType != combo) goto Usage;
          flags &= ~(CBS_DROPDOWNLIST);
          flags |= CBS_DROPDOWN;
          break;
        case 's':
          if (CtrlType != combo) goto Usage;
          xfree(cmdarray[j]);
          if (Falc_GetShort(interp, cmdarray[++j], &cmboSel) != TCL_OK) {
            goto Usage;
          }
          break;
        case 'v': {
           int onoff;
            if (CtrlType != check) goto Usage;
            xfree(cmdarray[j]);
            if (Falc_GetBoolean(interp, cmdarray[++j], &onoff) != TCL_OK) {
              goto Usage;
            }
            if (onoff == 1) HaveChkVal = yes;
          }
          break;
        default:
          xfree(cmdarray[j]);
          Tcl_AppendResult(interp, "Unknown option in DialogItem: ", itemString, NULL);
          return TCL_ERROR;
        }
        xfree(cmdarray[j]);
      } else {
        ExitLoopFlag = TRUE;
        --j;
      }
    }
  } else {
Usage:
    Tcl_AppendResult(interp,"Usage for a DialogItem is: <Type> [<options>] <ID> <x> <y> <width> <height> [<value>].",NULL);
    return TCL_ERROR;
  }

  newmemsize = sizeof(DLGDATA)+(sizeof(LPCTRLINFO)*((*info)->CtrlCount+1));
  *info = (LPDLGDATA) xrealloc(*info,newmemsize);
  (*info)->CtrlInfo[(*info)->CtrlCount] = (LPCTRLINFO) xmalloc(sizeof(CTRLINFO));
  (*info)->CtrlInfo[(*info)->CtrlCount]->TextID = cmdarray[j++];
  (*info)->CtrlInfo[(*info)->CtrlCount]->id = cntrlID = ItemIDCounter++;
  (*info)->CtrlInfo[(*info)->CtrlCount]->Type = cls;
  if (xUI) (*info)->CtrlInfo[(*info)->CtrlCount]->cmbo.xUI = TRUE;
  if ( HaveChkVal != not ) (*info)->CtrlInfo[(*info)->CtrlCount]->chk.val = HaveChkVal;
  if ( cmboSel != -1) {
    (*info)->CtrlInfo[(*info)->CtrlCount]->cmbo.RetSel = 1;
    (*info)->CtrlInfo[(*info)->CtrlCount]->cmbo.Sel = cmboSel;
  }

  (*info)->CtrlCount++;



  if (Falc_GetShort(interp, cmdarray[j], &x) != TCL_OK) {
    return TCL_ERROR;
  }
  xfree(cmdarray[j++]);


  if (Falc_GetShort(interp, cmdarray[j], &y) != TCL_OK) {
    return TCL_ERROR;
  }
  xfree(cmdarray[j++]);

  if (Falc_GetShort(interp, cmdarray[j], &w) != TCL_OK) {
    return TCL_ERROR;
  }
  xfree(cmdarray[j++]);

  if (Falc_GetShort(interp, cmdarray[j], &h) != TCL_OK) {
    return TCL_ERROR;
  }
  xfree(cmdarray[j++]);

  newmemsize = sizeof(DLGITEMS)+(sizeof(DLGITEM)*((*dlgstuff)->count+1));
  *dlgstuff = (LPDLGITEMS) xrealloc(*dlgstuff,newmemsize);

  lpw = (LPWORD) xmalloc (1024);
  ((*dlgstuff)->item[(*dlgstuff)->count]).mem = lpw;

  switch (cls) {
  case ComboBox:
    ((*dlgstuff)->item[(*dlgstuff)->count]).length = 
        AddDlgItem(lpw, cls, flags, cntrlID, helpID, x, y, w, h, NULL);
    if (! StuffCmboList(cmdarray[j],&(*info)->CtrlInfo[(*info)->CtrlCount-1])) {
      goto Usage;
    }
    break;

  default:
    ((*dlgstuff)->item[(*dlgstuff)->count]).length = 
        AddDlgItem(lpw, cls, flags, cntrlID, helpID, x, y, w, h, count-1 == j ? cmdarray[j] : NULL);
    break;

  }
  (*dlgstuff)->count++;

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * InitList --
 *
 *  fills the combobox when the dialog is created.
 *
 * Results:
 *  TRUE
 *
 * Side Effects:
 *  deallocates some memory.
 *
 *----------------------------------------------------------------------
 */
static BOOL
InitList (HWND hDlg, WORD id, LPCTRLINFO data) {
 int i;

  SendDlgItemMessage(hDlg, id, CB_RESETCONTENT, 0, 0);
  for (i=0;i<data->cmbo.count;i++) {
    SendDlgItemMessage(hDlg, id, CB_ADDSTRING, 0, (LPARAM) data->cmbo.string[i]);
    xfree(data->cmbo.string[i]);
  };
  SendDlgItemMessage(hDlg, id, CB_SETCURSEL , (WPARAM) data->cmbo.Sel, 0);
  return TRUE;
}

static BOOL StuffCmboList( char *list, CTRLINFO **data ) {
 int newmemsize;
 int size,parenthesized,result,count=0;
 char *element,*next, *cmd;

  cmd = list;

  do {
    result = Falc_FindElement(NULL, cmd, &element, &next, &size, &parenthesized);
    if (result != TCL_OK) return FALSE;
    if (*element == NULL) break;

    newmemsize = sizeof(CTRLINFO)+(sizeof(char *) * ((*data)->cmbo.count+1));
    *data = (LPCTRLINFO) xrealloc(*data,newmemsize);
    (*data)->cmbo.string[(*data)->cmbo.count] = (char *) xmalloc (size+1);

    if (parenthesized) {
      memcpy((void *) (*data)->cmbo.string[(*data)->cmbo.count], 
          (void *) element, (size_t) size);
      (*data)->cmbo.string[(*data)->cmbo.count][size] = '\0';
    } else {
      Falc_CopyAndCollapse(size, element, 
          (*data)->cmbo.string[(*data)->cmbo.count]);
    }
    cmd=next;
    (*data)->cmbo.count++;

  } while (*cmd != NULL);

  xfree(list);
  return TRUE;
}

static BOOL StuffCheckBox( char *string, CTRLINFO **data ) {
 int onoff;

  if (Falc_GetBoolean((Tcl_Interp *)NULL, string, &onoff) != TCL_OK) {
    return FALSE;
  }

  (*data)->chk.val = onoff;

  return TRUE;
}

