/***********************************************************
 ** AskBox.cpp  --  part of the Falcon DLL Extension v1.5
 **
 ** Abstract: procedures for doing messageboxes.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux   mailto:davygrvy@bigfoot.com
 **********************************************************/

#include "falcon.h"
#include "util.h"

#include <winuser.h>

extern dyn_AppendResult Tcl_AppendResult;
extern dyn_AppendElement Tcl_AppendElement;
extern dyn_DoOneEvent Tcl_DoOneEvent;
extern dyn_GlobalEval Tcl_GlobalEval;
extern dyn_ResetResult Tcl_ResetResult;
extern BOOL InXiRCON;

/*
 * For determining the default button in the message box code
 */
#define CDLG_ABORT  1
#define CDLG_RETRY  2
#define CDLG_IGNORE 3
#define CDLG_OK	    4
#define CDLG_CANCEL 5
#define CDLG_YES    6
#define CDLG_NO     7

typedef struct {
    char *hlpfilename;
    int hlpindex;
} CMDLGCUSTDATA, *LPCMDLGCUSTDATA;

LPCMDLGCUSTDATA cust;

typedef struct {
    MSGBOXPARAMS parms;
    int done;
    int retVal;	      /* 0 if OK,  */
    DWORD errorVal;   /* 0 if no error or Cancel was pressed */
} CMDLGMSGBOXINFO, *LPCMDLGMSGBOXINFO;

/* Protos */
static DWORD WINAPI MsgBoxThread(LPVOID arg);


/*
 *----------------------------------------------------------------------
 * MsgBoxDlgHook --
 *
 *  Hook function that is used to invoke help.
 *----------------------------------------------------------------------
 */
/*VOID CALLBACK MsgBoxCallback(LPHELPINFO lpHelpInfo) {

  if (cust != NULL) {
    WinHelp((HWND)lpHelpInfo->hItemHandle, cust->hlpfilename, HELP_CONTEXT, 
        cust->hlpindex);
  }
}*/

/*
 *--------------------------------------------------------------
 *
 * CmdAskBox --
 *
 *  Display the MessageBox common dialog box.
 *
 * Results:
 *  The button that was pressed
 *
 * Side effects:
 *  Creates a thread to manage the dialog box while it exists. 
 *
 *--------------------------------------------------------------
 */
extern int CmdAskBox(ClientData clientData,
                     Tcl_Interp *interp,int argc,char *argv[]) {
 //Tk_Window topwin = (Tk_Window) clientData;
 //Tk_Window tkwin;
 int arg;
 DWORD buttons = 0xFFFFFFFFL;
 DWORD icon = 0xFFFFFFFFL;
 DWORD dflt = 0xFFFFFFFFL;
 //DWORD help = 0;
 int len;
 char *text = "";
 char *textOut = NULL;
 char *title = DEFTITLE;
 char *s;
 char buff[33];
 int error;
 HANDLE owner = NULL;
 HANDLE thread = NULL;
 DWORD threadId;
 LPCMDLGMSGBOXINFO info;
 //CMDLGCUSTDATA cust;


  if (argc < 2) {
Usage:
    Tcl_AppendResult(interp, "wrong # args: ", (char *) NULL);
BadOpt:
    Tcl_AppendResult(interp, "should be \"", argv[0],
        /* " [-m [<TitleText>]]", */
        /* " [-h <helpfile> <context>]", */
        " [-t <Title>]",
        " [-b abort_retry_ignore|ok|ok_cancel|retry_cancel|yes_no|yes_no_cancel]",
        " [-d abort|retry|ignore|ok|cancel|yes|no]",
        " [-i exclamation|info|question|stop|0]",
        " <message>",
        (char *) NULL);
    return TCL_ERROR;
  }

  arg = 1;
  if (argc > 2) {
    while (arg+1 < argc) {
      if (strcmp(argv[arg], "-b") == 0) {
        arg++;
        if (arg == argc) {
          goto Usage;
        }
        if (buttons != 0xFFFFFFFFL) {
          Tcl_AppendResult(interp, "error: specify the -b option only once",
              (char *) NULL);
          return TCL_ERROR;
        }
        if (strcmp(argv[arg], "abort_retry_ignore") == 0) {
          buttons = MB_ABORTRETRYIGNORE;
        } else if (strcmp(argv[arg], "ok") == 0) {
          buttons = MB_OK;
        } else if (strcmp(argv[arg], "ok_cancel") == 0) {
          buttons = MB_OKCANCEL;
        } else if (strcmp(argv[arg], "retry_cancel") == 0) {
          buttons = MB_RETRYCANCEL;
        } else if (strcmp(argv[arg], "yes_no") == 0) {
          buttons = MB_YESNO;
        } else if (strcmp(argv[arg], "yes_no_cancel") == 0) {
          buttons = MB_YESNOCANCEL;
        } else {
          Tcl_AppendResult(interp, "unknown button combination \"",
              argv[arg], "\" for -b option: ",
              "should be one of ",
              "abort_retry_ignore, ok, ok_cancel, ",
              "retry_cancel, yes_no, or yes_no_cancel",
              (char *) NULL);
          return TCL_ERROR;
        }

      } else if (strcmp(argv[arg], "-d") == 0) {
        arg++;
        if (arg == argc) {
          goto Usage;
        }
        if (strcmp(argv[arg], "ok") == 0) {
          dflt = CDLG_OK;
        } else if (strcmp(argv[arg], "cancel") == 0) {
          dflt = CDLG_CANCEL;
        } else if (strcmp(argv[arg], "yes") == 0) {
          dflt = CDLG_YES;
        } else if (strcmp(argv[arg], "no") == 0) {
          dflt = CDLG_NO;
        } else if (strcmp(argv[arg], "abort") == 0) {
          dflt = CDLG_ABORT;
        } else if (strcmp(argv[arg], "retry") == 0) {
          dflt = CDLG_RETRY;
        } else if (strcmp(argv[arg], "ignore") == 0) {
          dflt = CDLG_IGNORE;
        } else if (argv[arg][0] == 0) {
          dflt = 0xFFFFFFFFL;
        } else {
          Tcl_AppendResult(interp, "unknown button \"", argv[arg],
              "\" for -d option: should be one ",
              "of ok, cancel, yes, no, abort, retry, ",
              "or ignore", (char *) NULL);
          return TCL_ERROR;
        }

      } else if (strcmp(argv[arg], "-i") == 0) {
        arg++;
        if (arg == argc) {
          goto Usage;
        }
        if (strcmp(argv[arg], "exclamation") == 0) {
          icon = MB_ICONEXCLAMATION;
        } else if (strcmp(argv[arg], "info") == 0) {
          icon = MB_ICONINFORMATION;
        } else if (strcmp(argv[arg], "question") == 0) {
          icon = MB_ICONQUESTION;
        } else if (strcmp(argv[arg], "stop") == 0) {
          icon = MB_ICONERROR;
        } else if (argv[arg][0] == '0') {
          icon = 0;
        } else {
          Tcl_AppendResult(interp, "unknown button \"", argv[arg],
              "\" to -i option: should be one ",
              "of exclamation, info, question, stop",
              "or 0 to specify none", (char *) NULL);
          return TCL_ERROR;
        }

      } else if (strcmp(argv[arg], "-t") == 0) {
        arg++;
        if (arg == argc) {
          goto Usage;
        }
        title = argv[arg];
      } else {
        Tcl_AppendResult(interp, "unknown option \"", argv[arg],
            "\": ", (char *) NULL);
        goto BadOpt;
      }
      arg++;
    }
  }

  text = argv[arg++];

  if (arg != argc) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
        argv[0], "[<options>] <message>", (char *) NULL);
    return TCL_ERROR;
  }

  if (icon == 0xFFFFFFFFL) {
    icon = 0;
  }

  if (buttons == 0xFFFFFFFFL) {
    buttons = MB_OK;
  }
  
  error = 0;

  if (dflt == 0xFFFFFFFFL) {
    dflt = MB_DEFBUTTON1;
  } else {
    if (buttons == MB_ABORTRETRYIGNORE) {
      if (dflt == CDLG_ABORT) {
        dflt = MB_DEFBUTTON1;
      } else if (dflt == CDLG_RETRY) {
        dflt = MB_DEFBUTTON2;
      } else if (dflt == CDLG_IGNORE) {
        dflt = MB_DEFBUTTON3;
      } else {
        error = 1;
      }
    
    } else if (buttons == MB_OK) {
      if (dflt == CDLG_OK) {
        dflt = MB_DEFBUTTON1;
      } else {
        error = 1;
      }
    
    } else if (buttons == MB_OKCANCEL) {
      if (dflt == CDLG_OK) {
        dflt = MB_DEFBUTTON1;
      } else if (dflt == CDLG_CANCEL) {
        dflt = MB_DEFBUTTON2;
      } else {
        error = 1;
      }

    } else if (buttons == MB_RETRYCANCEL) {
      if (dflt == CDLG_RETRY) {
        dflt = MB_DEFBUTTON1;
      } else if (dflt == CDLG_CANCEL) {
        dflt = MB_DEFBUTTON2;
      } else {
        error = 1;
      }

    } else if (buttons == MB_YESNO) {
      if (dflt == CDLG_YES) {
        dflt = MB_DEFBUTTON1;
      } else if (dflt == CDLG_NO) {
        dflt = MB_DEFBUTTON2;
      } else {
        error = 1;
      }

    } else if (buttons == MB_YESNOCANCEL) {
      if (dflt == CDLG_YES) {
        dflt = MB_DEFBUTTON1;
      } else if (dflt == CDLG_NO) {
        dflt = MB_DEFBUTTON2;
      } else if (dflt == CDLG_CANCEL) {
        dflt = MB_DEFBUTTON3;
      } else {
        error = 1;
      }

    } else {
      Tcl_AppendResult(interp, "internal error in \"", argv[0],
          "\"", (char *) NULL);
      return TCL_ERROR;
    }
  }

  if (error) {
    Tcl_AppendResult(interp, "-d option button does not exist in ",
        "specified button combination",
        (char *) NULL);
    return TCL_ERROR;
  }

  len = strlen(text);
  if (len > 0) {
    char *ip, *op;

    if(!(textOut = (char *) xmalloc(len+1))) {
      Tcl_AppendResult(interp,"HeapAlloc on line ",itoa(__LINE__,buff,10)," of file ",__FILE__,NULL);
      return TCL_ERROR;
    }
    ip = text; op = textOut;
    while (*ip != 0) {
      if (*ip == '\\' && ip[1] == 'n') {
        *op = '\n';
        ip++;
      } else {
        *op = *ip;
      }
      ip++; op++;
    }
    *op = 0;
    text = textOut;
  }


  if(!(info=(LPCMDLGMSGBOXINFO) xmalloc(sizeof(CMDLGMSGBOXINFO)))) {
    Tcl_AppendResult(interp,"HeapAlloc on line ",itoa(__LINE__,buff,10)," of file ",__FILE__,NULL);
    return TCL_ERROR;
  }

  info->parms.hwndOwner = Falc_GetXiRChWnd(interp);
  info->parms.cbSize = sizeof(MSGBOXPARAMS);
  info->parms.hInstance = NULL;
  info->parms.lpszIcon = NULL;
  info->parms.dwContextHelpId = 0L;
  info->parms.lpszText = textOut;
  info->parms.lpszCaption = title;
  info->parms.lpfnMsgBoxCallback = NULL; //MsgBoxCallback;
  info->parms.dwStyle = buttons | dflt | icon | MB_SETFOREGROUND; 
  info->done = 0;

  thread = CreateThread(NULL, 0, MsgBoxThread, info, 0, &threadId);

  /* check for error */
  if (!thread) {
    Tcl_AppendResult(interp,argv[0],": error starting non-blocking thread!",NULL);
    return TCL_ERROR;
  } else {
    /* dereference handle */
    CloseHandle( thread );
  }


  /*
   *  Loop, but yield to Tcl. Actually in Beta2 of XiRC this 
   *  was needed. But for beta4, Tcl functions aren't blocking
   *  to IRC events :) I don't know how this command will
   *  behave for re-entracy (like, what if its called a second 
   *  time while the first is still waiting on a return). 
   *  I should test this. 
   */
  while (info->done == 0) {
    Tcl_DoOneEvent(TCL_DONT_WAIT);
  }

  switch (info->retVal) {
  case IDABORT:
    s = "abort"; break;
  case IDCANCEL:
    s = "cancel"; break;
  case IDIGNORE:
    s = "ignore"; break;
  case IDNO:
    s = "no"; break;
  case IDOK:
    s = "ok"; break;
  case IDRETRY:
    s = "retry"; break;
  case IDYES:
    s = "yes"; break;
  default:
    Tcl_AppendResult(interp, "out of memory", (char *) NULL);
    xfree(textOut);
    xfree(info);
    return TCL_ERROR;
  }

  Tcl_AppendResult(interp, s, (char *) NULL);

  xfree(textOut);
  xfree(info);

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * MsgBoxThread --
 *
 *  This creates the dialog box and waits for the result.
 *
 * Results:
 *  Always 0
 *
 * Side Effects:
 *----------------------------------------------------------------------
 */
static DWORD WINAPI MsgBoxThread(LPVOID arg) {
 LPCMDLGMSGBOXINFO info = (LPCMDLGMSGBOXINFO) arg;

  info->retVal = MessageBoxIndirect(&info->parms);
  info->done = 1;
  return 0;
}

