/*************************************************************
 ** util.cpp  --  part of the Falcon DLL Extension v1.5
 **
 ** Abstract:  This file contains all the code that is 
 **            shared and used internally by the dll.
 **            Some (ahh..  most) of it was taken from 
 **            the tcl7.6p2 source.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux  mailto:davygrvy@bigfoot.com
 ************************************************************/

#include "falcon.h"
#include "XiRC_HACKS.h"
#include "util.h"
#include <limits.h>

extern dyn_AppendResult Tcl_AppendResult;
extern dyn_GlobalEval Tcl_GlobalEval;
extern dyn_ResetResult Tcl_ResetResult;

#define UCHAR(c) ((unsigned char) (c))
#define ERANGE    34


/*
 * Prototypes for procedures used internally in this file:
 */

static char Falc_Backslash(char *src, int *readPtr);
static int  ServiceFalcEvent (int flags);
static int  PanicEventProc (Tcl_Interp *interp, Tcl_Event *evPtr, int flags);


/*
 *----------------------------------------------------------------------
 *
 * DoFileName --
 *
 *  Allocates memory, replaces all "/" with "\", and substitutes
 *  environment variables in the filename.
 *
 * Results:
 *  pointer to the new filename.
 *
 * Side effects:
 *  must xfree() the pointer when done.
 *
 *----------------------------------------------------------------------
 */
extern char *Falc_DoFileName (char orgFile[]) {
 char *newFile,*lpEnv,*lpEnvSub;
 int i=0,k,file_pos=0,start_pos,envSub_len,env_len=0;
 BOOL chk_env=FALSE;

  if(!(newFile=(char *)xmalloc(1))) return NULL;
  for(i=0,file_pos=0;orgFile[i]!='\0';i++){
    if(!chk_env){
      if(orgFile[i]=='%'){chk_env=TRUE;start_pos=i;continue;};
      if(!(newFile=(char *)xrealloc(newFile,xmemsize(newFile)+1))) return NULL;
      if(orgFile[i]=='/'){
        *(newFile+(file_pos++))='\\';
      } else {
        *(newFile+(file_pos++))=orgFile[i];
      };
    } else {
      if(orgFile[i]!='%'){env_len++;}
      else {
        chk_env=FALSE;                 /* reset flag */
        if(!(lpEnv=(char *)xmalloc(env_len+1))) return NULL;
        for(k=0;k<env_len;k++)*(lpEnv+k)=orgFile[start_pos+1+k];
        *(lpEnv+k)='\0';
        if(!(lpEnvSub=getenv(lpEnv))){xfree(lpEnv);continue;};
        envSub_len=strlen(lpEnvSub);
        if(!(newFile=(char *)xrealloc(newFile,(file_pos+envSub_len+1)))){
          xfree(lpEnv); return NULL;
        };
        for(k=0;k<envSub_len;k++)*(newFile+(file_pos++))=*(lpEnvSub+k);
        xfree(lpEnv);
      };
    };
  };
  /* Add the terminating NULL to the string */
  if(!(newFile=(char *)xrealloc(newFile,xmemsize(newFile)+1))) return NULL;
  *(newFile+(i)+1)='\0';

  return newFile;
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_IsNumeric --
 *
 *  determines if the string contains only numbers.
 *
 * Results:
 *  True/False.
 *
 * Side effects:
 *  a '.' will return false. Should I add a check for this character?
 *
 *----------------------------------------------------------------------
 */
extern BOOL Falc_IsNumeric (char *text) {
 auto int i,j;

  j=strlen(text);
  for(i=0;i<j;i++) if(text[i]<'0'||text[i]>'9') return FALSE;
  return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_GetBoolean --
 *
 *  Given a string, return a 0/1 boolean value corresponding
 *  to the string. Borrowed from the Tcl76 source code.
 *
 * Results:
 *  The return value is normally TCL_OK;  in this case *boolPtr
 *  will be set to the 0/1 value equivalent to string.  If
 *  string is improperly formed then TCL_ERROR is returned and
 *  an error message will be left in interp->result.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
extern int
Falc_GetBoolean(Tcl_Interp *interp, char *string, int *boolPtr) {
    int i;
    char lowerCase[10], c;
    size_t length;

    /*
     * Convert the input string to all lower-case.
     */

    for (i = 0; i < 9; i++) {
      c = string[i];
      if (c == 0) {
        break;
      }
      if ((c >= 'A') && (c <= 'Z')) {
        c += (char) ('a' - 'A');
      }
      lowerCase[i] = c;
    }
    lowerCase[i] = 0;

    length = strlen(lowerCase);
    c = lowerCase[0];
    if ((c == '0') && (lowerCase[1] == '\0')) {
      *boolPtr = 0;
    } else if ((c == '1') && (lowerCase[1] == '\0')) {
      *boolPtr = 1;
    } else if ((c == 'y') && (strncmp(lowerCase, "yes", length) == 0)) {
      *boolPtr = 1;
    } else if ((c == 'n') && (strncmp(lowerCase, "no", length) == 0)) {
      *boolPtr = 0;
    } else if ((c == 't') && (strncmp(lowerCase, "true", length) == 0)) {
      *boolPtr = 1;
    } else if ((c == 'f') && (strncmp(lowerCase, "false", length) == 0)) {
      *boolPtr = 0;
    } else if ((c == 'o') && (length >= 2)) {
      if (strncmp(lowerCase, "on", length) == 0) {
        *boolPtr = 1;
      } else if (strncmp(lowerCase, "off", length) == 0) {
        *boolPtr = 0;
      } else {
        goto badBoolean;
      }
    } else {
      badBoolean:
        if (interp != (Tcl_Interp *) NULL) {
            Tcl_AppendResult(interp, "expected boolean value but got \"",
                    string, "\"", (char *) NULL);
        }
        return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_GetInt --
 *
 *  Given a string, produce the corresponding integer value.
 *  Borrowed from the Tcl76 source code.
 *
 * Results:
 *  The return value is normally TCL_OK;  in this case *intPtr
 *  will be set to the integer value equivalent to string.  If
 *  string is improperly formed then TCL_ERROR is returned and
 *  an error message will be left in interp->result.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
extern int
Falc_GetInt(Tcl_Interp *interp, char *string, int *intPtr) {
 char *end, *p;
 int i;

  /*
   * Note: use strtoul instead of strtol for integer conversions
   * to allow full-size unsigned numbers, but don't depend on strtoul
   * to handle sign characters;  it won't in some implementations.
   */

  errno = 0;
  for (p = string; isspace(UCHAR(*p)); p++) {
    /* Empty loop body. */
  }
  if (*p == '-') {
    p++;
    i = -(int)strtoul(p, &end, 0);
  } else if (*p == '+') {
    p++;
    i = strtoul(p, &end, 0);
  } else {
    i = strtoul(p, &end, 0);
  }
  if (end == p) {
    badInteger:
    if (interp != (Tcl_Interp *) NULL) {
      Tcl_AppendResult(interp, "expected integer but got \"", string,
          "\"", (char *) NULL);
    }
    return TCL_ERROR;
  }
  if (errno == ERANGE) {
    if (interp != (Tcl_Interp *) NULL) {
      Tcl_AppendResult(interp,"integer value too large to represent",NULL);
      //Tcl_SetErrorCode(interp, "ARITH", "IOVERFLOW",
      //     interp->result, (char *) NULL);
    }
    return TCL_ERROR;
  }
  while ((*end != '\0') && isspace(UCHAR(*end))) {
    end++;
  }
  if (*end != 0) {
    goto badInteger;
  }
  *intPtr = i;
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_GetInt --
 *
 *  Given a string, produce the corresponding integer value.
 *  Borrowed from the Tcl76 source code.
 *
 * Results:
 *  The return value is normally TCL_OK;  in this case *intPtr
 *  will be set to the integer value equivalent to string.  If
 *  string is improperly formed then TCL_ERROR is returned and
 *  an error message will be left in interp->result.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
extern int
Falc_GetShort(Tcl_Interp *interp, char *string, short *i16Ptr) {
 char *end, *p;
 int i;

  /*
   * Note: use strtoul instead of strtol for integer conversions
   * to allow full-size unsigned numbers, but don't depend on strtoul
   * to handle sign characters;  it won't in some implementations.
   */

  errno = 0;
  for (p = string; isspace(UCHAR(*p)); p++) {
    /* Empty loop body. */
  }
  if (*p == '-') {
    p++;
    i = -(int)strtoul(p, &end, 0);
  } else if (*p == '+') {
    p++;
    i = strtoul(p, &end, 0);
  } else {
    i = strtoul(p, &end, 0);
  }
  if (end == p) {
badInteger:
    if (interp != (Tcl_Interp *) NULL) {
      Tcl_AppendResult(interp, "expected short integer but got \"", string,
          "\"", (char *) NULL);
    }
    return TCL_ERROR;
  }
  if (errno == ERANGE) {
TooBig:
    if (interp != (Tcl_Interp *) NULL) {
      Tcl_AppendResult(interp,"__i16 (short int) value too large to represent",NULL);
      //Tcl_SetErrorCode(interp, "ARITH", "IOVERFLOW",
      //     interp->result, (char *) NULL);
    }
    return TCL_ERROR;
  }
  while ((*end != '\0') && isspace(UCHAR(*end))) {
    end++;
  }
  if (*end != 0) {
    goto badInteger;
  }
  if ( i < SHRT_MIN || i > SHRT_MAX ) goto TooBig;
  *i16Ptr = (short) i;
  return TCL_OK;
}

/*
 * The following values are used in the flags returned by Tcl_ScanElement
 * and used by Tcl_ConvertElement.  The value TCL_DONT_USE_BRACES is also
 * defined in tcl.h;  make sure its value doesn't overlap with any of the
 * values below.
 *
 * TCL_DONT_USE_BRACES -  1 means the string mustn't be enclosed in
 *                        braces (e.g. it contains unmatched braces,
 *                        or ends in a backslash character, or user
 *                        just doesn't want braces);  handle all
 *                        special characters by adding backslashes.
 * USE_BRACES -           1 means the string contains a special
 *                        character that can be handled simply by
 *                        enclosing the entire argument in braces.
 * BRACES_UNMATCHED -     1 means that braces aren't properly matched
 *                        in the argument.
 */

#define TCL_DONT_USE_BRACES   1
#define USE_BRACES            2
#define BRACES_UNMATCHED      4


/*
 *----------------------------------------------------------------------
 *
 * Falc_ScanElement --
 *
 *  This procedure is a companion procedure to Tcl_ConvertElement.
 *  It scans a string to see what needs to be done to it (e.g.
 *  add backslashes or enclosing braces) to make the string into
 *  a valid Tcl list element.
 *
 * Results:
 *  The return value is an overestimate of the number of characters
 *  that will be needed by Tcl_ConvertElement to produce a valid
 *  list element from string.  The word at *flagPtr is filled in
 *  with a value needed by Tcl_ConvertElement when doing the actual
 *  conversion.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

extern int Falc_ScanElement(char *string, int *flagPtr) {
 int flags, nestingLevel;
 register char *p;

  /*
   * This procedure and Tcl_ConvertElement together do two things:
   *
   * 1. They produce a proper list, one that will yield back the
   * argument strings when evaluated or when disassembled with
   * Tcl_SplitList.  This is the most important thing.
   * 
   * 2. They try to produce legible output, which means minimizing the
   * use of backslashes (using braces instead).  However, there are
   * some situations where backslashes must be used (e.g. an element
   * like "{abc": the leading brace will have to be backslashed.  For
   * each element, one of three things must be done:
   *
   * (a) Use the element as-is (it doesn't contain anything special
   * characters).  This is the most desirable option.
   *
   * (b) Enclose the element in braces, but leave the contents alone.
   * This happens if the element contains embedded space, or if it
   * contains characters with special interpretation ($, [, ;, or \),
   * or if it starts with a brace or double-quote, or if there are
   * no characters in the element.
   *
   * (c) Don't enclose the element in braces, but add backslashes to
   * prevent special interpretation of special characters.  This is a
   * last resort used when the argument would normally fall under case
   * (b) but contains unmatched braces.  It also occurs if the last
   * character of the argument is a backslash or if the element contains
   * a backslash followed by newline.
   *
   * The procedure figures out how many bytes will be needed to store
   * the result (actually, it overestimates).  It also collects information
   * about the element in the form of a flags word.
   */

  nestingLevel = 0;
  flags = 0;
  if (string == NULL) {
    string = "";
  }
  p = string;
  if ((*p == '{') || (*p == '"') || (*p == 0)) {
    flags |= USE_BRACES;
  }
  for ( ; *p != 0; p++) {
    switch (*p) {
    case '{':
      nestingLevel++;
      break;
    case '}':
      nestingLevel--;
      if (nestingLevel < 0) {
        flags |= TCL_DONT_USE_BRACES | BRACES_UNMATCHED;
      }
      break;
    case '[':
    case '$':
    case ';':
    case ' ':
    case '\f':
    case '\n':
    case '\r':
    case '\t':
    case '\v':
      flags |= USE_BRACES;
      break;
    case '\\':
      if ((p[1] == 0) || (p[1] == '\n')) {
        flags = TCL_DONT_USE_BRACES;
      } else {
        int size;

        (void) Falc_Backslash( p, &size);
        p += size - 1;
        flags |= USE_BRACES;
      }
      break;
    }
  }
  if (nestingLevel != 0) {
    flags = TCL_DONT_USE_BRACES | BRACES_UNMATCHED;
  }
  *flagPtr = flags;

  /*
   * Allow enough space to backslash every character plus leave
   * two spaces for braces.
   */

  return ( 2 * (p-string) + 2);
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_ConvertElement --
 *
 *  This is a companion procedure to Tcl_ScanElement.  Given the
 *  information produced by Tcl_ScanElement, this procedure converts
 *  a string to a list element equal to that string.
 *
 * Results:
 *  Information is copied to *dst in the form of a list element
 *  identical to src (i.e. if Tcl_SplitList is applied to dst it
 *  will produce a string identical to src).  The return value is
 *  a count of the number of characters copied (not including the
 *  terminating NULL character).
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
extern int
Falc_ConvertElement(register char *src, char *dst, int flags) {
 register char *p = dst;

  /*
   * See the comment block at the beginning of the Tcl_ScanElement
   * code for details of how this works.
   */

  if ((src == NULL) || (*src == 0)) {
    p[0] = '{';
    p[1] = '}';
    p[2] = 0;
    return 2;
  }
  if ((flags & USE_BRACES) && !(flags & TCL_DONT_USE_BRACES)) {
    *p = '{';
    p++;
    for ( ; *src != 0; src++, p++) {
      *p = *src;
    }
    *p = '}';
    p++;
  } else {
    if (*src == '{') {
      /*
       * Can't have a leading brace unless the whole element is
       * enclosed in braces.  Add a backslash before the brace.
       * Furthermore, this may destroy the balance between open
       * and close braces, so set BRACES_UNMATCHED.
       */

      p[0] = '\\';
      p[1] = '{';
      p += 2;
      src++;
      flags |= BRACES_UNMATCHED;
    }
    for (; *src != 0 ; src++) {
      switch (*src) {
      case ']':
      case '[':
      case '$':
      case ';':
      case ' ':
      case '\\':
      case '"':
        *p = '\\';
        p++;
        break;
      case '{':
      case '}':
        /*
         * It may not seem necessary to backslash braces, but
         * it is.  The reason for this is that the resulting
         * list element may actually be an element of a sub-list
         * enclosed in braces (e.g. if Tcl_DStringStartSublist
         * has been invoked), so there may be a brace mismatch
         * if the braces aren't backslashed.
         */

        if (flags & BRACES_UNMATCHED) {
          *p = '\\';
          p++;
        }
        break;
      case '\f':
        *p = '\\';
        p++;
        *p = 'f';
        p++;
        continue;
      case '\n':
        *p = '\\';
        p++;
        *p = 'n';
        p++;
        continue;
      case '\r':
        *p = '\\';
        p++;
        *p = 'r';
        p++;
        continue;
      case '\t':
        *p = '\\';
        p++;
        *p = 't';
        p++;
        continue;
      case '\v':
        *p = '\\';
        p++;
        *p = 'v';
        p++;
        continue;
      }
      *p = *src;
      p++;
    }
  }
  *p = '\0';
  return (p - dst);
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_Merge --
 *
 *  Given a collection of strings, merge them together into a
 *  single string that has proper Tcl list structured (i.e.
 *  Tcl_SplitList may be used to retrieve strings equal to the
 *  original elements, and Tcl_Eval will parse the string back
 *  into its original elements).
 *
 * Results:
 *  The return value is the address of a dynamically-allocated
 *  string containing the merged list.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
extern char *Falc_Merge(int argc, char **argv) {
#define LOCAL_SIZE 20
 int localFlags[LOCAL_SIZE], *flagPtr;
 int numChars;
 char *result;
 register char *dst;
 int i;

  /*
   * Pass 1: estimate space, gather flags.
   */

  if (argc <= LOCAL_SIZE) {
    flagPtr = localFlags;
  } else {
    flagPtr = (int *) xmalloc((unsigned) argc*sizeof(int));
  }
  numChars = 1;
  for (i = 0; i < argc; i++) {
    numChars += Falc_ScanElement(argv[i], &flagPtr[i]) + 1;
  }

  /*
   * Pass two: copy into the result area.
   */

  result = (char *) xmalloc((unsigned) numChars);
  dst = result;
  for (i = 0; i < argc; i++) {
    numChars = Falc_ConvertElement(argv[i], dst, flagPtr[i]);
    dst += numChars;
    *dst = ' ';
    dst++;
  }
  if (dst == result) {
    *dst = 0;
  } else {
    dst[-1] = 0;
  }

  if (flagPtr != localFlags) {
    xfree((char *) flagPtr);
  }
  return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_Backslash --
 *
 *  Figure out how to handle a backslash sequence.
 *
 * Results:
 *  The return value is the character that should be substituted
 *  in place of the backslash sequence that starts at src.  If
 *  readPtr isn't NULL then it is filled in with a count of the
 *  number of characters in the backslash sequence.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

static char Falc_Backslash(char *src, int *readPtr) {
 register char *p = src+1;
 char result;
 int count;

  count = 2;

  switch (*p) {
    /*
     * Note: in the conversions below, use absolute values (e.g.,
     * 0xa) rather than symbolic values (e.g. \n) that get converted
     * by the compiler.  It's possible that compilers on some
     * platforms will do the symbolic conversions differently, which
     * could result in non-portable Tcl scripts.
     */

  case 'a':
    result = 0x7;
    break;
  case 'b':
    result = 0x8;
    break;
  case 'f':
    result = 0xc;
    break;
  case 'n':
    result = 0xa;
    break;
  case 'r':
    result = 0xd;
    break;
  case 't':
    result = 0x9;
    break;
  case 'v':
    result = 0xb;
    break;
  case 'x':
    if (isxdigit(UCHAR(p[1]))) {
      char *end;

      result = (char) strtoul(p+1, &end, 16);
      count = end - src;
    } else {
      count = 2;
      result = 'x';
    }
    break;
  case '\n':
    do {
    	p++;
    } while ((*p == ' ') || (*p == '\t'));
    result = ' ';
    count = p - src;
    break;
  case 0:
    result = '\\';
    count = 1;
    break;
  default:
    if (isdigit(UCHAR(*p))) {
      result = (char)(*p - '0');
      p++;
      if (!isdigit(UCHAR(*p))) {
        break;
      }
      count = 3;
      result = (char)((result << 3) + (*p - '0'));
      p++;
      if (!isdigit(UCHAR(*p))) {
        break;
      }
      count = 4;
      result = (char)((result << 3) + (*p - '0'));
      break;
    }
    result = *p;
    count = 2;
    break;
  }

  if (readPtr != NULL) {
    *readPtr = count;
  }
  return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_FindElement --
 *
 *  Given a pointer into a Tcl list, locate the first (or next)
 *  element in the list.
 *
 * Results:
 *  The return value is normally TCL_OK, which means that the
 *  element was successfully located.  If TCL_ERROR is returned
 *  it means that list didn't have proper list structure;
 *  interp->result contains a more detailed error message.
 *
 *  If TCL_OK is returned, then *elementPtr will be set to point
 *  to the first element of list, and *nextPtr will be set to point
 *  to the character just after any white space following the last
 *  character that's part of the element.  If this is the last argument
 *  in the list, then *nextPtr will point to the NULL character at the
 *  end of list.  If sizePtr is non-NULL, *sizePtr is filled in with
 *  the number of characters in the element.  If the element is in
 *  braces, then *elementPtr will point to the character after the
 *  opening brace and *sizePtr will not include either of the braces.
 *  If there isn't an element in the list, *sizePtr will be zero, and
 *  both *elementPtr and *termPtr will refer to the null character at
 *  the end of list.  Note:  this procedure does NOT collapse backslash
 *  sequences.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
extern int
Falc_FindElement(
    Tcl_Interp *interp,   /* Interpreter to use for error reporting. 
                           * If NULL, then no error message is left
                           * after errors. */
    register char *list,  /* String containing Tcl list with zero
                           * or more elements (possibly in braces). */
    char **elementPtr,    /* Fill in with location of first significant
                           * character in first element of list. */
    char **nextPtr,       /* Fill in with location of character just
                           * after all white space following end of
                           * argument (i.e. next argument or end of
                           * list). */
    int *sizePtr,         /* If non-zero, fill in with size of
                           * element. */
    int *bracePtr         /* If non-zero fill in with non-zero/zero
                           * to indicate that arg was/wasn't
                           * in braces. */
){
    register char *p;
    int openBraces = 0;
    int inQuotes = 0;
    int size;

    /*
     * Skim off leading white space and check for an opening brace or
     * quote.   Note:  use of "isascii" below and elsewhere in this
     * procedure is a temporary hack (7/27/90) because Mx uses characters
     * with the high-order bit set for some things.  This should probably
     * be changed back eventually, or all of Tcl should call isascii.
     */

    while (isspace(UCHAR(*list))) {
      list++;
    }
    if (*list == '{') {
      openBraces = 1;
      list++;
    } else if (*list == '"') {
      inQuotes = 1;
      list++;
    }
    if (bracePtr != 0) {
      *bracePtr = openBraces;
    }
    p = list;

    /*
     * Find the end of the element (either a space or a close brace or
     * the end of the string).
     */

    while (1) {
      switch (*p) {

      /*
       * Open brace: don't treat specially unless the element is
       * in braces.  In this case, keep a nesting count.
       */

      case '{':
        if (openBraces != 0) {
          openBraces++;
        }
        break;

      /*
       * Close brace: if element is in braces, keep nesting
       * count and quit when the last close brace is seen.
       */

      case '}':
        if (openBraces == 1) {
          char *p2;

          size = p - list;
          p++;
          if (isspace(UCHAR(*p)) || (*p == 0)) {
            goto done;
          }
          for (p2 = p; (*p2 != 0) && (!isspace(UCHAR(*p2)))
              && (p2 < p+20); p2++) {
            /* null body */
          }
          if (interp != NULL) {
            //Tcl_ResetResult(interp);
            //sprintf(interp->result,
            //  "list element in braces followed by \"%.*s\" instead of space",
            //  (int) (p2-p), p);
          }
          return TCL_ERROR;
        } else if (openBraces != 0) {
          openBraces--;
        }
        break;

      /*
       * Backslash:  skip over everything up to the end of the
       * backslash sequence.
       */

      case '\\': {
        int size;

        (void) Falc_Backslash(p, &size);
        p += size - 1;
        break;
      }

      /*
       * Space: ignore if element is in braces or quotes;  otherwise
       * terminate element.
       */

      case ' ':
      case '\f':
      case '\n':
      case '\r':
      case '\t':
      case '\v':
        if ((openBraces == 0) && !inQuotes) {
          size = p - list;
          goto done;
        }
        break;

      /*
       * Double-quote:  if element is in quotes then terminate it.
       */

      case '"':
        if (inQuotes) {
          char *p2;

          size = p-list;
          p++;
          if (isspace(UCHAR(*p)) || (*p == 0)) {
            goto done;
          }
          for (p2 = p; (*p2 != 0) && (!isspace(UCHAR(*p2)))
              && (p2 < p+20); p2++) {
            /* null body */
          }
          if (interp != NULL) {
            //Tcl_ResetResult(interp);
            //sprintf(interp->result,
            //    "list element in quotes followed by \"%.*s\" %s", (int) (p2-p), p,
            //    "instead of space");
          }
          return TCL_ERROR;
        }
        break;

      /*
       * End of list:  terminate element.
       */

      case 0:
        if (openBraces != 0) {
          if (interp != NULL) {
            //Tcl_SetResult(interp, "unmatched open brace in list", TCL_STATIC);
          }
          return TCL_ERROR;
        } else if (inQuotes) {
          if (interp != NULL) {
            //Tcl_SetResult(interp, "unmatched open quote in list", TCL_STATIC);
          }
          return TCL_ERROR;
        }
        size = p - list;
        goto done;

      }
      p++;
    }

    done:
    while (isspace(UCHAR(*p))) {
      p++;
    }
    *elementPtr = list;
    *nextPtr = p;
    if (sizePtr != 0) {
      *sizePtr = size;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TclCopyAndCollapse --
 *
 *  Copy a string and eliminate any backslashes that aren't in braces.
 *
 * Results:
 *  There is no return value.  Count chars. get copied from src
 *  to dst.  Along the way, if backslash sequences are found outside
 *  braces, the backslashes are eliminated in the copy.
 *  After scanning count chars. from source, a null character is
 *  placed at the end of dst.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
extern void
Falc_CopyAndCollapse(
  int count,            /* Total number of characters to copy
                         * from src. */
  register char *src,   /* Copy from here... */
  register char *dst    /* ... to here. */
){
  register char c;
  int numRead;

  for (c = *src; count > 0; src++, c = *src, count--) {
    if (c == '\\') {
      *dst = Falc_Backslash(src, &numRead);
      dst++;
      src += numRead-1;
      count -= numRead-1;
    } else {
      *dst = c;
      dst++;
    }
  }
  *dst = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_freeProc --
 *
 *  Falcon's internal free() procedure.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  frees memory.
 *
 *----------------------------------------------------------------------
 */
extern void Falc_freeProc(char *blockPtr) {
   xfree(blockPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_GetXiRChWnd --
 *
 *	Retrieves XiRCON's window handle only once and stores it so 
 *  it doesn't need to waste any more memory.
 *
 * Results:
 *	Window's handle for XiRCON's main window.
 *
 * Side effects:
 *	Calls Tcl_ResetResult which leaves >200 bytes of unfreed memory.
 *
 *----------------------------------------------------------------------
 */
extern HWND Falc_GetXiRChWnd(Tcl_Interp *interp) {
  static HWND XiRChWnd = NULL;

  if (XiRChWnd == NULL) {
    /* run [window get_title main] in the interpreter 
       and check the result */
    if (Tcl_GlobalEval(interp,"window get_title main")==TCL_OK) {
      XiRChWnd = FindWindow(NULL,interp->result);
    };
    if (Tcl_ResetResult) {
      Tcl_ResetResult(interp);
    } else {
      HACK_ResetResult(interp);
    };
    //hInst = (HINSTANCE) GetWindowLong(hwndOwner,GWL_HINSTANCE);
  };

  return XiRChWnd;
}

/*
 * The following variables keep track of the event queue.  In addition
 * to the first (next to be serviced) and last events in the queue,
 * we keep track of a "marker" event.  This provides a simple priority
 * mechanism whereby events can be inserted at the front of the queue
 * but behind all other high-priority events already in the queue (this
 * is used for things like a sequence of Enter and Leave events generated
 * during a grab in Tk).
 */

static Tcl_Event *firstEventPtr = NULL;
        /* First pending event, or NULL if none. */
static Tcl_Event *lastEventPtr = NULL;
        /* Last pending event, or NULL if none. */
static Tcl_Event *markerEventPtr = NULL;
        /* Last high-priority event in queue, or
         * NULL if none. */

/*
 *----------------------------------------------------------------------
 *
 * Falc_QueueEvent --
 *
 *  Insert an event into the Falc event queue at one of three
 *  positions: the head, the tail, or before a floating marker.
 *  Events inserted before the marker will be processed in
 *  first-in-first-out order, but before any events inserted at
 *  the tail of the queue.  Events inserted at the head of the
 *  queue will be processed in last-in-first-out order.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
extern void Falc_QueueEvent(
  Tcl_Event *evPtr,               /* Event to add to queue.  The storage
                                   * space must have been allocated by the 
                                   * caller with xmalloc, and it becomes
                                   * the property of the event queue.  It
                                   * will be freed after the event has been
                                   * handled. */
  Falc_QueuePosition position) /* One of FALC_QUEUE_TAIL, FALC_QUEUE_HEAD,
                                   * FALC_QUEUE_MARK. */
{
  if (position == FALC_QUEUE_TAIL) {
    /*
     * Append the event on the end of the queue.
     */
    evPtr->nextPtr = NULL;
    if (firstEventPtr == NULL) {
      firstEventPtr = evPtr;
    } else {
      lastEventPtr->nextPtr = evPtr;
    }
    lastEventPtr = evPtr;
  } else if (position == FALC_QUEUE_HEAD) {
  /*
   * Push the event on the head of the queue.
   */

    evPtr->nextPtr = firstEventPtr;
    if (firstEventPtr == NULL) {
      lastEventPtr = evPtr;
    }
    firstEventPtr = evPtr;
  } else if (position == FALC_QUEUE_MARK) {
    /*
     * Insert the event after the current marker event and advance
     * the marker to the new event.
     */
    if (markerEventPtr == NULL) {
      evPtr->nextPtr = firstEventPtr;
      firstEventPtr = evPtr;
    } else {
      evPtr->nextPtr = markerEventPtr->nextPtr;
      markerEventPtr->nextPtr = evPtr;
    }
    markerEventPtr = evPtr;
    if (evPtr->nextPtr == NULL) {
      lastEventPtr = evPtr;
    }
  }
}

/*
 *----------------------------------------------------------------------
 *
 * ServiceFalcEvent --
 *
 *  Process one event from the event queue.  This routine is called
 *  by the notifier whenever it wants XiRC to process an event.  
 *
 * Results:
 *  The return value is 1 if the procedure actually found an event
 *  to process.  If no processing occurred, then 0 is returned.
 *
 * Side effects:
 *  Invokes all of the event handlers for the highest priority
 *  event in the event queue.  May collapse some events into a
 *  single event or discard stale events.
 *
 *----------------------------------------------------------------------
 */
static int ServiceFalcEvent(
  Tcl_Interp *interp, /* interpreter to grab */
  int flags)          /* Indicates what events should be processed.
                       * May be any combination of FALC_DDE_EVENTS,
                       * FALC_ALL_EVENTS
                       * or other flags defined elsewhere.  Events not
                       * matching this will be skipped for processing
                       * later. */
{
 Tcl_Event *evPtr, *prevPtr;
 Tcl_EventProc *proc;

  /*
   * No event flags is equivalent to FALC_ALL_EVENTS.
   */
    
  if ((flags & FALC_ALL_EVENTS) == 0) {
    flags |= FALC_ALL_EVENTS;
  }

  /*
   * Loop through all the events in the queue until we find one
   * that can actually be handled.
   */

  for (evPtr = firstEventPtr; evPtr != NULL; evPtr = evPtr->nextPtr) {
    /*
     * Call the handler for the event.  If it actually handles the
     * event then free the storage for the event.  There are two
     * tricky things here, but stemming from the fact that the event
     * code may be re-entered while servicing the event:
     *
     * 1. Set the "proc" field to NULL.  This is a signal to ourselves
     *    that we shouldn't reexecute the handler if the event loop
     *    is re-entered.
     * 2. When freeing the event, must search the queue again from the
     *    front to find it.  This is because the event queue could
     *    change almost arbitrarily while handling the event, so we
     *    can't depend on pointers found now still being valid when
     *    the handler returns.
     */

    proc = evPtr->proc;
    evPtr->proc = NULL;
    if ((proc != NULL) && (*proc)(interp, evPtr, flags)) {
      if (firstEventPtr == evPtr) {
        firstEventPtr = evPtr->nextPtr;
        if (evPtr->nextPtr == NULL) {
          lastEventPtr = NULL;
        }
        if (markerEventPtr == evPtr) {
          markerEventPtr = NULL;
        }
      } else {
        for (prevPtr = firstEventPtr; 
             prevPtr->nextPtr != evPtr;
             prevPtr = prevPtr->nextPtr) {
          /* Empty loop body. */
        }
        prevPtr->nextPtr = evPtr->nextPtr;
        if (evPtr->nextPtr == NULL) {
          lastEventPtr = prevPtr;
        }
        if (markerEventPtr == evPtr) {
          markerEventPtr = prevPtr;
        }
      }
      xfree(evPtr);
      return 1;
    } else {
      /*
       * The event wasn't actually handled, so we have to restore
       * the proc field to allow the event to be attempted again.
       */

      evPtr->proc = proc;
    }

    /*
     * The handler for this event asked to defer it.  Just go on to
     * the next event.
     */
    continue;
  }
  return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * Falc_DoOneEvent --
 *
 *  Process a single event of some sort.  If there's no work to
 *  do, wait for an event to occur, then process it.
 *
 * Results:
 *  The return value is 1 if the procedure actually found an event
 *  to process.  If no processing occurred, then 0 is returned (this
 *  can happen if the TCL_DONT_WAIT flag is set or if there are no
 *  event handlers to wait for in the set specified by flags).
 *
 * Side effects:
 *  May delay execution of process while waiting for an event,
 *  unless TCL_DONT_WAIT is set in the flags argument.  Event
 *  sources are invoked to check for and queue events.  Event
 *  handlers may produce arbitrary side effects.
 *
 *----------------------------------------------------------------------
 */
extern int Falc_DoOneEvent(
  Tcl_Interp *interp, /* interpreter to grab */
  int flags)          /* Miscellaneous flag values:  may be any
                       * combination of FALC_ALL_EVENTS, 
                       * FALC_DDE_EVENTS, or others defined 
                       * by event sources. */
{
 //TclEventSource *sourcePtr;
 //Tcl_Time *timePtr;

  /*
   * No event flags is equivalent to FALC_ALL_EVENTS.
   */

  if ((flags & FALC_ALL_EVENTS) == 0) {
    flags |= FALC_ALL_EVENTS;
  }

  /*
   * The core of this procedure is an infinite loop, even though
   * we only service one event.  The reason for this is that we
   * might think we have an event ready (e.g. the connection to
   * the server becomes readable), but then we might discover that
   * there's nothing interesting on that connection, so no event
   * was serviced.  Or, the select operation could return prematurely
   * due to a signal.  The easiest thing in both these cases is
   * just to loop back and try again.
   */

  //while (1) {

    /*
     * Ask to service a queued event, if there are any.
     */

    if (ServiceFalcEvent(interp, flags)) {
      return 1;
    }

    /*
     * There are no events already queued.  Invoke all of the
     * event sources to give them a chance to setup for the wait.
     */

    //blockTimeSet = 0;
    //for (sourcePtr = tclFirstEventSourcePtr; 
    //     sourcePtr != NULL;
    //     sourcePtr = sourcePtr->nextPtr) {
    //  (*sourcePtr->setupProc)(sourcePtr->clientData, flags);
    //}
    //if ((flags & TCL_DONT_WAIT) ||
    //  ((flags & TCL_IDLE_EVENTS) && TclIdlePending())) {
      /*
       * Don't block:  there are idle events waiting, or we don't
       * care about idle events anyway, or the caller asked us not
       * to block.
       */

    //  blockTime.sec = 0;
    //  blockTime.usec = 0;
    //  timePtr = &blockTime;
    //} else if (blockTimeSet) {
    //  timePtr = &blockTime;
    //} else {
    //  timePtr = NULL;
    //}

    /*
     * Wait until an event occurs or the timer expires.
     */

    //if (Tcl_WaitForEvent(timePtr) == TCL_ERROR) {
    //  return 0;
    //}

    /*
     * Give each of the event sources a chance to queue events,
     * then call ServiceEvent and give it another chance to
     * service events.
     */

    //for (sourcePtr = tclFirstEventSourcePtr; 
    //     sourcePtr != NULL;
    //     sourcePtr = sourcePtr->nextPtr) {
    //  (*sourcePtr->checkProc)(sourcePtr->clientData, flags);
    //}
    //if (ServiceFalcEvent(flags)) {
    //  return 1;
    //}

    /*
     * We've tried everything at this point, but nobody had anything
     * to do.  Check for idle events.  If none, either quit or go back
     * to the top and try again.
     */

//idleEvents:
    //if ((flags & FALC_IDLE_EVENTS) && TclServiceIdle()) {
    //  return 1;
    //}
    return 0;
  //}
}
