/*************************************************************
 ** Sockets.cpp  --  part of the Falcon DLL Extension v1.4
 **
 ** Abstract:  This file contains all the code that uses 
 **            the winsock API.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux  mailto:davygrvy@bigfoot.com
 ************************************************************/

#include "falcon.h"
#include "util.h"

extern dyn_AppendResult Tcl_AppendResult;
extern dyn_AppendElement Tcl_AppendElement;
extern dyn_Eval Tcl_Eval;
extern dyn_ResetResult Tcl_ResetResult;
extern dyn_DeleteCommand Tcl_DeleteCommand;
extern HANDLE hInstance;
extern BOOL InXiRCON;

FALC_SOCK winSock;
WSADATA wsaData;

/*
 *----------------------------------------------------------------------
 *
 * CmdIsip --
 *
 *  checks if the given string is an IP address
 *
 * Results:
 *  returns 1/0
 *
 * Side effects:
 *  The inet_addr() function doesn't seem to mind if you leave out 
 *  the "." (dots) in the string. I think that's strange behavior.
 *
 *----------------------------------------------------------------------
 */
int CmdIsip(ClientData clientData,
                   Tcl_Interp *interp,int argc,char *argv[]) {
  if(argc<2){
    Tcl_AppendResult(interp,"Usage: ",argv[0]," <text> ; returns 1 if <text> is an IP address in dotted 4 form. Or zero, if not an IP.",NULL);
    return TCL_ERROR;
  }; 

  if(winSock.inet_addr(argv[1])==INADDR_NONE){
    Tcl_AppendResult(interp,"0",NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp,"1",NULL);
    return TCL_OK;
  };
}

/*
 *----------------------------------------------------------------------
 *
 * CmdIp2Long --
 *
 *  returns the network long IP address from the given text
 *
 * Results:
 *  returns 1/0
 *
 * Side effects:
 *  The inet_addr() function doesn't seem to mind if you leave out 
 *  the "." (dots) in the string. I think that's strange behavior.
 *
 *----------------------------------------------------------------------
 */
int CmdIp2Long(ClientData clientData,
                   Tcl_Interp *interp,int argc,char *argv[]) {
 auto unsigned long addr=0;
 auto char buff[33];

  if(argc<2){
    Tcl_AppendResult(interp,"Usage: ",argv[0]," <text> ; returns the long IP in network-order, if <text> is an IP address in dotted 4 form. Or zero, if not an IP.",NULL);
    return TCL_ERROR;
  };

  if((addr=winSock.inet_addr(argv[1]))!=INADDR_NONE){
    sprintf(buff,"%lu",winSock.htonl(addr));
    Tcl_AppendResult(interp,buff,NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp,"0",NULL);
    return TCL_OK;
  };
}

/*
 *----------------------------------------------------------------------
 *
 * CmdLong2Ip --
 *
 *  returns a dotted4 IP from a network long IP number
 *
 * Results:
 *  see above
 *
 * Side effects:
 *  The inet_addr() function doesn't seem to mind if you leave out 
 *  the "." (dots) in the string. I think that's strange behavior.
 *
 *----------------------------------------------------------------------
 */
int CmdLong2Ip(ClientData clientData,
                   Tcl_Interp *interp,int argc,char *argv[]) {
 auto IN_ADDR addr;
 auto char *buff;

  if(argc<2){
    Tcl_AppendResult(interp,"Usage: ",argv[0]," <text> ; returns the IP in dotted decimal from a long network-order IP address. Or zero, if not an long IP.",NULL);
    return TCL_ERROR;
  }; 

  addr.s_addr = winSock.htonl(strtoul(argv[1],NULL,10));

  if((buff=winSock.inet_ntoa(addr))!=NULL){
    Tcl_AppendResult(interp,buff,NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp,"0",NULL);
    return TCL_OK;
  };
}

/*
 *----------------------------------------------------------------------
 *
 * CmdServ --
 *  Resolves a service name to a port number.
 *
 * Results:
 *  returns the port # from a service name
 *
 * Side effects:
 *  unknown.
 *
 *----------------------------------------------------------------------
 */
int CmdPort(ClientData clientData,
                   Tcl_Interp *interp,int argc,char *argv[]) {
  
 auto struct servent *result;
 auto char buff[33];

  result=NULL;

  if(argc<2){
    Tcl_AppendResult(interp,"Usage: ",argv[0],
        " <ServiceName>|<PortNum> ; resolves a service name to a port number.",NULL);
    return TCL_ERROR;
  };

  if (Falc_IsNumeric(argv[1])) {
    //result=winSock.getservbyport(winSock.htons((u_short)atoi(argv[1])),"tcp");
    //if (result) Tcl_AppendResult(interp,result->s_name,NULL);
    Tcl_AppendResult(interp,argv[1],NULL);
  } else {
    result=winSock.getservbyname(argv[1],"tcp");
    if (result){
      sprintf(buff,"%d",winSock.ntohs(result->s_port));
      Tcl_AppendResult(interp,buff,NULL);
    } else {
      Tcl_AppendResult(interp,"0",NULL);
    }
  }

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * InitSockets --
 *
 *  initializes WinSock when the DLL is first loaded by XiRC/Tclsh.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
BOOL InitSockets(void) {
 auto WORD wVersionRequired;
 auto OSVERSIONINFO info;

  /*
   * Find out if we're running on Win32s.
   */

  info.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  GetVersionEx(&info);

  /*
   * Check to see if Sockets are supported on this system.  Since
   * win32s panics if we call WSAStartup on a system that doesn't
   * have winsock.dll, we need to look for it on the system first.
   * If we find winsock, then load the library and initialize.
   */

  /*
  if (SearchPath(NULL, "WS2_32", ".DLL", 0, NULL, NULL) != 0) {

    winSock.hInstance = LoadLibrary("ws2_32.dll");
    wVersionRequired = MAKEWORD(2,2);

  } else
  */
  if ((info.dwPlatformId != VER_PLATFORM_WIN32s)
              || (SearchPath(NULL, "WINSOCK", ".DLL", 0, NULL, NULL) != 0)) {

    winSock.hInstance = LoadLibrary("wsock32.dll");
    wVersionRequired = MAKEWORD(1,1);

  } else {

    winSock.hInstance = NULL;

  };

  /*
   * Initialize the function table.
   */

  if (winSock.hInstance==NULL) return (FALSE);
    
  winSock.getservbyname   = (LPFN_GETSERVBYNAME) GetProcAddress(winSock.hInstance, "getservbyname");
  winSock.getservbyport   = (LPFN_GETSERVBYPORT) GetProcAddress(winSock.hInstance, "getservbyport");
  winSock.htonl           = (LPFN_HTONL) GetProcAddress(winSock.hInstance, "htonl");
  winSock.htons           = (LPFN_HTONS) GetProcAddress(winSock.hInstance, "htons");
  winSock.inet_addr       = (LPFN_INET_ADDR) GetProcAddress(winSock.hInstance, "inet_addr");
  winSock.inet_ntoa       = (LPFN_INET_NTOA) GetProcAddress(winSock.hInstance, "inet_ntoa");
  winSock.ntohl           = (LPFN_NTOHL) GetProcAddress(winSock.hInstance, "ntohl");
  winSock.ntohs           = (LPFN_NTOHS) GetProcAddress(winSock.hInstance, "ntohs");
  winSock.WSAStartup      = (LPFN_WSASTARTUP) GetProcAddress(winSock.hInstance, "WSAStartup");
  winSock.WSACleanup      = (LPFN_WSACLEANUP) GetProcAddress(winSock.hInstance, "WSACleanup");
  winSock.WSAGetLastError = (LPFN_WSAGETLASTERROR) GetProcAddress(winSock.hInstance, "WSAGetLastError");

  /*
   * Now check that all fields are properly initialized. If not, return
   * FALSE to indicate that we failed to initialize properly.
   */

  if ((winSock.getservbyname == NULL) ||
      (winSock.getservbyport == NULL) ||
      (winSock.htonl == NULL) ||
      (winSock.htons == NULL) ||
      (winSock.inet_addr == NULL) ||
      (winSock.inet_ntoa == NULL) ||
      (winSock.ntohl ==NULL) ||
      (winSock.ntohs == NULL) ||
      (winSock.WSAStartup == NULL) ||
      (winSock.WSACleanup == NULL) ||
      (winSock.WSAGetLastError == NULL)) {
    goto unloadLibrary;
  };
    
  /*
   * Initialize the winsock library and check the version number.
   */

  if (winSock.WSAStartup(wVersionRequired, &wsaData) != 0) {
    goto unloadLibrary;
  };

  if (wsaData.wVersion != wVersionRequired) {
    winSock.WSACleanup();
    goto unloadLibrary;
  };

  return (TRUE);

unloadLibrary:
  FreeLibrary(winSock.hInstance);
  winSock.hInstance = NULL;
  return (FALSE);
}
