/*************************************************************
 ** ini.cpp  --  part of the Falcon DLL Extension v1.5
 **
 ** Abstract:  This file contains the FALC_ini command.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux  mailto:davygrvy@bigfoot.com
 ************************************************************/

#include "falcon.h"
#include "util.h"

#define INI_MAX 32767

extern dyn_AppendElement Tcl_AppendElement;
extern dyn_AppendResult Tcl_AppendResult;

/*
 *----------------------------------------------------------------------
 *
 * CmdIni --
 *
 *  reads and writes ini files using the WinAPI's 
 *  GetPrivateProfileString() and WritePrivateProfileString()
 *
 * Results:
 *  the string if reading, or 0 when writing. a non-zero return for write 
 *  indicates an error.
 *
 * Side effects:
 *  none.
 *
 *----------------------------------------------------------------------
 */
extern int CmdIni(ClientData clientData,
                  Tcl_Interp *interp,int argc,char *argv[]) {
 auto char rcvbuff[INI_MAX]={'\0'};
 auto DWORD err;
 auto char* lpINIfile;
 auto LPVOID lpMsgBuf;

  if (argc < 2) {
    Tcl_AppendResult(interp,"Usage: ",argv[0]," <-write|-read> <parameters>",NULL);
    return TCL_ERROR;
  }

  if (!strnicmp(argv[1],"-r",2)) {
    if (argc>4) {
      auto int i=0;

      if(!(lpINIfile=Falc_DoFileName(argv[2]))) {
        sprintf(rcvbuff,"%s: HeapAlloc on line %d of file %s",argv[0],__LINE__,__FILE__);
        Tcl_AppendResult(interp,rcvbuff,NULL);
        return TCL_ERROR;
      };

      if(GetPrivateProfileString(strncmp(argv[3],"NULL",4)==0?NULL:argv[3],
          strncmp(argv[4],"NULL",4)==0?NULL:argv[4],argv[5]?argv[5]:"",
          rcvbuff,INI_MAX,lpINIfile)!=(INI_MAX-1)) {

        auto char* pos=rcvbuff;

        do {
          Tcl_AppendElement(interp,pos);
          while (*(pos)!='\0') pos++; pos++;
        } while (*(pos)!='\0');

        xfree(lpINIfile);
        return TCL_OK;

      } else {
        // a tiny error check for data length
        sprintf(rcvbuff,"%s: returned string is larger than %d characters!",argv[0],INI_MAX-1);

        Tcl_AppendResult(interp,rcvbuff,NULL);
        xfree(lpINIfile);
        return TCL_ERROR;
      }
    
    } else {
      Tcl_AppendResult(interp,"Usage: ",argv[0]," -read <inifile> <section> <key> [<default>]; returns data from an ini file",NULL);
      return TCL_ERROR;
    }
  }

  else if (!strnicmp(argv[1],"-w",2)) {  
    if (argc==6) {
      auto int i=0;
      auto BOOL rtn;
            
      if (strlen(argv[5])>(INI_MAX-1)){
        sprintf(rcvbuff,"%s: <string> is larger than %d characters!",argv[0],INI_MAX-1);
        Tcl_AppendResult(interp,rcvbuff,NULL);
        return TCL_ERROR;
      }

      if(!(lpINIfile=Falc_DoFileName(argv[2]))) {
        sprintf(rcvbuff,"%s: HeapAlloc on line %d of file %s",argv[0],__LINE__,__FILE__);
        Tcl_AppendResult(interp,rcvbuff,NULL);
        return TCL_ERROR;
      };

      if (strncmp(argv[4],"NULL",4) == 0 && strncmp(argv[5],"NULL",4)==0) {
        rtn = WritePrivateProfileSection(argv[3],"\0",lpINIfile);
      } else {
        rtn = WritePrivateProfileString(argv[3],strncmp(argv[4],"NULL",4)==0?NULL:argv[4],
            strncmp(argv[5],"NULL",4)==0?NULL:argv[5],lpINIfile);
      };

      if (rtn) {
        Tcl_AppendResult(interp,"0",NULL); //we return zero on success
        xfree(lpINIfile);
        return TCL_OK;
      } else {
        // if function fails report system error #
        xfree(lpINIfile);
        switch (err=GetLastError()) {
        case ERROR_FILE_NOT_FOUND:
          Tcl_AppendResult(interp,argv[0],": 64K limit reached. This is the Win32API ini file size limit.",NULL); 
          break;
        default :
          FormatMessage(
            FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, NULL,
            err, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
            (LPTSTR) &lpMsgBuf, 0, NULL );
          Tcl_AppendResult(interp,argv[0],": ",lpMsgBuf,NULL);
          // Free the buffer.
          LocalFree(lpMsgBuf);
        }
        return TCL_ERROR;  
      }
    } else {
      Tcl_AppendResult(interp,"Usage: ",argv[0]," -write <inifile> <section> <key> <string>; returns: 0 on success, system error msg on failure",NULL);
      return TCL_ERROR;
    }

  } else {
    Tcl_AppendResult(interp,"Usage: ",argv[0]," <-write|-read> <parameters>",NULL);
    return TCL_ERROR;
  }
}
