/*************************************************************
 ** DDE.cpp  --  part of the Falcon DLL Extension v1.50
 **
 ** Abstract: This file contains the FALC_dde command
 **           and related functions.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux  mailto:davygrvy@bigfoot.com
 **         The code in this file is based heavily from the tk8.1a1
 **         source code and dde11.c by Ken Corey. Both sources
 **         (c)1996-98 Sun MicroSystems Labs.
 ************************************************************/

#include "falcon.h"
#include "util.h"
#include "stddde.h"

extern dyn_AppendResult Tcl_AppendResult;
extern dyn_AppendElement Tcl_AppendElement;
//extern dyn_CreateCommand Tcl_CreateCommand;
extern dyn_ResetResult Tcl_ResetResult;
extern dyn_GlobalEval Tcl_GlobalEval;
extern BOOL InXiRCON;



DWORD ddeInstance = 0;	          // DDE Instance value for both Srv and client
#define SZ_SERVICENAME  "Falcon"  // service name
WORD MyFormats[] = {              // Format list (we only do ansi text)
    CF_TEXT,
    NULL
};

enum {
  DDE_ADVISE,
  DDE_EXECUTE,
  DDE_REQUEST,
  DDE_POKE,
  DDE_SERVICES
};

typedef struct {
  Tcl_Event header;
  int DDEType;
  char *DDEScript;
} DDEEvent;

/* Local Prototypes */
static void SetDdeError(Tcl_Interp *interp);
static DDEREQUESTFN SysHelpRequest;
static DDEREQUESTFN SysStatusRequest;
//static DDEREQUESTFN SysRtnMsgRequest;
static HDDEDATA MakeCFText(UINT wFmt, LPSTR lpszStr, HSZ hszItem);
static Tcl_EventProc DDEEventProc;
static FNCALLBACK DDECustomCallback;

/*
 *--------------------------------------------------------------
 *
 * StartDDEServer --
 *
 *  This procedure starts the interp's DDE server.
 *
 * Results:
 *  a unique name, if the original was taken.
 *
 * Side effects:
 *  unknown.
 *
 *--------------------------------------------------------------
 */
extern BOOL StartDDEServer (void) {
    
  // Do the basic initialization
  InitializeDDE(SZ_SERVICENAME, &ddeInstance, DDECustomCallback, 0);

  // Add each topic/item pair we support
  AddDDEItem("Action", "Script", MyFormats, NULL, NULL);

  // Add each system/item pair we support
  AddDDEItem(SZDDESYS_TOPIC, SZDDESYS_ITEM_HELP, MyFormats, SysHelpRequest, NULL);


  AddDDEItem(SZDDESYS_TOPIC, SZDDESYS_ITEM_STATUS, MyFormats, SysStatusRequest, NULL);
  //AddDDEItem(SZDDESYS_TOPIC, SZDDESYS_ITEM_RTNMSG, MyFormats, SysRtnMsgRequest, NULL);
  
  return TRUE;
}

//
// Return a string in CF_TEXT format
//
static HDDEDATA MakeCFText(UINT wFmt, LPSTR lpszStr, HSZ hszItem){
  if (wFmt != CF_TEXT) return NULL;
  return DdeCreateDataHandle(ddeInstance, (UCHAR *) lpszStr, 
                 strlen(lpszStr)+1, 0, hszItem, CF_TEXT, NULL);
}

//
// Return the Help info
//
static HDDEDATA SysHelpRequest(UINT wFmt, HSZ hszTopic, HSZ hszItem){
  static char *sz = "Please see the documentation for falcon.dll .";
  return MakeCFText(wFmt, sz, hszItem);
}

static HDDEDATA SysStatusRequest(UINT wFmt, HSZ hszTopic, HSZ hszItem){
  static char *sz = SZ_READY;
  return MakeCFText(wFmt, sz, hszItem);
}

static HDDEDATA SysRtnMsgRequest(UINT wFmt, HSZ hszTopic, HSZ hszItem){
  static char *sz = "";
  return MakeCFText(wFmt, sz, hszItem);
}

/*
 *--------------------------------------------------------------
 *
 * CmdDdeServ --
 *
 *  This procedure is invoked to process the FALC_ddeserv command.
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */
int
CmdDdeServ( ClientData clientData, 
       Tcl_Interp *interp, int argc,  char *argv[]) {

  // add loads of server type commands here.

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DDEEventProc --
 *
 *  This is the procedure that's called when a DDE event has been
 *  registered with Falc_QueueEvent() and the notifier now has 
 *  the interpreter and its time to handle the event.
 *
 * Results:
 *  unknown.
 *
 * Side effects:
 *  unknown.
 *
 *--------------------------------------------------------------
 */
static int DDEEventProc (Tcl_Interp *interp, Tcl_Event *evPtr, int flags) {
 DDEEvent *ddeevent;
 //int TCL_rtn;

  // we only handle DDE events here.
  if (!(flags & FALC_DDE_EVENTS)) return 0;

  ddeevent = (DDEEvent *)evPtr;
  switch (ddeevent->DDEType) {

  case DDE_POKE:     // poke doesn't return a value
    if((Tcl_GlobalEval(interp,ddeevent->DDEScript) != TCL_OK) && (Tcl_ResetResult)){
      Tcl_ResetResult(interp);
    }
    xfree(ddeevent->DDEScript);
    break;

  case DDE_REQUEST:  // request does returns a value
    if((Tcl_GlobalEval(interp,ddeevent->DDEScript) != TCL_OK)  && (Tcl_ResetResult)){
      Tcl_ResetResult(interp);
    }
    xfree(ddeevent->DDEScript);
    break;

  }
  
  
  return 1;
}

/*
 *--------------------------------------------------------------
 *
 * DDECustomCallback --
 *
 *  Sets the interp result to a cogent error message
 *  describing the last DDE error.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  The interp's result object is changed.
 *
 *--------------------------------------------------------------
 */
static 
HDDEDATA EXPENTRY DDECustomCallback (
    UINT uType,       /* The type of DDE transaction we are performing. */
    UINT uFmt,        /* The format that data is sent or received. */
    HCONV hConv,      /* The conversation associated with the current transaction. */
    HSZ ddeTopic,	    /* A string handle. Transaction-type dependent. */
    HSZ ddeItem,      /* A string handle. Transaction-type dependent. */
    HDDEDATA ddeData, /* DDE data. Transaction-type dependent. */
    DWORD dwData1,    /* Transaction-dependent data. */
    DWORD dwData2)    /* Transaction-dependent data. */
{
 PDDETOPICINFO pTopic;
 PDDEITEMINFO pItem;
 DDEEvent *ddeevent;
 LPBYTE dataReturnString;
 DWORD length;

  if (!(pTopic = FindTopicFromHsz(ddeTopic))) {
    return (HDDEDATA) DDE_FNOTPROCESSED;  //it ain't in the topic's list, so ditch
  }
  if (!(pItem = FindItemFromHsz(pTopic, ddeItem))) {
    return (HDDEDATA) DDE_FNOTPROCESSED;  //it ain't in the item list, so ditch
  }
  if (uFmt != CF_TEXT) {
    return (HDDEDATA) DDE_FNOTPROCESSED;  //it ain't CF_TEXT, so ditch
  }

  ddeevent = (DDEEvent *) xmalloc(sizeof(DDEEvent));
  ddeevent->header.proc = DDEEventProc;

  switch (uType) {
  case XTYP_POKE:
    ddeevent->DDEType = DDE_POKE;
    dataReturnString = DdeAccessData(ddeData, &length);
    ddeevent->DDEScript = (char *) xmalloc(length);
    memcpy(ddeevent->DDEScript,dataReturnString,length);
    Falc_QueueEvent((Tcl_Event *)ddeevent, FALC_QUEUE_TAIL);
    DdeUnaccessData(ddeData);
    break;

  case XTYP_REQUEST:
    ddeevent->DDEType = DDE_REQUEST;
    // createthread to add an event.
    // hold this thread in a loop waiting for the return from the event.
    // return the interp->result
    break;
  }

  return (HDDEDATA) DDE_FACK;
}

/*
 *--------------------------------------------------------------
 *
 * SetDdeError --
 *
 *  Sets the interp result to a cogent error message
 *  describing the last DDE error.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  The interp's result object is changed.
 *
 *--------------------------------------------------------------
 */
static void
SetDdeError(
    Tcl_Interp *interp)	    /* The interp to put the message in.*/
{
 UINT err=0;

  err = DdeGetLastError(ddeInstance);

  switch (err) {
  case DMLERR_UNADVACKTIMEOUT:
  case DMLERR_DATAACKTIMEOUT:
  case DMLERR_EXECACKTIMEOUT:
  case DMLERR_POKEACKTIMEOUT:
    Tcl_AppendResult(interp, "remote DDE server did not respond.", NULL);
    break;

  case DMLERR_BUSY:
    Tcl_AppendResult(interp, "remote DDE server is busy", NULL);
    break;

  case DMLERR_NOTPROCESSED:
    Tcl_AppendResult(interp, "remote server cannot handle this command.", NULL);
    break;

  case DMLERR_INVALIDPARAMETER:
    Tcl_AppendResult(interp, "invalid parameter(s) used within Falcon.dll (email author).", NULL);
    break;

  case DMLERR_LOW_MEMORY:
    Tcl_AppendResult(interp, "internal race condition aborted (server application outrunning the client).", NULL);
    break;

  case DMLERR_MEMORY_ERROR:
    Tcl_AppendResult(interp, "internal memory allocation error with DDEML.", NULL);
    break;

  case DMLERR_SERVER_DIED:
    Tcl_AppendResult(interp, "the server terminated before completing the transaction.", NULL);
    break;

  case DMLERR_UNFOUND_QUEUE_ID:
    Tcl_AppendResult(interp, "An invalid transaction identifier was passed to a DDEML function. Once the application had returned from an XTYP_XACT_COMPLETE callback, the transaction identifier for that callback function was no longer valid.", NULL);
    break;

  case DMLERR_REENTRANCY:
    Tcl_AppendResult(interp, "An application instance with a synchronous transaction already in progress attempted to initiate another synchronous transaction, or the DdeEnableCallback function was called from within a DDEML callback function.", NULL);
    break;

  default:
    Tcl_AppendResult(interp, "DDE command failed.", NULL);
  }
}

/*
 *--------------------------------------------------------------
 *
 * CmdDde --
 *
 *  This procedure is invoked to process the FALC_dde command.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *--------------------------------------------------------------
 */
int
CmdDde( ClientData clientData, 
       Tcl_Interp *interp, int argc,  char *argv[]) {
 int index /* , argIndex */;
 BOOL async = FALSE;
 int result = TCL_OK;
 HSZ ddeService = NULL;
 HSZ ddeTopic = NULL;
 HSZ ddeItem = NULL;
 HDDEDATA ddeData = NULL;
 HCONV hConv;
 char *serviceName, *topicName, *itemString, *dataString;
 int length, j;
 LPBYTE dataReturnString = NULL;
 DWORD ddeResult;
 HDDEDATA ddeReturn;
 BOOL useRaw = FALSE;

  serviceName = NULL;
  topicName = NULL;
  itemString = NULL;
  dataString = NULL;

  /* Decode args */
  if (argc >= 5) {
    for (j=1;argv[j][0] == '-';j++) {
      switch (argv[j][1]) {
      case 'a':    // async mode
        async = TRUE; break;
      case 'r':    // no Tab translation
        useRaw = TRUE; break;
      case 't':
        // set timeout value
      default:
        goto Usage;
      }
    }

    switch(argv[j][0]) {
    case 'a':
      //index=DDE_ADVISE; break;
      goto Usage;
    case 'e':
      index=DDE_EXECUTE; break;
    case 'p':
      index=DDE_POKE; break;
    case 'r':
      index=DDE_REQUEST; break;
    case 's':
      index=DDE_SERVICES; break;
    default:
      goto Usage;
    }
    serviceName = argv[++j];
    topicName   = argv[++j];
    itemString  = argv[++j];
    if (argc > (j+1)) dataString = argv[++j];

  } else {

Usage:
    Tcl_AppendResult(interp, "Usage: ", argv[0], " [-async] [-raw] <command> <service> <topic> <item> [<data>]", NULL);
    return TCL_ERROR;

  }


  if (serviceName) 
    ddeService = DdeCreateStringHandle(ddeInstance, serviceName, CP_WINANSI);
  if (topicName) 
    ddeTopic = DdeCreateStringHandle(ddeInstance, topicName, CP_WINANSI);


  switch (index) {
    case DDE_EXECUTE: {

      if (!ddeService || !ddeTopic) goto Usage;

      hConv = DdeConnect(ddeInstance, ddeService, ddeTopic, NULL);

      if (hConv == NULL) {
        SetDdeError(interp);
        result = TCL_ERROR;
        break;
      }

      if (ddeData = DdeCreateDataHandle(ddeInstance, (LPBYTE) itemString, 
          strlen(itemString)+1, 0, 0, CF_TEXT, 0)) {
        if (async) {
          DdeClientTransaction((LPBYTE) ddeData, 0xFFFFFFFF, hConv, 0, CF_TEXT, 
                XTYP_EXECUTE, TIMEOUT_ASYNC, &ddeResult);
          DdeAbandonTransaction(ddeInstance, hConv, ddeResult);
        } else {
          ddeReturn = DdeClientTransaction((LPBYTE) ddeData, 0xFFFFFFFF, hConv, 0, CF_TEXT, 
                XTYP_EXECUTE, 1000, NULL);
          if (ddeReturn == 0) {
            SetDdeError(interp);
            result = TCL_ERROR;
          }
        }
        DdeFreeDataHandle(ddeData);
      } else {
        SetDdeError(interp);
        result = TCL_ERROR;
      }
      DdeDisconnect(hConv);
      break;
    }
    
    case DDE_REQUEST: {

      if (!ddeService || !ddeTopic) goto Usage;

      hConv = DdeConnect(ddeInstance, ddeService, ddeTopic, NULL);

      if (hConv == NULL) {
        SetDdeError(interp);
        result = TCL_ERROR;
      } else {
        
        /* DDEML cannot create a handle to an empty string */
        if (*itemString == '\0') {
          itemString = " ";
        } 
        
        if (!(ddeItem = DdeCreateStringHandle(ddeInstance, itemString, CP_WINANSI))) {
          SetDdeError(interp);
          result = TCL_ERROR;
        } else if (!(ddeData = DdeClientTransaction(NULL, 0, hConv, ddeItem, 
              CF_TEXT, XTYP_REQUEST, 2000, NULL))) {
          SetDdeError(interp);
          result = TCL_ERROR;
        } else {
          dataReturnString = DdeAccessData(ddeData, NULL);

          if (!useRaw) {
            // Tab translation
            int i,j,pos=0,flag=0;
            do {
              for(i=pos,j=0;(*(dataReturnString+i)!='\0' && flag==0);i++,j++) {
                if(*(dataReturnString+i)=='\t') {
                  *(dataReturnString+i)='\0';
                  flag=1;
                }
              }
              Tcl_AppendElement(interp, (char *)(dataReturnString+pos));
              pos+=j; flag=0;
            } while (*(dataReturnString+pos)!='\0');
          } else {
            // No Tab translation
            if(dataReturnString)
              Tcl_AppendElement(interp, (char *)dataReturnString);
          }

          DdeUnaccessData(ddeData);
          DdeFreeDataHandle(ddeData);
        }

        DdeDisconnect(hConv);

      }
      break;
    }

    case DDE_POKE: {

      if (!ddeService || !ddeTopic) goto Usage;

      hConv = DdeConnect(ddeInstance, ddeService, ddeTopic, NULL);

      if (hConv == NULL) {
        SetDdeError(interp);
        result = TCL_ERROR;
      } else {
        
        /* DDEML cannot create a handle to an empty string */
        if (*itemString == '\0') {
          itemString = " ";
        }

        if (!(ddeItem = DdeCreateStringHandle(ddeInstance, itemString, CP_WINANSI))) {
          SetDdeError(interp);
          result = TCL_ERROR;
        } else if (!(ddeData = DdeCreateDataHandle(ddeInstance, (LPBYTE) dataString, 
              (strlen(dataString) + 1), 0, ddeItem, CF_TEXT, 0))) {
          SetDdeError(interp);
          result = TCL_ERROR;
        } else {
          if (async) {
            DdeClientTransaction( (LPBYTE) ddeData, 0xFFFFFFFF, hConv, 
                  ddeItem, CF_TEXT, XTYP_POKE, TIMEOUT_ASYNC, &ddeResult);
            DdeAbandonTransaction(ddeInstance, hConv, ddeResult);
          } else {
            ddeReturn = DdeClientTransaction( (LPBYTE) ddeData, 0xFFFFFFFF, hConv, 
                  ddeItem, CF_TEXT, XTYP_POKE, 1000, NULL);
            if (ddeReturn == 0) {
              SetDdeError(interp);
              result = TCL_ERROR;
            }
          }
          DdeFreeDataHandle(ddeData);
        }

        DdeDisconnect(hConv);

      }
      break;
    }
    
    case DDE_SERVICES: {
      HCONVLIST hConvList;
      CONVINFO convInfo;
      char *name[2],*out;

      convInfo.cb = sizeof(CONVINFO);
      hConvList = DdeConnectList(ddeInstance, ddeService, ddeTopic, 0, NULL);
      hConv = 0;
      name[0] = (char *)xmalloc(1);
      name[1] = (char *)xmalloc(1);

      while (hConv = DdeQueryNextServer(hConvList, hConv), hConv != 0) {
        DdeQueryConvInfo(hConv, QID_SYNC, &convInfo);
        length = DdeQueryString(ddeInstance, convInfo.hszSvcPartner, NULL, 0, CP_WINANSI);
        name[0] = (char *)xrealloc(name[0],length+1);
        DdeQueryString(ddeInstance, convInfo.hszSvcPartner, name[0], length+1, CP_WINANSI);
        
        length = DdeQueryString(ddeInstance, convInfo.hszTopic, NULL, 0, CP_WINANSI);
        name[1] = (char *)xrealloc(name[1],length+1);
        DdeQueryString(ddeInstance, convInfo.hszTopic, name[1], length+1, CP_WINANSI);

        out = Falc_Merge(2,name);
        Tcl_AppendElement(interp,out);
        xfree(out);
      }

      DdeDisconnectList(hConvList);
      xfree(name[0]);
      xfree(name[1]);
      result = TCL_OK;
      break;
    }

    case DDE_ADVISE:

      // add advise code here
      break;

  }

  if (ddeService != NULL) {
    DdeFreeStringHandle(ddeInstance, ddeService);
  }

  if (ddeTopic != NULL) {
    DdeFreeStringHandle(ddeInstance, ddeTopic);
  }

  if (ddeItem != NULL) {
    DdeFreeStringHandle(ddeInstance, ddeItem);
  }

  return result;
}

