/*************************************************************
 ** Config.cpp  --  part of the Falcon DLL Extension v1.5
 **
 ** Abstract: This file contains the FALC_config command
 **           and related functions.
 **
 ** Copyrights: See 'main.cpp'
 **
 ** Author: David Gravereaux  mailto:davygrvy@bigfoot.com
 **         The code in this file is based heavily on 
 **         tclWinReg.c from the tcl8.1a2 source code 
 **         (c)1996-98 Sun MicroSystems Labs.
 ************************************************************/

#include "falcon.h"

#include <winreg.h>

extern dyn_AppendResult Tcl_AppendResult;
extern dyn_AppendElement Tcl_AppendElement;
extern dyn_CreateCommand Tcl_CreateCommand;
//extern dyn_Eval Tcl_Eval;
//extern dyn_GetVar Tcl_GetVar;
//extern dyn_GlobalEval Tcl_GlobalEval;
extern dyn_ResetResult Tcl_ResetResult;
extern BOOL InXiRCON;

/* Local prototypes */
static int GetType(Tcl_Interp *interp, char *keyName, char *valueName);
static int GetValue(Tcl_Interp *interp, char *keyName, char *valueName);
static int OpenKey(Tcl_Interp *interp, char *keyName, REGSAM mode, int flags, HKEY *keyPtr);
static DWORD OpenSubKey(char *hostName, HKEY rootKey, char *keyName, REGSAM mode, int flags, HKEY *keyPtr);
static int ParseKeyName(Tcl_Interp *interp, char *name, char **hostNamePtr, HKEY *rootKeyPtr, char **keyNamePtr);
static void AppendSystemError(Tcl_Interp *interp, DWORD error);
static DWORD ConvertDWORD(DWORD type, DWORD value);


/*
 * The following macros convert between different endian ints.
 */
#define SWAPWORD(x) MAKEWORD(HIBYTE(x), LOBYTE(x))
#define SWAPLONG(x) MAKELONG(SWAPWORD(HIWORD(x)), SWAPWORD(LOWORD(x)))

/*
 * The following flag is used in OpenKeys to indicate that the specified
 * key should be created if it doesn't currently exist.
 */
#define REG_CREATE 1

/*
 * The following tables contain the mapping from registry root names
 * to the system predefined keys.
 */
static char *rootKeyNames[] = {
    "HKEY_LOCAL_MACHINE", "HKEY_USERS", "HKEY_CLASSES_ROOT",
    "HKEY_CURRENT_USER", "HKEY_CURRENT_CONFIG", NULL
};
static HKEY rootKeys[] = {
    HKEY_LOCAL_MACHINE, HKEY_USERS, HKEY_CLASSES_ROOT, HKEY_CURRENT_USER,
    HKEY_CURRENT_CONFIG, HKEY_PERFORMANCE_DATA, HKEY_DYN_DATA
};

/*
 * The following table maps from registry types to strings.  Note that
 * the indices for this array are the same as the constants for the
 * known registry types so we don't need a separate table to hold the
 * mapping.
 */
static char *typeNames[] = {
    "none", "sz", "expand_sz", "binary", "dword", 
    "dword_big_endian", "link", "multi_sz", "resource_list", NULL
};
static DWORD lastType = REG_RESOURCE_REQUIREMENTS_LIST;


/*
 *----------------------------------------------------------------------
 *
 * GetType --
 *
 *	This function gets the type of a given registry value and
 *	places it in the interpreter result.
 *
 * Results:
 *	Returns a normal Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
GetType(
  Tcl_Interp *interp,		/* Current interpreter. */
  char *keyName,        /* Name of key. */
  char *valueName)      /* Name of value to get. */
{
 HKEY key;
 DWORD result;
 DWORD type;
    
  /*
   * Attempt to open the key for reading.
   */
  if (OpenKey(interp, keyName, KEY_QUERY_VALUE, 0, &key) != TCL_OK) {
    return TCL_ERROR;
  }

  /*
   * Get the type of the value.
   */

  result = RegQueryValueEx(key, valueName, NULL, &type, NULL, NULL);
  RegCloseKey(key);

  if (result != ERROR_SUCCESS) {
    Tcl_AppendResult(interp, "unable to get type of value \"",
        valueName, "\" from key \"",
        keyName, "\": ", NULL);
    AppendSystemError(interp, result);
    return TCL_ERROR;
  }

  /*
   * Set the type into the result.  Watch out for unknown types.
   * If we don't know about the type, just use the numeric value.
   */
  if (type > lastType) {
    //Tcl_SetIntObj(interp, type);
  } else {
    Tcl_AppendResult(interp, typeNames[type], NULL);
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetValue --
 *
 *	This function gets the contents of a registry value and places
 *	a list containing the data and the type in the interpreter
 *	result.
 *
 * Results:
 *	Returns a normal Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static int GetValue( Tcl_Interp *interp, char *keyName, char *valueName) {
 HKEY key;
 DWORD result, length, type;
 LPBYTE data;

  /*
   * Attempt to open the key for reading.
   */

  if (OpenKey(interp, keyName, KEY_QUERY_VALUE, 0, &key) != TCL_OK) {
    return TCL_ERROR;
  }

  /*
   * Get the value once to determine the length then again to store
   * the data in the buffer.
   */

  //Tcl_DStringInit(&data);
  //resultPtr = Tcl_GetObjResult(interp);

  //valueName = Tcl_GetStringFromObj(valueNameObj, (int*) &length);
  result = RegQueryValueEx(key, valueName, NULL, &type, NULL, &length);
  if (result == ERROR_SUCCESS) {
    data = (LPBYTE) xmalloc(length);
    result = RegQueryValueEx(key, valueName, NULL, &type, data, &length);
  }
  RegCloseKey(key);
  if (result != ERROR_SUCCESS) {
    Tcl_AppendResult(interp, "unable to get value \"",
        valueName, "\" from key \"",
        keyName, "\": ", NULL);
    AppendSystemError(interp, result);
    xfree(data);
    return TCL_ERROR;
  }

  /*
   * If the data is a 32-bit quantity, store it as an integer object.  If it
   * is a multi-string, store it as a list of strings.  For null-terminated
   * strings, append up the to first null.  Otherwise, store it as a binary
   * string.
   */

  if (type == REG_DWORD || type == REG_DWORD_BIG_ENDIAN) {
    char buff[33];
    _ultoa(ConvertDWORD(type,*data),buff,10);
    Tcl_AppendResult(interp,buff,NULL);

  } else if (type == REG_MULTI_SZ) {

    char *p = (char *)data;
    char *lastChar = (char *)data + length;

    /*
     * Multistrings are stored as an array of null-terminated strings,
     * terminated by two null characters.  Also do a bounds check in
     * case we get bogus data.
     */

    while (p < lastChar && *p != '\0') {
      Tcl_AppendElement(interp, p);
      while (*p++ != '\0') {}
    }
  } else if ((type == REG_SZ) || (type == REG_EXPAND_SZ)) {
    Tcl_AppendResult(interp, (char *)data, NULL);
  } else {
    Tcl_AppendResult(interp, (char *)data, NULL);
  }
  
  return result;
}

/*
 *----------------------------------------------------------------------
 *
 * OpenKey --
 *
 *	This function opens the specified key.  This function is a
 *	simple wrapper around ParseKeyName and OpenSubKey.
 *
 * Results:
 *	Returns the opened key in the keyPtr argument and a Tcl
 *	result code.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static int
OpenKey(
  Tcl_Interp *interp,   /* Current interpreter. */
  char *keyName,        /* Key to open. */
  REGSAM mode,          /* Access mode. */
  int flags,            /* 0 or REG_CREATE. */
  HKEY *keyPtr)         /* Returned HKEY. */
{
 char *hostName;
 //int length;
 HKEY rootKey;
 DWORD result;

  result = ParseKeyName(interp, keyName, &hostName, &rootKey, &keyName);
  if (result == TCL_OK) {
    result = OpenSubKey(hostName, rootKey, keyName, mode, flags, keyPtr);
    if (result != ERROR_SUCCESS) {
      Tcl_AppendResult(interp, "unable to open key: ", NULL);
      AppendSystemError(interp, result);
      result = TCL_ERROR;
    } else {
      result = TCL_OK;
    }
  }

  return result;
}

/*
 *----------------------------------------------------------------------
 *
 * OpenSubKey --
 *
 *	This function opens a given subkey of a root key on the
 *	specified host.
 *
 * Results:
 *	Returns the opened key in the keyPtr and a Windows error code
 *	as the return value.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static DWORD
OpenSubKey(
  char *hostName, /* Host to access, or NULL for local. */
  HKEY rootKey,   /* Root registry key. */
  char *keyName,  /* Subkey name. */
  REGSAM mode,    /* Access mode. */
  int flags,      /* 0 or REG_CREATE. */
  HKEY *keyPtr)   /* Returned HKEY. */
{
 DWORD result;

  /*
   * Attempt to open the root key on a remote host if necessary.
   */
  if (hostName) {
    result = RegConnectRegistry(hostName, rootKey, &rootKey);
    if (result != ERROR_SUCCESS) {
      return result;
    }
  }

  /*
   * Now open the specified key with the requested permissions.  Note
   * that this key must be closed by the caller.
   */
  if (flags & REG_CREATE) {
    DWORD create;
    result = RegCreateKeyEx(rootKey, keyName, 0, "",
        REG_OPTION_NON_VOLATILE, mode, NULL, keyPtr, &create);
  } else {
    result = RegOpenKeyEx(rootKey, keyName, 0, mode, keyPtr);
  }

  /*
   * Be sure to close the root key since we are done with it now.
   */
  if (hostName) {
    RegCloseKey(rootKey);
  }
  return result; 
}

/*
 *----------------------------------------------------------------------
 *
 * ParseKeyName --
 *
 *	This function parses a key name into the host, root, and subkey
 *	parts. 
 *
 * Results:
 *	The pointers to the start of the host and subkey names are
 *	returned in the hostNamePtr and keyNamePtr variables.  The
 *	specified root HKEY is returned in rootKeyPtr.  Returns
 *	a standard Tcl result.
 *
 *
 * Side effects:
 *	Modifies the name string by inserting nulls.
 *
 *----------------------------------------------------------------------
 */
static int
ParseKeyName(
  Tcl_Interp *interp,		/* Current interpreter. */
  char *name,
  char **hostNamePtr,
  HKEY *rootKeyPtr,
  char **keyNamePtr)
{
 char *rootName;
 int result, index;
 char *rootObj;

  /*
   * Split the key into host and root portions.
   */
  *hostNamePtr = *keyNamePtr = rootName = NULL;
  if (name[0] == '\\') {
    if (name[1] == '\\') {
      *hostNamePtr = name;
      for (rootName = name+2; *rootName != '\0'; rootName++) {
        if (*rootName == '\\') {
          *rootName++ = '\0';
          break;
        }
      }
    }
  } else {
    rootName = name;
  }
  if (!rootName) {
    Tcl_AppendResult(interp, "bad key \"", name,
        "\": must start with a valid root", NULL);
    return TCL_ERROR;
  }

  /*
   * Split the root into root and subkey portions.
   */

  for (*keyNamePtr = rootName; **keyNamePtr != '\0'; (*keyNamePtr)++) {
    if (**keyNamePtr == '\\') {
      **keyNamePtr = '\0';
      (*keyNamePtr)++;
      break;
    }
  }

  /*
   * Look for a matching root name.
   */
  //rootObj = Tcl_NewStringObj(rootName, -1);
  //result = Tcl_GetIndexFromObj(interp, rootObj, rootKeyNames, "root name",
  //    TCL_EXACT, &index);
  //Tcl_DecrRefCount(rootObj);
  //if (result != TCL_OK) {
  //  return TCL_ERROR;
  //}
  *rootKeyPtr = rootKeys[index];
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AppendSystemError --
 *
 *	This routine formats a Windows system error message and places
 *	it into the interpreter result.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
AppendSystemError(
  Tcl_Interp *interp, /* Current interpreter. */
  DWORD error)        /* Result code from error. */
{
 int length;
 char *msg;

  length = FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM
      | FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, error,
      MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (char *) msg,
      0, NULL);

  if (length == 0) {
    if (error == ERROR_CALL_NOT_IMPLEMENTED) {
      msg = "function not supported under Win32s";
    } else {
      sprintf(msg, "unknown error: %d", error);
    }
  } else {

    /*
     * Trim the trailing CR/LF from the system message.
     */
    if (msg[length-1] == '\n') {
      msg[--length] = '\0';
    }
    if (msg[length-1] == '\r') {
      msg[--length] = '\0';
    }
  }

  Tcl_AppendResult(interp, msg, NULL);
  LocalFree(msg);
}

/*
 *----------------------------------------------------------------------
 *
 * ConvertDWORD --
 *
 *	This function determines whether a DWORD needs to be byte
 *	swapped, and returns the appropriately swapped value.
 *
 * Results:
 *	Returns a converted DWORD.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static DWORD
ConvertDWORD(
  DWORD type,   /* Either REG_DWORD or REG_DWORD_BIG_ENDIAN */
  DWORD value)  /* The value to be converted. */
{
 DWORD order = 1;
 DWORD localType;

  /*
   * Check to see if the low bit is in the first byte.
   */
  localType = (*((char*)(&order)) == 1) ? REG_DWORD : REG_DWORD_BIG_ENDIAN;
  return (type != localType) ? SWAPLONG(value) : value;
}

