#include "CServer.h"

CServer::CServer (CBot *b)
{
  // paranoid init :>
  bot = b;
  hosts = host_current = NULL;
  *virtualhost = *nick = *nick_orig = *user = *name = 0;
  *away = *quit = 0;
  connected = 0;
  last_try = 0;
  change_time = 0;
  time_now = bot->time_now + change_time;
  cmd_i = 0;
  users = NULL;
  vars = new CVar (this);
  services = new CServices (this);
  script = new CScript (this);
  int i;
  for(i = 0; i < CHANNEL_MAX; i++)
    channels[i] = NULL;
  channel_num = 0;
  for(i = 0; i < DCC_MAX; i++)
    dccs[i] = NULL;
  dcc_num = 0;
  dcc_port = 0;
  *dcc_file = *dcc_motdfile = *bufread = *bufwrite = 0;
  dcc_motd = NULL;
  bufpos = 0;
  bytesin = bytesout = 0;
  time_read = 0;
  uptime = 0;
}

CServer::~CServer (void)
{
  irc_restart ();				// turn off networking
  delete users;
  delete vars;
  delete services;
  delete script;
  if (dcc_motd != NULL)
    delete dcc_motd;
}

void
CServer::irc_pong (const char *pong)
{
  snprintf (bufwrite, MSG_SIZE, "PONG :%s", pong);
  irc_write (bufwrite);
}

// if msg is \0, use the default
void
CServer::irc_quit (const char *msg)
{
  char quitcmd[] = "QUIT :";
  if (*msg == 0) 
    snprintf (bufwrite, MSG_SIZE, "%s%s", quitcmd, quit);
  else 
    snprintf (bufwrite, MSG_SIZE, "%s%s", quitcmd, msg);
  irc_write (bufwrite);
}

void
CServer::irc_privmsg (const char *to, const char *msg)
{
  snprintf (bufwrite, MSG_SIZE, "PRIVMSG %s :%s", to, msg);
  irc_write (bufwrite);
}

void
CServer::irc_notice (const char *to, const char *msg)
{
  snprintf (bufwrite, MSG_SIZE, "NOTICE %s :%s", to, msg);
  irc_write (bufwrite);
}

void
CServer::irc_nick (const char *new_nick)
{
  snprintf (bufwrite, MSG_SIZE, "NICK %s", new_nick);
  irc_write (bufwrite);
}

void
CServer::irc_user (const char *new_user, const char *new_name)
{
  snprintf (bufwrite, MSG_SIZE, "USER %s 0 %s :%s", new_user,
            host_current->host, new_name);
  irc_write (bufwrite);
}

void
CServer::irc_whois (const char *what_nick)
{
  snprintf (bufwrite, MSG_SIZE, "WHOIS %s", what_nick);
  irc_write (bufwrite);
}

void
CServer::irc_ison (const char *nick_list)
{
  snprintf (bufwrite, MSG_SIZE, "ISON %s", nick_list);
  irc_write (bufwrite);
}

void
CServer::irc_who (const char *what)
{
  snprintf (bufwrite, MSG_SIZE, "WHO %s", what);
  irc_write (bufwrite);
}

void
CServer::irc_away (const char *reason)
{
  snprintf (bufwrite, MSG_SIZE, "AWAY :%s", reason);
  irc_write (bufwrite);
}

void
CServer::irc_oper (const char *what_nick, const char *pass)
{
  snprintf (bufwrite, MSG_SIZE, "OPER %s %s", what_nick, pass);
  irc_write (bufwrite);
}

void
CServer::irc_kill (const char *what_nick, const char *reason)
{
  snprintf (bufwrite, MSG_SIZE, "KILL %s :%s", what_nick, reason);
  irc_write (bufwrite);
}

void
CServer::write_botlog (const char *msg)
{
  char time_buf[20];
  string_time (bot->time_now, time_buf); // don't use this server's local time
  fprintf (stderr, "%s %s %s: %s\n", time_buf, nick, host_current->host, msg);
  fflush (stderr);
}

void
CServer::irc_write (const char *raw_msg)
{
  if (bot->debug)
    printf ("write: %s: %s\n", nick, raw_msg);
  bytesout += strlen (raw_msg) + 1; // +1 because a '\n' will be appended
  writesock (sock, raw_msg);
}

// read as many bytes as the socket has, until a \n or buf has max size
// return -1 on error, 0 to ignore, 1 if a new string is ready
int
CServer::irc_read (void)
{
  int i = readok (sock);
  while (i == 1)	// while there's something to read
    {
      i = read (sock, bufread + bufpos, 1);		// read it
      if (i == 1)					// success?
        {
          bytesin++;
          if (bufread[bufpos] == '\n' || bufpos == MSG_SIZE) // end msg
            {
              bufread[bufpos] = 0;
              strip_crlf (bufread);
              if (bot->debug)
                printf ("read: %s: %s\n", nick, bufread);
              bufpos = 0;
              time_read = time_now;
              return 1;
            }
          bufpos++;			// next char in buffer
        }
      else				// -1 on error, 0 when the server dies
        return -1;
    }
  return i;
}

void
CServer::irc_connect (void)
{
  if (host_current != NULL)
    {
      usleep (1);
      time_read = time_now;
      if (difftime (time_now, last_try) > TIME_RETRY)
        {
          last_try = time_now;
          write_botlog ("Connecting to server.");
          sock = openhost (virtualhost, host_current->host,
                           host_current->port);
          if (sock == -1)
            {
              snprintf (buf, MSG_SIZE, "ERROR connecting to server: %s",
                        strerror (errno));
              write_botlog (buf);
              host_current = host_current->next;
            }
          else
            {
              uptime = time_now;
              sock_linger (sock, 60 * 100); // wait up to one minute in close()
              irc_nick (nick);
              irc_user (user, name);
              connected = 1;
            }
        }
    }
  else
    {
      // don't use write_botlog() here because it would crash (host_current..)
      string_time (bot->time_now, buf);	// don't use this server's local time
      fprintf (stderr, "%s %s: No host defined, can't connect!\n", buf, nick);
      fflush (stderr);
      bot->irc_exit ();
    }
}

// close socket, dccs and delete users from channels
void
CServer::irc_restart (void)
{
  int i,i2;
  close (sock);
  connected = 0;
  my_strncpy (nick, nick_orig, NICK_SIZE);
  services->exist = (*services->nickserv_pass != 0);
  services->identified = 0;
  for (i = 0; i < channel_num; i++)
    {
      for (i2 = 0; i2 < channels[i]->user_num; i2++)
        delete channels[i]->users[i2];
      channels[i]->user_num = 0;
      channels[i]->joined = 0;
    }
  for (i = 0; i < dcc_num; i++)
    delete dccs[i];
  dcc_num = 0;
  *bufread = 0;
  bufpos = 0;
  bytesin = bytesout=0;
}

void
CServer::work (void)
{
  time_t time_last = time_now;
  time_now = bot->time_now + change_time;
  if (connected)
    {
      int i = irc_read ();
      if (i == 1)			// there's a new msg to parse
        {
          irc_parse ();
          script->irc_reply ();
          script->irc_event ();
        }
      else
        if (i == -1)			// error while reading
          {
            snprintf (buf, MSG_SIZE, "ERROR reading from socket: %s",
                      strerror (errno));
            write_botlog (buf);
            irc_restart ();
            host_current = host_current->next;
            last_try = time_now;    // wait TIME_RETRY before connecting
            return;
          }
        else
          if (difftime (time_now, time_read) > TIME_STONED)
            {
              write_botlog ("ERROR: server timed out, disconnecting");
              irc_restart ();
              host_current = host_current->next;
              last_try = time_now;    // wait TIME_RETRY before connecting
              return;
            }

      // big mess with dcc's, gotta clean it up sometime
      for(i = 0; i < dcc_num; i++)
        if (dccs[i] != NULL)
          if (dccs[i]->dcc_work () == 0)	// inactive dcc, delete it
            {
              int i2;
              for (i2 = 0; i2 < dcc_num; i2++)	// delete dependencies
                {
                  if (dccs[i2]->dcc_from_index == i)
                    dccs[i2]->dcc_from_index = -1;
                  if (dccs[i2]->dcc_from_index > i)
                    dccs[i2]->dcc_from_index--;
                }
              delete dccs[i];
              for (i2 = i; i2 < dcc_num; i2++)
                {
                  dccs[i2] = dccs[i2+1];
                  if (dccs[i2] != NULL)
                    dccs[i2]->index--;
                }
              dcc_num--;
            }

      if (time_last != time_now)
        {
          CScript::f_type *buf = script->timers;
          while (buf != NULL)
            {
              buf->f (this);
              buf = buf->next;
            }
        }
    }
  else
    irc_connect ();
}

// return 0 if ok, 1 if limit exceeded, 2 if it already exists
int
CServer::channel_add (const char *chan_name, const char *chan_key)
{
  if (channel_num < CHANNEL_MAX)			// limit
    {
      if (channel_index (chan_name) != -1)			// exists
        return 2;
      channels[channel_num] = new CChannel (chan_name, chan_key, this);
      channel_num++;
      return 0;
    }
  else
    return 1;
}

// return 0 if ok, 1 if nonexistent
int
CServer::channel_del (const char *chan_name)
{
  int i = channel_index (chan_name);
  if (i != -1)
    {
      delete channels[i];
      for (; i < channel_num; i++)
        channels[i] = channels[i+1];
      channel_num--;
      return 0;
    }
  else
    return 1;
}

// return channel's table index, -1 if nonexistent
int
CServer::channel_index (const char *chan_name)
{
  int i;
  for (i = 0; i < CHANNEL_MAX; i++)
    if (channels[i] != NULL)
      if (strcasecmp (chan_name, channels[i]->name) == 0)
        return i;
  return -1;
}

void
CServer::irc_parse (void)
{
  size_t i = 0, i2 = 0;
  cmd_i = 0;
  memset (cmd, 0, sizeof (cmd));

  if (bufread[0] == ':')
    my_strncpy (bufread, bufread + 1, MSG_SIZE);

  size_t buflen = strlen (bufread);

  if (buflen != 0)
    while (bufread[i] != ':' && i < buflen && cmd_i < CMD_SIZE)
      {
        while (bufread[i] != ' ' && bufread[i] != '\n' && i != buflen)
          i++;
        my_strncpy (cmd[cmd_i], bufread + i2, i - i2);
        cmd_i++;
        i2 = ++i;
      }

  if (bufread[i] == ':')
    {
      i++;
      strncpy (cmd[cmd_i], bufread + i, strlen (bufread + i));
    }

}

// add a server to h_list, return this bot's server list (recursive function)
CServer::host_type *
CServer::host_add (host_type *h_list, const char *host, int port)
{
  if (h_list == NULL)					// if it's the first
    {
      h_list = new host_type;
      my_strncpy (h_list->host, host, HOST_SIZE);
      h_list->port = port;
      h_list->next = h_list;
      host_current = h_list;
    }
  else if (h_list->next == host_current)		// if it's the last
    {
      h_list->next = new host_type;
      my_strncpy (h_list->next->host, host, HOST_SIZE);
      h_list->next->port = port;
      h_list->next->next = host_current;	// host_current has the first
    }
  else
    host_add (h_list->next, host, port);		// try next one
  return h_list;
}

