#include "CBot.h"

CBot::CBot (void)
{
  line = 0;
  debug = 0;
  for (int i = 0; i < SERVER_MAX; i++)
    servers[i] = NULL;
  server_num = 0;
  conf = NULL;
  my_strncpy (conf_file, CONF_FILE, FILE_SIZE);
  my_strncpy (pid_file, PID_FILE, FILE_SIZE);
  module_list = NULL;
  time_now = get_time ();
}

CBot::~CBot (void)
{
  if (conf != NULL)
    delete conf;
  remove (pid_file);
}

// write <msg> to botlog
void 
CBot::write_botlog (const char *msg)
{
  char time_buf[20];
  string_time (time_now, time_buf);
  fprintf (stderr, "%s %s\n", time_buf, msg);
  fflush (stderr);
}

// add a module to the bot's module list and load it, return its pointer
CModule *
CBot::add_module (const char *filename)
{
  CModule *m_temp = new CModule (this);
  if (m_temp->init (filename) == 0)
    {
      delete m_temp;
      return NULL;
    }
  module_type *buf = module_list;
  module_list = new CBot::module_type;
  module_list->m = m_temp;
  module_list->next = buf;
  return m_temp;
}

// delete a module from the bot's module list and unload it, return 1 if ok
bool
CBot::del_module (const char *name) {
  module_type *mt_temp;
  if (module_list != NULL)
    {
      if (module_list->m != NULL)		// check if it's the first
        if (module_list->m->module->name != NULL)
          if (strcmp (name, module_list->m->module->name) == 0)
            {
              delete module_list->m;
              mt_temp = module_list->next;
              delete module_list;
              module_list = mt_temp;
              return 1;
            }
      module_type *buf = module_list;
      while (buf->next != NULL)			// nope, try the others
        {
          if (buf->next->m != NULL)
            if (buf->next->m->module->name != NULL)
            if (strcmp (name, buf->next->m->module->name) == 0)
              {
                delete buf->next->m;
                mt_temp = buf->next->next;
                delete buf->next;
                buf->next = mt_temp;
                return 1;
              }
          buf = buf->next;
        }
    }
  return 0;
}

// command line parser
void
CBot::parse_cmd (int argc, char *argv[])
{
  int i;
  for (i = 1; i < argc; i++)
    while (1)
      {
        if ((strcmp (argv[i], "-c") == 0)
            || (strcmp (argv[i], "--conf") == 0))
          {
            if (argv[++i] == NULL)
              {
                printf (" ERROR: missing file for -c\n\n");
                exit (1);
              }
            my_strncpy (conf_file, argv[i], FILE_SIZE);
            break;
          }
        else if ((strcmp (argv[i], "-h") == 0)
                 || (strcmp (argv[i], "--help") == 0))
          {
            printf (" Options:\n");
            printf ("  -h,  --help               this help\n");
            printf ("  -c,  --conf <file>        load configuration file\n");
            printf ("  -d,  --debug              enter debug mode\n");
            printf ("  -q,  --quiet              quiet mode\n");
            printf ("  -v,  --version            show version and exit\n");
            printf ("\n Report bugs to <mirage@PTlink.net>\n");
            exit (1);
          }
        else if ((strcmp (argv[i], "-d") == 0)
                 || (strcmp (argv[i], "--debug") == 0))
          {
            debug = 1;
            break;
          }
        else if ((strcmp (argv[i], "-q") == 0)
                 || (strcmp (argv[i], "--quiet") == 0))
          {
            freopen ("/dev/null", "r", stdin);
            freopen ("/dev/null", "w", stdout);
            freopen ("/dev/null", "w", stderr);
            break;
          }
        else if ((strcmp (argv[i], "-v") == 0)
                 || (strcmp (argv[i], "--version") == 0))
          {
            printf ("    Copyright (C) 1998,1999  Tiago Sousa <mirage@PTlink.net>\n\n");
            exit (0);
          }
        fprintf (stderr, "ERROR: unknown option \"%s\", try --help for help.\n\n",
                 argv[i]);
        exit (0);
      }
}

// check if other bot is running
void
CBot::check_pid (void)
{
  int i;
  FILE *pfd;
  pfd = fopen (pid_file, "r");
  if (pfd != NULL)
    {
      char pidbuf[21];
      fgets (pidbuf, 20, pfd);
      fclose (pfd);
      i = atoi (pidbuf);
      kill (i, SIGCHLD);
      if (errno != ESRCH)
        {
          printf (" ERROR: Another copy of the bot is running\n\n");
          exit (1);
        }
    }
}

// save file with the pid
void
CBot::write_pid (void)
{
  FILE *pfd;
  pfd = fopen(pid_file, "w");
  if (pfd != NULL)
    {
      fprintf (pfd, "%lu\n", (u_long)getpid());
      fclose (pfd);
    }
  else
    {
      snprintf (buf, MSG_SIZE, "ERROR: Can't write in %s: %s", pid_file,
                strerror (errno));
      write_botlog (buf);
      exit (1);
    }
}
  
// show each server's info
void
CBot::server_info (void)
{
  CServer *s;
  for (int i = 0; i < server_num; i++)
    {
      s = servers[i];
      printf ("  %s:%d\n", s->host_current->host, s->host_current->port);
      printf ("  %s!%s (%s)\n", s->nick, s->user, s->name);
      printf ("  %d/%d users, %d/%d channels\n\n",
              (s->users == NULL?0:s->users->user_num), USER_MAX,
              s->channel_num, CHANNEL_MAX);
    }
}

// show the error and exit
void
CBot::conf_error (const char *error)
{
  char conf_buf[BUF_SIZE+1];
  snprintf (conf_buf, BUF_SIZE, "ERROR in configuration (line %d): %s", line,
            error);
  write_botlog (conf_buf);
  exit (1);
}

// show the warning
void
CBot::conf_warn (const char *warning)
{
  char conf_buf[BUF_SIZE+1];
  snprintf (conf_buf, BUF_SIZE, "Warning in configuration (line %d): %s", line,
            warning);
  write_botlog (conf_buf);
}

// check that basic parameters of a server are specified 
void
CBot::check_server (CServer *s)
{
  if (s != NULL)
    if (s->hosts == NULL)
      conf_error ("no hosts were defined, can't continue.");
}

// configuration line parser
void
CBot::parse_conf (void)
{
  int i = 0;
  const char *bufline;
  char buf[10][MSG_SIZE+1];
  server_num = -1;
  CServer *s = NULL;
  bool changed = 0, bot_defined = 0;

  conf = new CText ();
  if (!conf->read_file (conf_file))
    conf_error ("can't open configuration file.");
  conf->strip_text_crlf ();
  conf->rewind_text ();
  while ((bufline = conf->get_line ()) != NULL)
    {
      line++;
      if (bufline[0] != 0 && bufline[0] != '#')
        {

          strsplit (bufline, buf, 1);
          changed = 0;

          if (strcasecmp (buf[1], "pid") == 0)
            {
              strcpy (pid_file, strcut(buf[2], FILE_SIZE));
              changed = 1;
            }

          else if (strcasecmp (buf[1], "debug") == 0)
            debug = changed = 1;

          else if (strcasecmp (buf[1], "quiet") == 0)
            {
              if ((freopen ("/dev/null", "r", stdin) == NULL)
                  || (freopen ("/dev/null", "w", stdout) == NULL)
                  || (freopen ("/dev/null", "w", stderr) == NULL))
                conf_error ("can't redirect std to /dev/null");
              changed = 1;
            }

          else if (strcasecmp (buf[1], "module") == 0)
            {
              if (!add_module(buf[2]))
                conf_error ("can't load module.");
              changed = 1;
            }

          else if ((strcasecmp (buf[1], "server") == 0)
              || (strcasecmp (buf[1], "bot") == 0))
            {
              check_server (s);
              if (server_num + 1 < SERVER_MAX)
                {
                  servers[++server_num] = new CServer (this);
                  s = servers[server_num];
                }
              else
                conf_error ("maximum servers exceeded.");
              changed = bot_defined = 1;
            }

          else if (strcasecmp (buf[1], "botlog") == 0)
            {
              if (freopen (buf[2], "a", stderr) == NULL)
                conf_error ("can't open botlog's file for writing.");
              changed = 1;
            }

          if (! (changed || bot_defined))
            conf_error("bot configuration before a bot is initialized.");

          if (strcasecmp (buf[1], "host") == 0)
            {
              strsplit (bufline, buf, 3);
              s->hosts = s->host_add (s->hosts, strcut (buf[2], HOST_SIZE),
                                      atoi (buf[3]));
            }

          else if (strcasecmp (buf[1], "vhost") == 0)
            my_strncpy (s->virtualhost, buf[2], HOST_SIZE);

          else if (strcasecmp (buf[1], "nick") == 0)
            {
              my_strncpy (s->nick, buf[2], NICK_SIZE);
              my_strncpy (s->nick_orig, buf[2], NICK_SIZE);
            }

          else if (strcasecmp (buf[1], "user") == 0)
            my_strncpy (s->user, buf[2], USER_SIZE);

          else if (strcasecmp (buf[1], "name") == 0)
            my_strncpy (s->name, buf[2], NAME_SIZE);

          else if (strcasecmp (buf[1], "nickserv_pass") == 0)
            {
              if (!s->services->exist)
                {
                  my_strncpy (s->services->nickserv_pass, buf[2], NICK_PASS_SIZE);
                  s->services->exist = 1;
                }
              else
                conf_error ("nickserv already defined.");
            }
          else if (strcasecmp (buf[1], "nickserv_mask") == 0)
            {
              my_strncpy (s->services->nickserv_mask, buf[2], MASK_SIZE);
            }
          else if (strcasecmp (buf[1], "nickserv_auth") == 0)
            {
              my_strncpy (s->services->nickserv_auth, buf[2], MSG_SIZE);
            }
          else if (strcasecmp (buf[1], "servicesmsg") == 0)
            s->services->services_privmsg = 1;

          else if (strcasecmp (buf[1], "away") == 0)
            my_strncpy (s->away, buf[2], AWAY_SIZE);

          else if (strcasecmp (buf[1], "quit") == 0)
            my_strncpy (s->quit, buf[2], QUIT_SIZE);

          else if (strcasecmp (buf[1], "unbind") == 0)
            {
              if (!s->script->unbind_cmd (buf[2]))
                conf_error ("inexistent bind.");
            }

          else if (strcasecmp (buf[1], "changetime") == 0)
            s->change_time = atoi (buf[2]) * 60;

          else if (strcasecmp (buf[1], "channel") == 0)
            {
              strsplit (bufline, buf, 2);
              i = s->channel_add (buf[2], buf[3]);
              if (i == 1)
                conf_error ("maximum channels exceeded.");
              else if (i == 2)
                conf_error ("channel already defined.");
            }

          else if (strcasecmp (buf[1], "users") == 0)
            {
              if (s->users == NULL)
                s->users = new CUser (s, strcut (buf[2], FILE_SIZE));
              else
                conf_error ("users already defined.");
            }

          else if (strcasecmp (buf[1], "set") == 0)
            {
              strsplit (bufline, buf, 2);
              if (!s->vars->var_set (buf[2], buf[3]))
                {
                  snprintf (buf[4], MSG_SIZE, "inexistant variable \"%s\".",
                            buf[2]);
                  conf_warn (buf[4]);
                }
            }

          // pass control to the modules
          module_type *mlist = module_list;
          while (mlist != NULL)
            {
              if (mlist->m->module->conf != NULL)
                mlist->m->module->conf (s, bufline);
              mlist = mlist->next;
            }

        }
    }
  check_server (s);
  server_num++;
}

void
CBot::irc_exit (void)
{
  for (int i = 0; i < server_num; i++)
    delete servers[i];
  server_num = 0;
  write_botlog ("Bot exited.");
  exit (0);
}

void
CBot::work (void)
{
  while (1)			// main cycle
    {
      time_now = get_time ();
      for (int i = 0; i < server_num; i++)
        servers[i]->work ();
    }
}

