/*

  This module provides a dictionary-like database implementation. The 
keywords are not case sensitive. There are 6 available commands, they are:

!learn- word num
Delete <num> words from <word>'s definition, starting from the end.

!learn+ word text
Add <text> to <word>'s definition. If <word> doesn't exist, create it.

!learn word [text]
Add or replace <word> with <text>. If <text> if not especified, delete <word>
instead (if it exists).

!forget word
Delete the especified <word>.

!wordcount
Say the number of words defined.

?? word
Search <word> in the database and return its content.

  The options to configure:

"word somefile.word"
which of course contains the database.

"wordnotice"
which makes the "??" queries to be answered with notices instead of privmsg.

"wordnotdef That word is not defined!"
is to change the default message when words not defined are asked for.

*/

#include "mbot.h"

#define HASH_SIZE 1000			// hash table's size
#define WORD_SIZE 30			// dictionary word's size
#define TEXT_SIZE MSG_SIZE		// dictionary definition's size
#define NOTDEF_DEFAULT "Not defined... lousy word anyway :D"

#define DEST s->script->dest
#define BUF s->script->buf
#define SEND_TEXT s->script->send_text

/////////////////////
// class definition
/////////////////////

class CWord {
public:
  CWord (CServer *, const char *);
  ~CWord (void);

  struct word_type {
    char word[WORD_SIZE+1];			// key word
    long index;					// it's position on the file
    word_type *next;				// next word
  };
  struct word_type *words[HASH_SIZE+1];

  void setup_words (void);
  void register_word (const char *, long);
  void unregister_word (const char *);
  void delete_words (void);
  void word2text (const char *, char *);
  word_type *get_word (const char *);
  void add_word (const char *, const char *);
  bool del_word (const char *);

  char filename[FILE_SIZE+1];			// name of the word file
  int word_num;					// number of words
  char wordnotdef[MSG_SIZE+1];	// msg to show when a word is not defined
  bool notice;

  struct serverlist_type {
    CServer *s;
    serverlist_type *next;
  };
  serverlist_type *serverlist;			// servers to which belongs

private:

  FILE *wfd;					// file descriptor
  char buf[BUF_SIZE+1];
};

struct wordlist_type {
  CWord *word;
  wordlist_type *next;
};
struct wordlist_type *wordlist;

///////////////
// prototypes
///////////////

static void word_cmd_learn_minus (CServer *);
static void word_cmd_learn_plus (CServer *);
static void word_cmd_learn (CServer *);
static void word_cmd_forget (CServer *);
static void word_cmd_wordcount (CServer *);
static void word_cmd_word2text (CServer *);
char *learn_minus_num (char *, int );
static CWord *file2word (const char *);
static CWord *server2word (CServer *);
static void word_add (CServer *, const char *);
static void word_conf (CServer *, const char *);
static void word_stop (CModule *);
static void word_start (CModule *);

////////////////////
// class functions
////////////////////

CWord::CWord (CServer *server, const char *name)
{
  serverlist = new serverlist_type;
  serverlist->s = server;
  serverlist->next = NULL;
  my_strncpy (filename, name, FILE_SIZE);
  word_num = 0;
  my_strncpy (wordnotdef, NOTDEF_DEFAULT, MSG_SIZE);
  wfd = fopen (filename,"r+");
  if (wfd == NULL)
    {
      snprintf (buf, BUF_SIZE, "ERROR opening %s: %s", filename,
                strerror (errno));
      server->write_botlog (buf);
      server->bot->irc_exit ();
    }
  notice = 0;
  setup_words ();
}

CWord::~CWord (void)
{
  delete_words ();
  fclose (wfd);
  serverlist_type *bufserver = serverlist, *bufserver2;
  while (bufserver != NULL)
    {
      bufserver2 = bufserver->next;
      delete bufserver;
      bufserver = bufserver2;
    }
}

// read the word file into the table
void
CWord::setup_words (void)
{
  int i, f_index;
  for (i = 0; i < HASH_SIZE; i++)
    words[i]=NULL;
  word_num = 0;
  fseek (wfd, 0, SEEK_SET);
  while (1)
    {
      f_index = ftell (wfd);
      if (fgets (buf, BUF_SIZE, wfd) == NULL)
        return;
      buf[num_notspaces(buf)] = 0;		// buf now has the word
      register_word (buf, f_index);
    }
}

// put the word on the table
void
CWord::register_word (const char *word, long f_index)
{
  int i = hash (word, HASH_SIZE);
  word_type *w = words[i], *w_previous = NULL;
  if (w == NULL)			// if the list is empty
    {
      words[i] = new word_type;
      my_strncpy (words[i]->word, word, WORD_SIZE);
      words[i]->index = f_index;
      words[i]->next = NULL;
      word_num++;
      return;
    }
  while (w != NULL)			// else, find the end of the table
    {
      w_previous = w;
      w = w->next;
    }
  w = new word_type;
  my_strncpy (w->word, word, WORD_SIZE);
  w->index = f_index;
  w->next = NULL;
  w_previous->next = w;
  word_num++;
}

// delete the word from the table
void
CWord::unregister_word (const char *word)
{
  int i = hash (word, HASH_SIZE);
  word_type *w = words[i], *w_next = NULL;
  if (w != NULL)				// if the list is not empty
    {
      if (strcasecmp (word, w->word) == 0)	// if it's the first
        {
          words[i] = w->next;
          delete w;
          word_num--;
        }
      else
        while (w->next != NULL)			// else find it
          {
            if (strcasecmp (word, w->next->word) == 0)
              {
                w_next = w->next->next;
                delete w->next;
                w->next = w_next;
                word_num--;
                return;
              }
            w = w->next;
          }
    }
}

// delete everything in the word table
void
CWord::delete_words (void)
{
  word_type *w, *w2;
  for (int i = 0; i < HASH_SIZE; i++)
    {
      w = words[i];
      words[i] = NULL;
      while (w != NULL)
        {
          w2 = w->next;
          delete w;
          w = w2;
        }
    }
}

// return pointer to word, NULL if nonexistant
struct CWord::word_type *
CWord::get_word (const char *word)
{
  word_type *w;
  w = words[hash (word, HASH_SIZE)];
  while (w != NULL)
    {
      if (strcasecmp (word, w->word) == 0)
        return w;
      w = w->next;
    }
  return NULL;
}

// put in text the definition of word, or \0 if nonexistant
void
CWord::word2text (const char *word, char *text)
{
  word_type *w = get_word (word);
  if (w != NULL)
    {
      fseek (wfd, w->index, SEEK_SET);
      fgets (buf, BUF_SIZE, wfd);
      strip_crlf (buf);
      my_strncpy (text, buf + strlen (word) + 1, TEXT_SIZE);
      return;
    }
  text[0] = 0;
}

// create a new word at the end of file
void
CWord::add_word (const char *word, const char *text)
{
  if (text[0] != 0)
    {
      fseek (wfd, 0, SEEK_END);
      register_word (word, ftell (wfd));
      fprintf (wfd, "%s %s\n", word, text);
      fflush (wfd);
    }
}

// delete a word, return 0 on failure
bool
CWord::del_word (const char *word)
{
  word_type *w = get_word (word);
  if (w != NULL)
    {
      // delete from file
      FILE *tmpfd = tmpfile ();
      if (tmpfd == NULL)
        {
          serverlist->s->write_botlog ("ERROR: can't create temporary file in del_word()");
          return 0;
        }
      fseek (wfd, w->index, SEEK_SET);		// offset of word to delete
      fgets (buf, BUF_SIZE, wfd);		// move a line forward
      while (fgets (buf, BUF_SIZE, wfd) != NULL) // copy the rest to tmp
        fputs (buf, tmpfd);
      fseek (wfd, w->index, SEEK_SET);
      fseek (tmpfd, 0, SEEK_SET);
      while (fgets (buf, BUF_SIZE, tmpfd) != NULL)
        fputs (buf, wfd);
      fclose (tmpfd);
      ftruncate (fileno (wfd), ftell (wfd));
      fflush (wfd);

      // delete from memory
      unregister_word(word);

      // rearrange other word's index
      fseek (wfd, 0, SEEK_SET);
      int pos = 0, wordread = 0;
      while (wordread < word_num)
        {
          fgets (buf, BUF_SIZE, wfd);
          buf[num_notspaces (buf)] = 0;		// buf now has the word
          w = get_word (buf);
          if (w != NULL)
            w->index = pos;
          pos = ftell (wfd); 
          wordread++;
        }
      return 1;
    }
  return 0;
}

/////////////
// commands
/////////////

// !learn- word num
static void
word_cmd_learn_minus (CServer *s)
{
  CWord *w = server2word (s);
  if (w != NULL)
    {
      strsplit (CMD[3] + 8, BUF, 1);
      if (CMD[3][7] == ' ' && BUF[2][0] != 0)
        {
          if (strlen (BUF[1]) > WORD_SIZE)
            SEND_TEXT (DEST, "Word too large.");
          else
            {
              w->word2text (BUF[1], BUF[3]);		// get the previous one
              if (BUF[3][0] != 0)			// if it exists
                {
                  w->del_word (BUF[1]);			// delete it
                  SEND_TEXT (DEST, "Word changed.");
                }
              else
                SEND_TEXT (DEST, "Word added.");
              snprintf (BUF[4], MSG_SIZE, "%s",
                        learn_minus_num (BUF[3], atoi (BUF[2])));
              w->add_word (BUF[1], BUF[4]);
            }
        }
      else
        SEND_TEXT (DEST, "!learn- == !learn- word num");
    }
}

// !learn+ word text
static void
word_cmd_learn_plus (CServer *s)
{
  CWord *w = server2word (s);
  if (w != NULL)
    {
      strsplit (CMD[3] + 8, BUF, 1);
      if (CMD[3][7] == ' ' && BUF[2][0] != 0)
        {
          if (strlen (BUF[1]) > WORD_SIZE)
            SEND_TEXT (DEST, "Word too large.");
          else
            {
              w->word2text (BUF[1], BUF[3]);		// get the previous one
              if (BUF[3][0] != 0)			// if it exists
                {
                  w->del_word (BUF[1]);			// delete it
                  SEND_TEXT (DEST, "Word changed.");
                }
              else
                SEND_TEXT (DEST, "Word added.");
              snprintf (BUF[4], MSG_SIZE, "%s %s", BUF[3], BUF[2]);
              w->add_word (BUF[1], BUF[4]);
            }
        }
      else
        SEND_TEXT (DEST, "!learn+ == !learn+ word text");
    }
}

// !learn word [text]
static void
word_cmd_learn (CServer *s)
{
  CWord *w = server2word (s);
  if (w != NULL)
    {
      strsplit (CMD[3] + 7, BUF, 1);
      if (BUF[1][0] != 0 && CMD[3][6] == ' ')
        {
          if (strlen (BUF[1]) > WORD_SIZE)
            SEND_TEXT (DEST, "Word too large.");
          else
            {
              my_strncpy (BUF[3], "Word added.", MSG_SIZE);
              if (w->get_word (BUF[1]) != NULL)	// if it already exists
                {
                  w->del_word (BUF[1]);				// delete it
                  if (BUF[2][0] == 0)
                    my_strncpy (BUF[3], "Word deleted.", MSG_SIZE);
                  else
                    my_strncpy (BUF[3], "Word changed.", MSG_SIZE);
                }
              w->add_word (BUF[1], BUF[2]);
              SEND_TEXT (DEST, BUF[3]);
            }
        }
      else
        SEND_TEXT (DEST, "!learn == !learn word [text]");
    }
}

// !forget word
static void
word_cmd_forget (CServer *s)
{
  CWord *w = server2word (s);
  if (w != NULL)
    {
      strsplit (CMD[3] + 8, BUF, 1);
      if (BUF[1][0] != 0 && CMD[3][7] == ' ')
        {
          if (!w->del_word (BUF[1]))
            SEND_TEXT (DEST, "That word does not exist.");
          else
            SEND_TEXT (DEST, "Word deleted.");
        }
      else
        SEND_TEXT (DEST, "!forget == !forget word");
    }
}

// !wordcount
static void
word_cmd_wordcount (CServer *s)
{
  CWord *w = server2word (s);
  if (w != NULL && CMD[3][10] == 0)
    {
      snprintf (BUF[1], MSG_SIZE, "There are %d words defined.", w->word_num);
      SEND_TEXT (DEST, BUF[1]);
    }
}

// ?? word
static void
word_cmd_word2text (CServer *s)
{
  CWord *w = server2word (s);
  if (w != NULL)
    {
      strsplit (CMD[3] + 3, BUF, 1);
      if (BUF[1][0] != 0 && CMD[3][2] == ' ')
        {
          w->word2text(BUF[1], BUF[2]);
          if (BUF[2][0] == 0)
            my_strncpy (BUF[2], w->wordnotdef, MSG_SIZE);
          if (w->notice)
            {
              snprintf (BUF[3], MSG_SIZE, "%s == %s", BUF[1], BUF[2]);
              s->irc_notice (DEST, BUF[3]);		// broken in dcc chat!
            }
          else
            SEND_TEXT(DEST, BUF[2]);
        }
    }
}

////////////////////
// module managing
////////////////////

// Possidon made this one.. so blame him, not me *g*
char *
learn_minus_num (char *ori, int m)
{
  int i = strlen (ori) - 1, n = 0, l = 0;
  while (i > 0 && n < m)
    {
      if (ori[i] != ' ')
        l = 1;
      while (ori[i] == ' ')
        {
          if (l != 0)
            {
              n++;
              l = 0;
            }
          i--;
        }
      i--;
    }
  if (i > 0)
    ori[i + 2] = '\0';
  return ori;
}

// returns the word object for a given word file, NULL if nonexistant
static CWord *
file2word (const char *filename)
{
  wordlist_type *wl = wordlist;

  while (wl != NULL)
    {
      if (wl->word != NULL)
        if (strcmp (wl->word->filename, filename) == 0)
          return wl->word;
      wl = wl->next;
    }
  return NULL;
}

// returns the word object for a given server, NULL if nonexistant
static CWord *
server2word (CServer *s)
{
  wordlist_type *wl = wordlist;

  while (wl != NULL && wl->word != NULL)
    {
      CWord::serverlist_type *sl = wl->word->serverlist;
      while (sl != NULL)
        {
          if (sl->s == s)
            return wl->word;
          sl = sl->next;
        }
      wl = wl->next;
    }
  return NULL;
}

// adds a server/channel pair to the list
static void
word_add (CServer *s, const char *name)
{
  if (wordlist == NULL)			// if it's empty
    {
      wordlist = new wordlist_type;
      wordlist->word = new CWord (s, name);
      wordlist->next = NULL;
      return;
    }

  CWord *word = file2word (name);
  if (word != NULL)
    {
      if (word->serverlist != NULL)	// should never fail, though
        {
          CWord::serverlist_type *sl_old = word->serverlist;
          word->serverlist = new CWord::serverlist_type;
          word->serverlist->s = s;
          word->serverlist->next = sl_old;
        }
      return;
    }

  wordlist_type *buf = wordlist;
  while (buf->next != NULL)		// else finds the last
    buf = buf->next;
  buf->next = new wordlist_type;
  buf->next->word = new CWord (s, name);
  buf->next->next = NULL;
}

// configuration file's local parser
static void
word_conf (CServer *s, const char *bufread)
{
  char buf[10][MSG_SIZE+1];

  strsplit (bufread, buf, 1);

  if (strcasecmp (buf[1], "words") == 0)
    {
      if (server2word (s) == NULL)
        {
          word_add (s, buf[2]);
          s->script->bind_cmd (word_cmd_learn_minus, 7, "!learn-");
          s->script->bind_cmd (word_cmd_learn_plus, 7, "!learn+");
          s->script->bind_cmd (word_cmd_learn, 7, "!learn");
          s->script->bind_cmd (word_cmd_forget, 7, "!forget");
          s->script->bind_cmd (word_cmd_wordcount, 0, "!wordcount");
          s->script->bind_cmd (word_cmd_word2text, 0, "??");
        } else
          s->bot->conf_error ("words already defined for this server.");
    }
  else if (strcasecmp(buf[1], "wordnotice") == 0)
    {
      CWord *w = server2word (s);
      if (w != NULL)
        w->notice = 1;
      else
        s->bot->conf_error ("words not yet defined for this server.");
    }

  if (strcasecmp (buf[1], "wordnotdef") == 0)
    {
      CWord *w = server2word (s);
      if (w != NULL)
        my_strncpy (w->wordnotdef, buf[2], MSG_SIZE);
      else
        s->bot->conf_error ("words not yet defined for this server.");
    }
}

// module termination
static void
word_stop (CModule *m)
{
  wordlist_type *buf = wordlist, *buf2;
  while (buf != NULL)
    {
      // serverlist is deleted by CWord's destructor
      CWord::serverlist_type *sl_buf = buf->word->serverlist;
      while (sl_buf != NULL)
        {
          sl_buf->s->script->unbind_cmd ("!learn-");
          sl_buf->s->script->unbind_cmd ("!learn+");
          sl_buf->s->script->unbind_cmd ("!learn");
          sl_buf->s->script->unbind_cmd ("!forget");
          sl_buf->s->script->unbind_cmd ("!wordcount");
          sl_buf->s->script->unbind_cmd ("??");
          sl_buf = sl_buf->next;
        }
      delete buf->word;
      buf2 = buf->next;
      delete buf;
      buf = buf2;
    }
}

// module initialization
static void
word_start (CModule *m)
{
  wordlist = NULL;
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "word",
  word_start,
  word_stop,
  word_conf,
  NULL
};
