/*

  This module implements the dcc related commands. Although the mechanisms
these commands use are in the main source, there may be interest in not
activating them. They are:

!dcckill dcc_index
Terminate the DCC specified by <dcc_index>, use !dcclist to see the indexes.

!dcclist
Show the list of active DCCs (Chat and Send).

!chat
Make the bot start a DCC Chat with who executed it.

!get [file]
Without parameters, show the files available through DCC Send, else requests
the specified <file>.

  The options to configure:

"dccget somefile.get"
Uses somefile.get's informations to configure !get. See example.get for more
details on this.

"dccport port"
Uses the specified port in dcc chats and sends. Very useful when the bot is
behind a firewall. The default is 0, which makes it use any available port.

"dccmotd somefile.motd"
Show this file to everyone who enters the dcc chat partyline.

*/

#include "mbot.h"

#define SOURCE s->script->source
#define DEST s->script->dest
#define BUF s->script->buf
#define SEND_TEXT s->script->send_text

///////////////
// prototypes
///////////////

static void main_dcc_cmd_dcclist (CServer *);
static void main_dcc_cmd_dcckill (CServer *);
static void main_dcc_cmd_chat (CServer *);
static void main_dcc_cmd_get (CServer *);
static void main_dcc_conf (CServer *, const char *);
static void main_dcc_stop (CModule *);

/////////////
// commands
/////////////

// !dcckill dcc_index
static void
main_dcc_cmd_dcckill (CServer *s)
{
  strsplit (CMD[3] + 9, BUF, 1);
  if (BUF[1][0] != 0 && CMD[3][8] == ' ')
    {
      int dcc_index = atoi (BUF[1]);
      if (dcc_index < 0 || dcc_index > DCC_NUM - 1)
        SEND_TEXT (DEST, "Invalid DCC index.");
      else
        {
          if (DCCS[dcc_index]->status == 5)
            {
              snprintf (BUF[0], MSG_SIZE, "%s killed %s.", SOURCE,
                        DCCS[dcc_index]->nick);
              s->script->send_partyline (BUF[0]);
            }
          DCCS[dcc_index]->dcc_stop ();
        }
    }
  else
    SEND_TEXT (DEST, "dcckill == !dcckill dcc_index");
}

// !dcclist
static void
main_dcc_cmd_dcclist (CServer *s)
{
  if (CMD[3][9] == 0)
    {
      if (DCC_NUM == 0)
        {
          SEND_TEXT (DEST, "No DCCs running.");
          return;
        }
      for (int i = 0; i < DCC_NUM; i++)
        {
          switch (DCCS[i]->status)
            {
              case DCC_STOP:
                strcpy(BUF[1], "closing");
                break;
              case DCC_SEND_INIT:
                strcpy(BUF[1], "Send (not connected yet)");
                break;
              case DCC_SEND:
                strcpy(BUF[1], "Send");
                break;
              case DCC_CHAT_INIT:
                strcpy(BUF[1], "Chat (not connected yet)");
                break;
              case DCC_CHAT_AUTH:
                strcpy(BUF[1], "Chat (authenticating)");
                break;
              case DCC_CHAT:
                strcpy(BUF[1], "Chat");
                break;
              default:
                strcpy(BUF[1], "unknown");
            }
          sprintf (BUF[0], "%d: DCC %s with %s", i, BUF[1], DCCS[i]->mask);
          SEND_TEXT (DEST, BUF[0]);
        }
    }
}

// !chat
static void
main_dcc_cmd_chat (CServer *s)
{
  if (CMD[3][5] != 0)
    return;
  if ((USERS == NULL ? 0 : USERS->mask_level(CMD[0])) >= 1)
    {
      for (int i = 0; i < DCC_NUM; i++)
        if (strcasecmp (DCCS[i]->nick, SOURCE) == 0
            && DCCS[i]->status >= 3 && DCCS[i]->status <= 5)
          {
            SEND_TEXT (SOURCE, "Another DCC Chat is in progress for you.");
            return;
          }
      if (DCC_NUM == DCC_MAX)
        {
          SEND_TEXT (SOURCE, "No free DCC slots available, please try later.");
          return;
        }
      int dcc_index = ((CMD[2][0] >= '0' && CMD[2][0] <= '9') ? atoi (CMD[2]) : -1);
      DCCS[DCC_NUM] = new CDCC (s, CMD[0], DCC_NUM, dcc_index);
      // if it gives error, work()'ll delete it
      if (DCCS[DCC_NUM]->dcc_chat_start ())
        SEND_TEXT (SOURCE, DCCS[DCC_NUM]->dcc_make_ctcp_chat (BUF[3], MSG_SIZE));
      DCC_NUM++;
    }
  else
    SEND_TEXT (SOURCE, "Sorry, you don't have access.");
}

// !get [file]
static void
main_dcc_cmd_get (CServer *s)
{
  FILE *tmpfd = fopen (s->dcc_file, "r+");
  if (tmpfd == NULL)
    {
      if (s->bot->debug)
        perror ("ERROR opening DCC get's file");
      SEND_TEXT (SOURCE, "Error opening the configuration file!");
      return;
    }
  strsplit (CMD[3] + 5, BUF, 1);
  if (BUF[1][0] != 0 && CMD[3][4] == ' ')
    {
      for (int i = 0; i < DCC_NUM; i++)
        if (strcasecmp (DCCS[i]->nick, SOURCE) == 0 &&
            (DCCS[i]->status == 1 || DCCS[i]->status == 2))
          {
            SEND_TEXT (SOURCE, "Another DCC Send is in progress for you.");
            fclose (tmpfd);
            return;
          }
      if (DCC_NUM == DCC_MAX)
        {
          SEND_TEXT (SOURCE, "No free DCC slots available, please try later.");
          fclose (tmpfd);
          return;
        }
      char file[FILE_SIZE+1];			// to store the requested file
      my_strncpy (file, BUF[1], FILE_SIZE);
      while (fgets (BUF[7], MSG_SIZE, tmpfd) != NULL)
        if (BUF[7][0] != ':' && BUF[7][0] != '#')
          {
            strsplit (BUF[7], BUF, 2);
            if (strcasecmp (BUF[1], file) == 0)
              {
                if(BUF[2][strlen (BUF[2])-1] == 10)		// remove the '\n'
                  BUF[2][strlen (BUF[2])-1] = 0;
                int dcc_index = ((CMD[2][0] >= '0' && CMD[2][0] <= '9') ? atoi (CMD[2]) : -1);
                // create the dcc
                DCCS[DCC_NUM] = new CDCC (s, CMD[0], DCC_NUM, dcc_index);
                // if it gives error, work()'ll delete it
                if (DCCS[DCC_NUM]->dcc_send_start (BUF[2]))
                  SEND_TEXT (SOURCE, DCCS[DCC_NUM]->dcc_make_ctcp_send (BUF[2], MSG_SIZE));
                DCC_NUM++;
                fclose (tmpfd);
                return;
              }
          }
      SEND_TEXT (SOURCE, "Invalid file.");
    }
  else
    {
      while (fgets (BUF[7], MSG_SIZE, tmpfd) != NULL)
        {
          strsplit (BUF[7], BUF, 0);		// delete spaces from the start
          if (BUF[1][0] == ':')			// if it's help
            {
              my_strncpy (BUF[8], BUF[1] + 1, MSG_SIZE); // remove the ':'
              strip_crlf (BUF[8]);		// remove the '\n'
              SEND_TEXT (SOURCE, BUF[8]);
            }
        }
    }
  fclose(tmpfd);
}

////////////////////
// module managing
////////////////////

// look for ctcp dcc chat on privmsg
static void
main_dcc_event (CServer *s)
{
  if (CMD[3][0] == '' && strcmp(CMD[1], "PRIVMSG") == 0)
    {
      mask2nick (CMD[0], DEST);
      if (strncmp (CMD[3], "DCC CHAT", 9) == 0)
        {
          s->irc_notice (DEST, "DCC REJECT CHAT chat");
          SEND_TEXT (DEST, "Please use the !chat command to start a DCC Chat.");
        }
    }
}

// configuration file's local parser
static void
main_dcc_conf (CServer *s, const char *confline)
{
  char buf[10][MSG_SIZE+1];

  strsplit (confline, buf, 1);

  if (strcasecmp (buf[1], "bot") == 0)
    {
      s->script->bind_cmd (main_dcc_cmd_dcckill, 9, "!dcckill");
      s->script->bind_cmd (main_dcc_cmd_dcclist, 5, "!dcclist");
      s->script->bind_cmd (main_dcc_cmd_chat, 0, "!chat");
      s->script->add_event (main_dcc_event);
    }

  else if (strcasecmp (buf[1], "dccget") == 0)
    {
      my_strncpy (s->dcc_file, buf[2], FILE_SIZE);
      s->script->bind_cmd (main_dcc_cmd_get, 0, "!get");
    }

  else if (strcasecmp (buf[1], "dccport") == 0)
    s->dcc_port = atoi(buf[2]);

  else if (strcasecmp (buf[1], "dccmotd") == 0)
    {
      my_strncpy (s->dcc_motdfile, buf[2], FILE_SIZE);
      s->dcc_motd = new CText ();
      if (!s->dcc_motd->read_file (s->dcc_motdfile))
        {
          snprintf (buf[2], MSG_SIZE, "ERROR opening %s: %s",
                    s->dcc_motdfile, strerror (errno));
          s->write_botlog (buf[2]);
          delete s->dcc_motd;
          s->dcc_motd = NULL;
        }
      else
        s->dcc_motd->strip_text_crlf ();
    }
}

// module termination
static void
main_dcc_stop (CModule *m)
{
  // try to remove from all servers, even if it's not there (we don't know)
  for (int i = 0; i < m->b->server_num; i++)
    {
      if (m->b->servers[i]->dcc_motd != NULL)
        {
          delete m->b->servers[i]->dcc_motd;
          m->b->servers[i]->dcc_motd = NULL;
        }
      m->b->servers[i]->script->unbind_cmd ("!dcckill");
      m->b->servers[i]->script->unbind_cmd ("!dcclist");
      m->b->servers[i]->script->unbind_cmd ("!chat");
      m->b->servers[i]->script->del_event (main_dcc_event);
      m->b->servers[i]->script->unbind_cmd ("!get");
    }
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "maindcc",
  NULL,
  main_dcc_stop,
  main_dcc_conf,
  NULL
};
