/*

  This module adds some standard CTCP commands, namely: version, finger,
userinfo, ping, time, source, echo and clientinfo. It does NOT provide dcc
chat or send. The following ctcp replies can be set:

set ctcp_version <text>
set ctcp_finger <text>
set ctcp_source <text>

*/

#include "mbot.h"

/* default replies */
#define CTCP_VERSION VERSION_STRING
#define CTCP_FINGER "Finger?! Stick it up your ass!"
#define CTCP_SOURCE "Source available at http://alunos.uevora.pt/~l12697/prog/mbot.tgz"

#define DEST s->script->dest
#define BUF s->script->buf
#define SEND_TEXT s->script->send_text

struct ctcp_type {
  CServer *server;
  char version[MSG_SIZE+1];
  char finger[MSG_SIZE+1];
  char source[MSG_SIZE+1];
  ctcp_type *next;
};
struct ctcp_type *ctcp_list;

///////////////
// prototypes
///////////////

static void ctcp_add (CServer *);
static ctcp_type *server2ctcp (CServer *);
static void ctcp_event (CServer *);
static void ctcp_conf (CServer *, const char *);
static void ctcp_stop (CModule *);
static void ctcp_start (CModule *);
static void ctcp_var (CServer *, const char *, char *, size_t);

////////////////////
// module managing
////////////////////

// add a ctcp to the list
static void
ctcp_add (CServer *s)
{
  ctcp_type *buf = ctcp_list;
  ctcp_list = new ctcp_type;
  ctcp_list->server = s;
  my_strncpy (ctcp_list->version, CTCP_VERSION, MSG_SIZE);
  my_strncpy (ctcp_list->finger, CTCP_FINGER, MSG_SIZE);
  my_strncpy (ctcp_list->source , CTCP_SOURCE, MSG_SIZE);
  ctcp_list->next = buf;
}

// returns the ctcp for a given server, NULL if nonexistant
static ctcp_type *
server2ctcp (CServer *s)
{
  ctcp_type *buf = ctcp_list;
  while (buf != NULL)
    {
      if (buf->server == s)
        return buf;
      buf = buf->next;
    }
  return NULL;
}

// look for ctcp's on privmsg
static void
ctcp_event (CServer *s)
{
  if (CMD[3][0] == '' && strcmp (CMD[1], "PRIVMSG") == 0)
    {
      mask2nick (CMD[0], DEST);
      ctcp_type *ctcp = server2ctcp (s);
      if (ctcp == NULL)
        return;

      if (strcmp (CMD[3], "VERSION") == 0)
        {
          snprintf (BUF[1], MSG_SIZE, "VERSION %s", ctcp->version);
          s->irc_notice (DEST, BUF[1]);
        }
      else if (strcmp (CMD[3], "FINGER") == 0)
        {
          snprintf (BUF[1], MSG_SIZE, "FINGER %s", ctcp->finger);
          s->irc_notice (DEST, BUF[1]);
        }
      else if (strcmp (CMD[3], "USERINFO") == 0)
        {
          snprintf (BUF[1], MSG_SIZE, "USERINFO %s running on %s", VERSION_STRING, HOST);
          s->irc_notice (DEST, BUF[1]);
        }
      else if (strncmp (CMD[3], "PING", 5) == 0)
        {
          s->irc_notice (DEST, CMD[3]);
        }
      else if (strcmp (CMD[3], "TIME") == 0)
        {
          snprintf (BUF[1], MSG_SIZE, "TIME %s", get_asctime (s->time_now, BUF[2], MSG_SIZE));
          s->irc_notice (DEST, BUF[1]);
        }
      else if (strcmp (CMD[3], "SOURCE") == 0)
        {
          snprintf (BUF[1], MSG_SIZE, "SOURCE %s", ctcp->source);
          s->irc_notice (DEST, BUF[1]);
        }
      else if (strncmp (CMD[3], "ECHO", 5) == 0)
        {
          s->irc_notice (DEST, CMD[3]);
        }
      else if (strcmp (CMD[3], "CLIENTINFO") == 0)
        {
          s->irc_notice(DEST, "CLIENTINFO VERSION FINGER USERINFO PING TIME SOURCE ECHO CLIENTINFO");
        }
    }
}

// configuration file's local parser
static
void ctcp_conf (CServer *s, const char *bufread)
{
  char buf[10][MSG_SIZE+1];

  strsplit (bufread, buf, 1);

  if (strcasecmp (buf[1], "server") == 0
      || strcasecmp (buf[1], "bot") == 0)
    {
      if (server2ctcp (s) == NULL)
        {
          ctcp_add (s);
          s->script->add_event (ctcp_event);
          s->vars->var_add ("ctcp_version", ctcp_var);
          s->vars->var_add ("ctcp_finger", ctcp_var);
          s->vars->var_add ("ctcp_source", ctcp_var);
        }
    }
  else if (strcasecmp (buf[1], "version") == 0)
    {
      ctcp_type *ctcp = server2ctcp (s);
      if (ctcp != NULL)
        my_strncpy (ctcp->version, buf[2], MSG_SIZE);
    }
}

// module termination
static void
ctcp_stop (CModule *m)
{
  ctcp_type *buf = ctcp_list, *buf2;
  while (buf != NULL)
    {
      buf->server->script->del_event (ctcp_event);
      buf2 = buf->next;
      delete buf;
      buf = buf2;
    }
  // try to remove from all servers, even if it's not there (we don't know)
  for (int i = 0; i < m->b->server_num; i++)
    {
      m->b->servers[i]->vars->var_del ("ctcp_version");
      m->b->servers[i]->vars->var_del ("ctcp_finger");
      m->b->servers[i]->vars->var_del ("ctcp_source");
    }
}

// module initialization
static void
ctcp_start (CModule *m)
{
  ctcp_list = NULL;
}

static void
ctcp_var (CServer *s, const char *name, char *data, size_t n)
{
  ctcp_type *buf = server2ctcp (s);
  if (buf == NULL)
    return;
  if (strcasecmp (name, "ctcp_version") == 0)
    {
      if (n != 0)
        my_strncpy (data, buf->version, n);
      else
        my_strncpy (buf->version, data, MSG_SIZE);
    }
  else if (strcasecmp (name, "ctcp_finger") == 0)
    {
      if (n != 0)
        my_strncpy (data, buf->finger, n);
      else
        my_strncpy (buf->finger, data, MSG_SIZE);
    }
  else if (strcasecmp (name, "ctcp_source") == 0)
    {
      if (n != 0)
        my_strncpy (data, buf->source, n);
      else
        my_strncpy (buf->source, data, MSG_SIZE);
    }
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "ctcp",
  ctcp_start,
  ctcp_stop,
  ctcp_conf,
  ctcp_var
};
