/*

  This module creates a html file with status about a certain channel in
which the bot is. Note that only one channel per server can be chosen. The
page is by default refreshed each minute. The html design was taken from an
eggdrop script, but you can change it at will.

  Just put this in the configuration file:
"chanstat #channel filename_html filename_stat number"

 filename_html: the html to be created
 filename_stat: where the history stats are kept. THE FILENAME_STAT IS NOT 
   BEING USED YET! So don't worry about it. Just put there a valid name,
   although it won't be used (yet).
 number: 1 or 0, depending on whether you want to show the users' address
         or not.

  You can also set the time between stat refreshes with
"set chanstat_refresh <number of seconds>"
whose default is 60.

*/

#include "mbot.h"

#define DEFAULT_REFRESH 60		// refresh the pages each 60 seconds

#define DEST s->script->dest
#define BUF s->script->buf
#define SEND_TEXT s->script->send_text

struct chanstat_type {
  CServer *server;
  char channel[CHANNEL_SIZE+1];
  char file_html[FILE_SIZE+1];
  char file_stat[FILE_SIZE+1];
  bool show_addr;
  time_t last_time;
  time_t refresh;
  struct top10_type {
    char nick[NICK_SIZE+1];
    time_t uptime;
  } top10[10];
  int usersperday;	// number of users who entered the channel today
  int currentusers;		// number of users currently in the channel
  time_t time_top_currentusers;	// when 
  chanstat_type *next;
};
struct chanstat_type *chanstat_list;

///////////////
// prototypes
///////////////

static void chanstat_work (chanstat_type *);
static void chanstat_add (CServer *, const char *, const char *, const char *, const char *);
static chanstat_type *server2chanstat (CServer *);
static void chanstat_event (CServer *);
static void chanstat_conf (CServer *, const char *);
static void chanstat_stop (CModule *);
static void chanstat_start (CModule *);
static void chanstat_var (CServer *, const char *, char *, size_t);

////////////////////
// create the html
////////////////////

static
void chanstat_work (chanstat_type *c)
{
  FILE *fd = fopen (c->file_html, "w");			// overwrite the file
  if (fd == NULL)
    return;
  #define _BUF_SIZE 100
  char buf[_BUF_SIZE+1], buf2[_BUF_SIZE+1];

  fprintf (fd, "<META HTTP-EQUIV=\"refresh\" content=\"%d; url=%s\">\n",
           (int)c->refresh, fullpath2file (c->file_html));
  get_asctime (c->server->time_now, buf, _BUF_SIZE);
  buf[strlen (buf) - 1]=0;
  fprintf (fd, "<TITLE>%s IRC channel stats (Written at %s)</TITLE>\n",
           c->channel, buf);
  fprintf (fd, "<CENTER><FONT SIZE=\"6\">%s's %s stats</FONT></CENTER>\n",
           c->server->nick, c->channel);

  int index = c->server->channel_index (c->channel);	// get the channel
  int inside = (index != -1);
  if (inside)
    inside = c->server->channels[index]->joined;

  if (inside)						// if inside it
    {
      CChannel *chan=c->server->channels[index];

      if (chan->topic[0] == 0)
        fprintf (fd, "<CENTER><FONT SIZE=\"5\">No topic is set.</FONT></CENTER><BR>\n");
      else
        fprintf (fd, "<CENTER><FONT SIZE=\"5\">Topic: %s</FONT></CENTER><BR>\n",
                 chan->topic);

      fprintf (fd, "<CENTER><TABLE BORDER=3>\n");
      fprintf (fd, "<TR><TD COLSPAN=3><CENTER><FONT SIZE=5>Channel %s, %d users</FONT></CENTER></TD></TR>\n",
               c->channel, chan->user_num);
      fprintf (fd, "<TR><TD bgcolor=0000ff><FONT COLOR=ffff00><B>Nickname</B></td><td bgcolor=0000ff><FONT COLOR=ffff00><B>Ident</B></td><td bgcolor=0000ff><font color=ffff00><b>Address</B></td></tr>\n");

      int i;
      for (i = 0; i < chan->user_num; i++)		// list ops
        if (chan->users[i]->is_op)
          {
            mask2user(chan->users[i]->mask, buf);
            if (c->show_addr)
              mask2host(chan->users[i]->mask, buf2);
            else
              my_strncpy(buf2, "(hidden)", _BUF_SIZE);
            fprintf (fd, "<TR><TD><FONT COLOR=ff00ff>@</FONT>%s</TD><TD>%s</TD><TD>%s</TD></TR>\n",
                     chan->users[i]->nick, buf, buf2);
          }
      for (i = 0; i < chan->user_num; i++)		// list voices
        if ((!chan->users[i]->is_op) && (chan->users[i]->is_voice))
          {
            mask2user(chan->users[i]->mask, buf);
            if (c->show_addr)
              mask2host(chan->users[i]->mask, buf2);
            else
              my_strncpy(buf2, "(hidden)", _BUF_SIZE);
            fprintf (fd, "<TR><TD><FONT COLOR=009900>+</FONT>%s</TD><TD>%s</TD><TD>%s</TD></TR>\n",
                     chan->users[i]->nick, buf, buf2);
          }
      for (i = 0; i < chan->user_num; i++)	// list non-ops and non-voices
        if ((!chan->users[i]->is_op) && (!chan->users[i]->is_voice))
          {
            mask2user(chan->users[i]->mask, buf);
            if (c->show_addr)
              mask2host(chan->users[i]->mask, buf2);
            else
              my_strncpy(buf2, "(hidden)", _BUF_SIZE);
            fprintf (fd, "<TR><TD>%s</TD><TD>%s</TD><TD>%s</TD></TR>\n",
                     chan->users[i]->nick, buf, buf2);
          }
      fprintf (fd, "</TABLE>\n");

      // put here bans table

      fprintf(fd, "</CENTER>\n");

    }
  else
    {
      fprintf (fd, "<BR><FONT SIZE=5><B>I (%s) am currently not on %s, try again later.</B></FONT>\n",
               c->server->nick, c->channel);
    }
  
  fprintf (fd, "<BR>\n<HR>This page is updated every %d seconds.\n",
           (int)c->refresh);

  fclose (fd);
}

////////////////////
// module managing
////////////////////

// add a chanstat to the list
static void
chanstat_add (CServer *s, const char *name, const char *f_html, const char *f_stat, const char *show)
{
  chanstat_type *buf = chanstat_list;
  chanstat_list = new chanstat_type;
  chanstat_list->server = s;
  my_strncpy (chanstat_list->channel, name, CHANNEL_SIZE);
  my_strncpy (chanstat_list->file_html, f_html, FILE_SIZE);
  my_strncpy (chanstat_list->file_stat, f_stat, FILE_SIZE);
  chanstat_list->show_addr = !(show[0] == '0');
  chanstat_list->last_time = 0;
  chanstat_list->refresh = DEFAULT_REFRESH;
  chanstat_list->next = buf;
}

// return the chanstat for a given server, NULL if nonexistant
static chanstat_type *
server2chanstat (CServer *s)
{
  chanstat_type *buf = chanstat_list;
  while (buf != NULL)
    {
      if (buf->server == s)
        return buf;
      buf = buf->next;
    }
  return NULL;
}

// check if it's time to update the file whenever an event occurs
static void
chanstat_event (CServer *s)
{
  chanstat_type *c = server2chanstat (s);
  if (c != NULL)
    if (difftime (s->time_now, c->last_time) > c->refresh)
      {
        chanstat_work (c);
        c->last_time = s->time_now;
      }
}

// configuration file's local parser
static void
chanstat_conf (CServer *s, const char *bufread)
{
  char buf[10][MSG_SIZE+1];

  strsplit (bufread, buf, 5);

  if (strcasecmp (buf[1], "chanstat") == 0)
    if (server2chanstat (s) == NULL)
      {
        // 5th parameter is optional for backwards compatibility
        if (buf[4][0] != 0) {
          chanstat_add (s, buf[2], buf[3], buf[4], buf[5]);
          s->script->add_event (chanstat_event);
        } else
          s->bot->conf_error ("required parameter missing for chanstat.");
        s->vars->var_add ("chanstat_refresh", chanstat_var);
      }
    else
      s->bot->conf_error ("chanstat already defined for this server.");
}

// module termination
static void
chanstat_stop (CModule *m)
{
  chanstat_type *buf = chanstat_list, *buf2;
  while (buf != NULL)
    {
      buf->server->script->del_event (chanstat_event);
      buf2 = buf->next;
      delete buf;
      buf = buf2;
    }
  // try to remove from all servers, even if it's not there (we don't know)
  for (int i = 0; i < m->b->server_num; i++)
    m->b->servers[i]->vars->var_del ("chanstat_refresh");
}

// module initialization
static void
chanstat_start (CModule *m)
{
  chanstat_list = NULL;
}

static void
chanstat_var (CServer *s, const char *name, char *data, size_t n)
{
  if (strcasecmp (name, "chanstat_refresh") == 0)
    {
      chanstat_type *buf = server2chanstat (s);
      if (buf == NULL)
        return;
      if (n != 0)
        ltoa (buf->refresh, data, n);
      else
        buf->refresh = atol (data);
    }
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "chanstat",
  chanstat_start,
  chanstat_stop,
  chanstat_conf,
  chanstat_var
};
