#include "CUser.h"

CUser::CUser (CServer *server, const char *name)
{
  s = server;
  my_strncpy (filename, name, FILE_SIZE);
  user_num = 0;
  ufd = NULL;
  setup_users (1);
}

CUser::~CUser (void)
{
  close_users ();
}

void
CUser::open_users (void)
{
  if (ufd == NULL)
    {
      ufd = fopen (filename, "r+");
      if (ufd == NULL)
        {
          if (s->bot->debug)
          fprintf (stderr, "ERROR opening %s: %s\n", filename,
                   strerror (errno));
          s->bot->irc_exit ();
        }
    }
}

void
CUser::close_users (void)
{
  if (ufd != NULL)
    {
      fclose (ufd);
      ufd = NULL;
    }
}

void
CUser::setup_users (int reset)
{
  user_num = 0;
  users[0].index = 0;
  char *b;
  open_users ();
  fseek (ufd, 0, SEEK_SET);
  while (fgets (buf, sizeof (buf), ufd) != NULL)
    {
      b = strtok (buf, ":");
      if (b != NULL)
        my_strncpy (users[user_num].mask, b, MASK_SIZE);
      b = strtok (NULL, ":");
      if (b != NULL)
        my_strncpy (users[user_num].pass, b, ENC_PASS_SIZE);
      b = strtok (NULL, ":");
      if (b != NULL)
        users[user_num].level = atoi(b);
      int user_error = 1;
      b = strtok (NULL, "");
      if (b != NULL)
        {
          my_strncpy (users[user_num].msg, b, MSG_SIZE);
          user_error = 0;
        }
      if (user_error)
        {
          if (s->bot->debug)
            fprintf (stderr, "ERROR reading %s: invalid field\n", filename);
          s->bot->irc_exit ();
        }
      strip_crlf (users[user_num].msg);
      if (reset) 
        users[user_num].id = 0;
      user_num++;
      users[user_num].index = ftell (ufd);
      users[user_num].abs_mask[0] = 0;
    }
  close_users ();
}

// add a user, return 0 on failure
bool
CUser::add_user (const char *mask, const char *crypt_pass, int level,
                 const char *msg)
{
  itoa (level, buf, MSG_SIZE);
  if (mask[0] != 0 && buf[0] != '0')
    {
      // if it exists, change it
      for (int i = 0; i < user_num; i++)
        if (strcasecmp (mask, users[i].mask) == 0)
          {
            int c;
            FILE *tmpfd = tmpfile ();
            if (tmpfd == NULL)
              {
                s->write_botlog ("ERROR: can't create temporary file in add_user()");
                return 0;
              }
            open_users ();
            fseek (ufd, users[i+1].index, SEEK_SET);
            while ( (c=fgetc (ufd)) != EOF)
              fputc (c, tmpfd);
            fseek (ufd, users[i].index, SEEK_SET);
            fprintf (ufd, "%s:%s:%s:%s\n", mask, crypt_pass, buf, msg);
            fseek (tmpfd, 0, SEEK_SET);
            while (fgets (buf, sizeof (buf), tmpfd) != NULL)
              fputs (buf, ufd);
            fclose (tmpfd);
            ftruncate (fileno (ufd), ftell (ufd));
            setup_users (0);
            close_users ();
            return 1;
          }
      // otherwise create one at the end
      if (user_num < USER_MAX)
        {
          open_users ();
          fseek (ufd, 0, SEEK_END);
          fprintf (ufd, "%s:%s:%s:%s\n", mask, crypt_pass, buf, msg);
          my_strncpy (users[user_num].mask, mask, MASK_SIZE);
          my_strncpy (users[user_num].pass, crypt_pass, ENC_PASS_SIZE);
          users[user_num].level = level;
          users[user_num].id = 0;
          my_strncpy (users[user_num].msg, msg, MSG_SIZE);
          user_num++;
          users[user_num].index = ftell (ufd);
          users[user_num].abs_mask[0] = 0;
          close_users ();
          return 1;
        }
    }
  return 0;
}

// delete a user, return 0 on failure
bool
CUser::del_user (const char *mask)
{
  for (int i = 0; i < user_num; i++)
    if (strcasecmp (mask, users[i].mask) == 0)
      {
        FILE *tmpfd = tmpfile ();
        if (tmpfd == NULL)
          {
             s->write_botlog ("ERROR: can't create temporary file in del_user()");
             return 0;
          }
        open_users ();
        fseek (ufd, users[i+1].index, SEEK_SET);
        while (fgets (buf, sizeof (buf), ufd) != NULL)
          fputs (buf, tmpfd);
        fseek (ufd, users[i].index, SEEK_SET);
        fseek (tmpfd, 0, SEEK_SET);
        while (fgets (buf, sizeof (buf), tmpfd) != NULL)
          fputs (buf, ufd);
        fclose (tmpfd);
        ftruncate (fileno (ufd), ftell (ufd));
        setup_users (0);
        for (i = i; i < user_num; i++)
          users[i].id = users[i+1].id;
        close_users ();
        return 1;
      }
  return 0;
}

// return 1 if <pass> is user <mask>'s password
bool
CUser::check_pass (const char *mask, const char *pass)
{
  int i;
  for (i = 0; i < user_num; i++)
    if (match_mask (mask, users[i].mask))
      {
        if (strcmp (users[i].pass, crypt (pass, SALT)) == 0)
          return 1;
        return 0;
      }
  return 0;
}

// set user <mask>'s password to <pass>
void
CUser::set_pass (const char *mask, const char *pass)
{
  int i;
  for (i = 0; i < user_num; i++)
    if (match_mask (mask, users[i].mask))
      {
        my_strncpy (users[i].pass, crypt (pass, SALT), ENC_PASS_SIZE);
        open_users ();
        fseek (ufd, users[i].index + strlen (users[i].mask) + 1, SEEK_SET);
        fputs (users[i].pass, ufd);
        close_users ();
        return;
      }
}

void
CUser::abs_set_pass (const char *abs_mask, const char *pass)
{
  int i;
  for (i = 0; i < user_num; i++)
    if (strcasecmp (abs_mask, users[i].mask) == 0)
      {
        my_strncpy (users[i].pass, crypt (pass, SALT), ENC_PASS_SIZE);
        open_users ();
        fseek (ufd, users[i].index + strlen (users[i].mask) + 1, SEEK_SET);
        fputs (users[i].pass, ufd);
        close_users ();
        return;
      }
}

void
CUser::set_msg (const char *mask, const char *msg)
{
  int i;
  i = mask_index (mask);
  abs_set_msg (users[i].mask, msg);
}

void
CUser::abs_set_msg (const char *abs_mask, const char *msg)
{
  int i = abs_mask_index (abs_mask);
  add_user (abs_mask, users[i].pass, users[i].level, msg);
}

void
CUser::abs_set_mask (const char *abs_mask, const char *new_abs_mask)
{
  int i = abs_mask_index (abs_mask);
  add_user (new_abs_mask, users[i].pass, users[i].level, users[i].msg);
  users[abs_mask_index (new_abs_mask)].id = users[i].id;
  del_user (abs_mask);
}

void
CUser::set_level (const char *mask, int level)
{
  int i = mask_index (mask);
  abs_set_level (users[i].mask, level);
}

void
CUser::abs_set_level (const char *abs_mask, int level)
{
  int i = abs_mask_index (abs_mask);
  add_user (abs_mask, users[i].pass, level, users[i].msg);
}

void
CUser::set_id (const char *mask, int id)
{
  int i;
  for (i = 0; i < user_num; i++)
    if (match_mask (mask, users[i].mask))
      {
        users[i].id = id;
        my_strncpy (users[i].abs_mask, mask, MASK_SIZE);
        return;
      }
}

int
CUser::mask_index (const char *mask)
{
  int i;
  for (i = 0; i < user_num; i++)
    if (match_mask (mask, users[i].mask))
      return i;
  return -1;
}

int
CUser::abs_mask_index(const char *abs_mask)
{
  int i;
  for (i = 0; i < user_num; i++)
    if (strcasecmp (abs_mask, users[i].mask) == 0)
      return i;
  return -1;
}

int
CUser::mask_level (const char *mask)
{
  int i;
  for (i = 0; i < user_num; i++)
    if (match_mask (mask, users[i].mask))
      {
        if (users[i].level < 0) 
          return users[i].level;
        else
          if (users[i].id && strcmp (mask, users[i].abs_mask) == 0)
            return users[i].level; 
          else
            return 1;
      }
  return 0;
}

int
CUser::mask_reallevel (const char *mask)
{
  int i;
  for (i = 0; i < user_num; i++)
    if (match_mask (mask, users[i].mask))
      return users[i].level;
  return 0;
}

