/*

  This module manages the !seen command. It's porpuse is to keep record of
when a certain nick was last seen by the bot. To acomplish this, it looks for
quit, part, kick, nick and join events and saves the nicks, along with a
timestamp, in the specified file. That file is NOT meant to be editable.
  So that's the only configuration option to place in the .conf:
"seen somefile.seen"
If this .seen file doesn't exist (which normally happens when the bot is being
configured for the first time), it will be automaticaly created.
Put one of those for each server, because mbot keeps servers separated.

*/

#include "mbot.h"

#define SEEN_MAX 1000		// maximum nicks on seen list at the same time

#define FIELD_SIZE (NICK_SIZE+10+1)

#define DEST s->script->dest
#define BUF s->script->buf
#define SEND_TEXT s->script->send_text

/////////////////////
// class definition
/////////////////////

class CSeen {
public:
  CSeen (CServer *, const char *);
  ~CSeen (void);

  void setup_seen (void);
  int seen_index (const char *);
  time_t seen_time (const char *);
  void change_seen (int);
  void write_seen (const char *, int);
  void update_seen (const char *);

  char filename[FILE_SIZE+1];			// file where the database is
  struct seen_type {
    char nick[NICK_SIZE+1];			// nick
    time_t time;				// when was it last seen
  } seen[SEEN_MAX];
  int seen_num;					// number of nicks

private:

  CServer *s;					// server to which belongs
  FILE *sfd;					// file descriptor
  char buf[BUF_SIZE+1];
};

struct seen_type {
  CServer *server;
  CSeen *seen;
  seen_type *next;
};
struct seen_type *seen_list;

///////////////
// prototypes
///////////////

static void seen_cmd_seen (CServer *);
static void seen_add (CServer *, const char *);
static CSeen *server2seen (CServer *);
static void seen_event (CServer *);
static void seen_conf (CServer *, const char *);
static void seen_stop (CModule *);
static void seen_start (CModule *);

////////////////////
// class functions
////////////////////

CSeen::CSeen (CServer *server, const char *name)
{
  s = server;
  my_strncpy (filename, name, FILE_SIZE);
  seen_num = 0;
  sfd = fdopen (open (filename, O_RDWR | O_CREAT, S_IWUSR | S_IRUSR), "r+");
  if (sfd == NULL)
    {
      snprintf (buf, BUF_SIZE, "ERROR opening %s: %s\n", name,
                strerror (errno));
      s->write_botlog (buf);
      s->bot->irc_exit ();
    }
  setup_seen ();
}

CSeen::~CSeen (void)
{
  fclose (sfd);
}

// load file to memory
void
CSeen::setup_seen (void)
{
  seen_num = 0;
  int i;
  fseek (sfd, 0, SEEK_SET);

  while (!feof (sfd))
    {
      i = fread (buf, 1, FIELD_SIZE, sfd);
      if (i == 0)					// eof
        return;
      if (i != FIELD_SIZE)				// invalid field
        {
          snprintf (buf, BUF_SIZE,
                    "ERROR reading %s: invalid field (probably you must delete the file)\n",
                    filename);
          s->write_botlog (buf);
          s->bot->irc_exit ();
        }
      if (seen_num == SEEN_MAX)
        {
          snprintf (buf, BUF_SIZE, 
                    "ERROR reading %s: too many fields (probably you must delete the file)\n",
                    filename);
          s->write_botlog (buf);
          s->bot->irc_exit ();
        }
      my_strncpy (seen[seen_num].nick, buf, NICK_SIZE);
      strip_crlf (buf);
      seen[seen_num].time = atoi (buf + NICK_SIZE);
      seen_num++;
    }
}

// table index for that nick, -1 if nonexistent
int
CSeen::seen_index (const char *nick)
{
  for (int i = 0; i < seen_num; i++)
    if (strcasecmp (nick, seen[i].nick) == 0)
      return i;
  return -1;
}

// return time for nick, -1 if nonexistant
time_t
CSeen::seen_time (const char *nick)
{
  int i = seen_index (nick);
  if (i != -1)
    return seen[i].time;
  else
    return -1;
}

// update time for position i
void
CSeen::change_seen (int i)
{
  seen[i].time = s->time_now;
  fseek (sfd, FIELD_SIZE*i + NICK_SIZE, SEEK_SET);
  if (fprintf (sfd, "%010ld", (long int)seen[i].time) != 10)
    {
      snprintf (buf, BUF_SIZE, "ERROR writing to %s: %s\n", filename,
                strerror (errno));
      s->write_botlog (buf);
      s->bot->irc_exit ();
    }
}

// write nick to file and memory, with the current time, at position i
void
CSeen::write_seen (const char *nick, int i)
{
  char b[FIELD_SIZE+1];
  memset (b, 0, FIELD_SIZE + 1);
  my_strncpy (seen[i].nick, nick, NICK_SIZE);
  seen[i].time = s->time_now;
  strncpy (b, nick, NICK_SIZE);
  sprintf (b + NICK_SIZE, "%010ld\n", (long int)seen[i].time);
  fseek (sfd, FIELD_SIZE * i, SEEK_SET);
  if (fwrite (b, FIELD_SIZE, 1, sfd) != 1)
    {
      snprintf (buf, BUF_SIZE, "ERROR writing to %s: %s\n", filename,
                strerror (errno));
      s->write_botlog (buf);
      s->bot->irc_exit ();
    }
}

// update nick, change time or add it if nonexistant, possibly overwriting another
void
CSeen::update_seen (const char *nick)
{
  int i = seen_index (nick);
  if (i == -1)			// doesn't exist
    {
      int added;
      if (seen_num == SEEN_MAX)			// if on the limit
        {
          time_t t = s->time_now;
          i = 0;
          for (int i2 = 0; i2 < seen_num; i2++)		// seach the oldest
            if (seen[i2].time < t)
              {
                i = i2;
                t = seen[i2].time;
              }
          added = 0;
        }
      else						// else add to the end
        {
          added = 1;
          i = seen_num;
        }
      write_seen (nick, i);				// write
      if (added)
        seen_num++;
    } else
      change_seen (i);					// change time
}

/////////////
// commands
/////////////

// !seen nick
static void
seen_cmd_seen (CServer *s)
{
  CSeen *seen = server2seen (s);
  if (seen == NULL)
    {
      SEND_TEXT (DEST, "This command is not available.");
      return;
    }
  strsplit (CMD[3] + 6, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][5] == ' ')
    {
      for (int i = 0; i < s->channel_num; i++)		// search in channels
        if (CHANNELS[i]->user_index(BUF[1]) != -1)
          {
            snprintf (BUF[2], MSG_SIZE, "%s is currently online.", BUF[1]);
            SEND_TEXT (DEST, BUF[2]);
            return;
          }
      time_t t = seen->seen_time(BUF[1]);		// search in seen
      if (t != -1)					// if it's there..
        {
          my_strncpy (BUF[2], asctime (localtime (&t)), MSG_SIZE);
          BUF[2][strlen (BUF[2]) - 1] = 0;
          snprintf (BUF[3], MSG_SIZE, "I last saw %s on %s.", BUF[1], BUF[2]);
          SEND_TEXT (DEST, BUF[3]);
        }
      else
        SEND_TEXT(DEST, "Humm, I don't remember that nick.");
    }
  else
    SEND_TEXT(DEST, "!seen == !seen nick");
}

////////////////////
// module managing
////////////////////

// add a seen to the list
static void
seen_add (CServer *s, const char *name)
{
  seen_type *buf = seen_list;
  seen_list = new seen_type;
  seen_list->server = s;
  seen_list->seen = new CSeen (s, name);
  seen_list->next = buf;
}

// return the seen for a given server, NULL if nonexistant
static CSeen *
server2seen (CServer *s)
{
  seen_type *buf = seen_list;
  while (buf != NULL)
    {
      if (buf->server == s)
        return buf->seen;
      buf = buf->next;
    }
  return NULL;
}

// look for join, quit, kick part and nick to update seen database
static void
seen_event (CServer *s)
{
  CSeen *seen = server2seen (s);
  if (seen != NULL)
    {
      if (strcmp (CMD[1], "JOIN") == 0
          || strcmp (CMD[1], "QUIT") == 0
          || strcmp (CMD[1], "KICK") == 0
          || strcmp (CMD[1], "PART") == 0)
        {
          mask2nick (CMD[0], BUF[0]);
          seen->update_seen (BUF[0]);
        }
      else
        if (strcmp (CMD[1], "NICK") == 0)
          {
            mask2nick (CMD[0], BUF[0]);
            seen->update_seen (BUF[0]);
            seen->update_seen (CMD[2]);
          }
    }
}

// look for who replies to update seen database
static void
seen_reply (CServer *s)
{
  CSeen *seen = server2seen (s);
  if (seen != NULL)
    if (strcmp (CMD[1], RPL_WHOREPLY) == 0)
      seen->update_seen (CMD[7]);
}

// configuration file's local parser
static void
seen_conf (CServer *s, const char *bufread)
{
  char buf[10][MSG_SIZE+1];

  strsplit (bufread, buf, 1);

  if (strcasecmp (buf[1], "seen") == 0)
    if (server2seen (s) == NULL)
      {
        seen_add (s, buf[2]);
        s->script->add_event (seen_event);
        s->script->add_reply (seen_reply);
        s->script->bind_cmd (seen_cmd_seen, 0, "!seen");
      }
    else
      s->bot->conf_error ("seen already defined for this server.");

}

// module termination
static void
seen_stop (CModule *m)
{
  seen_type *buf = seen_list, *buf2;
  while (buf != NULL)
    {
      buf->server->script->del_event (seen_event);
      buf->server->script->del_reply (seen_reply);
      buf->server->script->unbind_cmd ("!seen");
      if (buf->seen != NULL)
        delete buf->seen;
      buf2 = buf->next;
      delete buf;
      buf = buf2;
    }
}

// module initialization
static void
seen_start (CModule *m)
{
  seen_list = NULL;
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "seen",
  seen_start,
  seen_stop,
  seen_conf,
  NULL
};
