/*

  This module implements the user system commands. Although the mechanisms
these commands use are in the main source, there may be interest in not
activating them. They are:

!useradd mask level [msg]
Add a user with <mask> and <level>. If <msg> is not specified, turns off the
join msg. If <mask> already exists, overwrites the old one.

!userdel mask
Delete the user with <mask>.

!userpass mask pass
Change user's password with <mask> to <pass>.

!usermsg mask msg | <none>
Change user's join msg with <mask> to <msg>, or turns it off with "<none>".

!usermask mask newmask
Change user's mask from <mask> to <newmask>.

!userlevel mask level
Change user's access level with <mask> to <level>.

!id
Show the currently identified users with the bot.

!setpass password
Change to <password> the password of who executed it.

!setmsg msg | <none>
Change the join msg of who executed it to <msg>, or turn it off with "<none>".

!userlist
Show the list of registered users in the bot.

!pass password
Authentifies with the bot. Cannot be used inside a channel.

!level
Show the access level of who executed it.

*/

#include "mbot.h"

#define SOURCE s->script->source
#define DEST s->script->dest
#define BUF s->script->buf
#define SEND_TEXT s->script->send_text

///////////////
// prototypes
///////////////

static void main_user_cmd_useradd (CServer *);
static void main_user_cmd_userdel (CServer *);
static void main_user_cmd_userpass (CServer *);
static void main_user_cmd_usermsg (CServer *);
static void main_user_cmd_usermask (CServer *);
static void main_user_cmd_userlevel (CServer *);
static void main_user_cmd_id (CServer *);
static void main_user_cmd_userlist (CServer *);
static void main_user_cmd_setpass (CServer *);
static void main_user_cmd_setmsg (CServer *);
static void main_user_cmd_pass (CServer *);
static void main_user_cmd_level (CServer *);
static void main_user_conf (CServer *, const char *);
static void main_user_stop (CModule *);

/////////////
// commands
/////////////

// !useradd mask level [msg]
static void
main_user_cmd_useradd (CServer *s)
{
  strsplit(CMD[3] + 9, BUF, 2);
  if (BUF[2][0] != 0 && CMD[3][8] == ' ')
    {
      if (USERS->mask_level (CMD[0]) > USERS->mask_reallevel (BUF[1]) &&
          USERS->mask_level (CMD[0]) > atoi (BUF[2]))
        {
          if (USERS->user_num != USER_MAX)
            {
              if (USERS->add_user (strcut(BUF[1], MASK_SIZE),
                                   crypt (DEFAULT_USER_PASS, SALT),
                                   atoi (BUF[2]), BUF[3]))
                SEND_TEXT (DEST, "User added.");
              else
                SEND_TEXT (DEST, "Error adding user.");
            }
          else
            SEND_TEXT (DEST, "Maximum users reached.");
        } 
      else
        SEND_TEXT (DEST, "Not enough access level.");
    }
  else
    SEND_TEXT (DEST, "!useradd == !useradd mask level [msg]");
}

// !userdel mask
static void
main_user_cmd_userdel (CServer *s)
{
  strsplit (CMD[3] + 9, BUF, 1);
  if (BUF[1][0] != 0 && CMD[3][8] == ' ')
    {
      int i = USERS->abs_mask_index (BUF[1]);
      if (i != -1)
        {
          if (USERS->mask_level (CMD[0]) > USERS->users[i].level)
            {
              if (USERS->del_user (BUF[1]))
                SEND_TEXT (DEST, "User deleted.");
              else
                SEND_TEXT (DEST, "Error deleting user.");
            }
          else
            SEND_TEXT (DEST, "Not enough access level.");
        }
      else
        SEND_TEXT (DEST, "User does not exist.");
    }
  else
    SEND_TEXT (DEST, "!userdel == !userdel mask");
}

// !userpass mask pass
static void
main_user_cmd_userpass (CServer *s)
{
  strsplit (CMD[3] + 10, BUF, 2);
  if (BUF[2][0] != 0 && CMD[3][9] == ' ')
    {
      int i = USERS->abs_mask_index (BUF[1]);
      if (i != -1)
        {
          if (USERS->mask_level (CMD[0]) > USERS->users[i].level )
            {
              USERS->abs_set_pass (BUF[1], BUF[2]);
              snprintf (BUF[1], MSG_SIZE, "Password set to %s.", BUF[2]);
              SEND_TEXT (DEST, BUF[1]);
            }
          else
            SEND_TEXT (DEST, "Not enough access level.");
        }
      else
        SEND_TEXT (DEST, "User does not exist.");
    }
  else
    SEND_TEXT (DEST, "!userpass == !userpass mask pass");
}

// !usermsg mask msg|<none>
static void
main_user_cmd_usermsg (CServer *s)
{
  strsplit (CMD[3] + 9, BUF, 1);
  if (BUF[2][0] != 0 && CMD[3][8] == ' ')
    {
      int i = USERS->abs_mask_index (BUF[1]);
      if (i != -1)
        {
          if ( USERS->mask_level (CMD[0]) > USERS->users[i].level )
            {
              if (strcasecmp (BUF[2], "<none>") == 0)
                {
                  USERS->abs_set_msg (BUF[1], "");
                  snprintf (BUF[1], MSG_SIZE, "Message disabled.");
                }
              else
                {
                  USERS->abs_set_msg (BUF[1], BUF[2]);
                  snprintf (BUF[1], MSG_SIZE, "Message set to %s.", BUF[2]);
                }
              SEND_TEXT (DEST, BUF[1]);
            } 
          else
            SEND_TEXT (DEST, "Not enough access level.");
        }
      else
        SEND_TEXT (DEST, "User does not exist.");
    }
  else
    SEND_TEXT (DEST, "!usermsg == !usermsg mask msg|<none>");
}

// !usermask mask newmask
static void
main_user_cmd_usermask (CServer *s)
{
  strsplit (CMD[3] + 10, BUF, 2);
  if (BUF[2][0] != 0 && CMD[3][9] == ' ')
    {
      int i = USERS->abs_mask_index (BUF[1]);
      if (i != -1)
        {
          if (USERS->mask_level (CMD[0]) > USERS->users[i].level)
            {
              USERS->abs_set_mask (BUF[1], BUF[2]);
              snprintf (BUF[1], MSG_SIZE, "Mask set to %s.", BUF[2]);
              SEND_TEXT (DEST, BUF[1]);
            }
          else
            SEND_TEXT (DEST, "Not enough access level.");
        }
      else
        SEND_TEXT (DEST, "User does not exist.");
    }
  else
    SEND_TEXT (DEST, "!usermask == !usermask mask newmask");
}

// !userlevel mask level
static void
main_user_cmd_userlevel (CServer *s)
{
  strsplit (CMD[3] + 11, BUF, 2);
  if (BUF[2][0] != 0 && CMD[3][10] == ' ')
    {
      int i = USERS->abs_mask_index (BUF[1]);
      if (i != -1)
        {
          if (atoi (BUF[2]) >= LEVEL_MIN && atoi (BUF[2]) <= LEVEL_MAX)
            {
              if (USERS->mask_level (CMD[0]) > USERS->users[i].level
                  && USERS->mask_level (CMD[0]) > atoi (BUF[2]))
                {
                  USERS->abs_set_level (BUF[1], atoi (BUF[2]));
                  snprintf (BUF[1], MSG_SIZE, "Level set to %s.", BUF[2]);
                  SEND_TEXT (DEST, BUF[1]);
                }
              else
                SEND_TEXT (DEST, "Not enough access level.");
            }
          else
            SEND_TEXT (DEST, "Invalid level.");
        }
      else
        SEND_TEXT (DEST, "User does not exist.");
    }
  else
    SEND_TEXT (DEST, "!userlevel == !userlevel mask level");
}

// !id
static void
main_user_cmd_id (CServer *s)
{
  if (CMD[3][3] == 0)
    {
      my_strncpy (BUF[1], "Identified users:", MSG_SIZE);
      for (int i = 0; i < USERS->user_num; i++)
        if (USERS->users[i].id
            && strlen (BUF[1]) + strlen (USERS->users[i].abs_mask) + 1 < MSG_SIZE)
          sprintf (BUF[1] + strlen (BUF[1]), " %s", USERS->users[i].abs_mask);
      SEND_TEXT (DEST, BUF[1]);
    }
}

// !userlist
static void
main_user_cmd_userlist (CServer *s)
{
  if (CMD[3][9] == 0)
    {
      bool has_user;
      SEND_TEXT (SOURCE, "Users:");
      for (int i = LEVEL_MAX; i >= LEVEL_MIN; i--)
        {
          has_user = 0;
          snprintf (BUF[1], MSG_SIZE, "Level %d:", i);
          for (int i2 = 0; i2 < USERS->user_num; i2++)
            if (USERS->users[i2].level == i)
              if (strlen (BUF[1]) + strlen (USERS->users[i2].mask) > MSG_SIZE)
                {
                  SEND_TEXT (SOURCE, BUF[1]);
                  my_strncpy (BUF[1], USERS->users[i2].mask, MSG_SIZE);
                }
              else
                {
                  snprintf (BUF[1] + strlen (BUF[1]),
                            MSG_SIZE - strlen (BUF[1]),
                            " %s", USERS->users[i2].mask);
                  has_user = 1;
                }
          if (has_user)
            SEND_TEXT (SOURCE, BUF[1]);
        }
    }
}

// !setpass password
static void
main_user_cmd_setpass (CServer *s)
{
  if (DEST[0] == '#')
    SEND_TEXT (DEST, "Do not use this command on a channel.");
  else
    {
      strsplit (CMD[3] + 9, BUF, 1);
      if (BUF[1][0] != 0 && CMD[3][8] == ' ')
        {
          USERS->set_pass (CMD[0], BUF[1]);
          snprintf (BUF[2], MSG_SIZE,
                    "Password set to %s - remember it for later use.",
                    BUF[1]);
          SEND_TEXT (DEST, BUF[2]);
        }
      else
        SEND_TEXT (DEST, "!setpass == !setpass password");
    }
}

// !setmsg msg | <none>
static void
main_user_cmd_setmsg (CServer *s)
{
  strsplit (CMD[3] + 8, BUF, 0);
  if (BUF[1][0] != 0 && CMD[3][7] == ' ')
    {
      if (strcasecmp (BUF[1], "<none>") == 0)
        {
          USERS->set_msg (CMD[0], "");
          snprintf (BUF[2], MSG_SIZE, "Message disabled.");
        }
      else
        {
          USERS->set_msg (CMD[0], BUF[1]);
          snprintf (BUF[2], MSG_SIZE, "Message set to %s.", BUF[1]);
        }
      SEND_TEXT (DEST, BUF[2]);
    } else
      SEND_TEXT (DEST, "!setmsg == !setmsg msg | <none>");
}

// !pass password
static void
main_user_cmd_pass (CServer *s)
{
  if (DEST[0] == '#')
    SEND_TEXT (DEST, "Do not use this command on a channel.");
  else
    {
      strsplit (CMD[3] + 6, BUF, 1);
      if (BUF[1][0] != 0 && CMD[3][5] == ' ')
        {
          if (USERS->check_pass (CMD[0], BUF[1]))
            {
              USERS->set_id (CMD[0], 1);
              SEND_TEXT (DEST, "Password correct.");
            }
          else
            SEND_TEXT (DEST, "Password incorrect.");
        }
      else
        SEND_TEXT (DEST, "!pass == !pass password");
    }
}

// !level
static void
main_user_cmd_level (CServer *s)
{
  if (CMD[3][6] == 0)
    {
      snprintf (BUF[1], MSG_SIZE, "Your access level is %d",
                USERS->mask_reallevel(CMD[0]));
      if (USERS->mask_reallevel (CMD[0]) >= 1)
        {
          if (USERS->mask_level(CMD[0]) > 1)
            snprintf (BUF[1] + strlen(BUF[1]), MSG_SIZE - strlen(BUF[1]),
                      ", identified");
          else
            snprintf (BUF[1] + strlen(BUF[1]), MSG_SIZE - strlen(BUF[1]),
                      ", unidentified");
        }
      strcat (BUF[1], ".");
      SEND_TEXT (DEST, BUF[1]);
    }
}

////////////////////
// module managing
////////////////////

// configuration file's local parser
static void
main_user_conf (CServer *s, const char *bufread)
{
  char buf[10][MSG_SIZE+1];

  strsplit (bufread, buf, 1);

  // add these when a userfile is specified
  if (strcasecmp (buf[1], "users") == 0)
    {
      s->script->bind_cmd (main_user_cmd_useradd, 9, "!useradd");
      s->script->bind_cmd (main_user_cmd_userdel, 9, "!userdel");
      s->script->bind_cmd (main_user_cmd_userpass, 9, "!userpass");
      s->script->bind_cmd (main_user_cmd_usermsg, 9, "!usermsg");
      s->script->bind_cmd (main_user_cmd_usermask, 9, "!usermask");
      s->script->bind_cmd (main_user_cmd_userlevel, 9, "!userlevel");
      s->script->bind_cmd (main_user_cmd_id, 7, "!id");
      s->script->bind_cmd (main_user_cmd_userlist, 3, "!userlist");
      s->script->bind_cmd (main_user_cmd_setpass, 2, "!setpass");
      s->script->bind_cmd (main_user_cmd_setmsg, 2, "!setmsg");
      s->script->bind_cmd (main_user_cmd_pass, 0, "!pass");
      s->script->bind_cmd (main_user_cmd_level, 0, "!level");
    }
}

// module termination
static void
main_user_stop (CModule *m)
{
  // try to remove from all servers, even if it's not there (we don't know)
  for (int i = 0; i < m->b->server_num; i++)
    {
      m->b->servers[i]->script->unbind_cmd ("!useradd");
      m->b->servers[i]->script->unbind_cmd ("!userdel");
      m->b->servers[i]->script->unbind_cmd ("!userpass");
      m->b->servers[i]->script->unbind_cmd ("!usermsg");
      m->b->servers[i]->script->unbind_cmd ("!usermask");
      m->b->servers[i]->script->unbind_cmd ("!userlevel");
      m->b->servers[i]->script->unbind_cmd ("!id");
      m->b->servers[i]->script->unbind_cmd ("!userlist");
      m->b->servers[i]->script->unbind_cmd ("!setpass");
      m->b->servers[i]->script->unbind_cmd ("!setmsg");
      m->b->servers[i]->script->unbind_cmd ("!pass");
      m->b->servers[i]->script->unbind_cmd ("!level");
    }
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "mainuser",
  NULL,
  main_user_stop,
  main_user_conf,
  NULL
};
