/*

  This module implements the standard irc and mbot commands. Although the
mechanisms these commands use are in the main source, there may be interest
in not activating them. They are:

!away [msg]
Put the bot away with <msg>, or without parameters, remove the away.

!nick nick
Change the bot's nick to <nick>.

!unbind !command
Disable the given <!command>. Note that it can only be recovered by restarting
the bot or reloading that command's module.

!join #channel
Add <#channel> to the channel list.

!part #channel
Remove <#channel> from the channel list.

!status
Show some bot status.

!topic [#channel] text
Change the topic in the current or in the specified <#channel>, to <text>

!opme
Give op to who executed the command. Can only be used inside a channel.

!deopme
Take op from who executed the command. Can only be used inside a channel.

!op nick [#channel]
Give op to <nick> on the specified <#channel>.

!deop nick [#channel]
Take op from <nick> on the specified <#channel>.

!kick nick [#channel] [reason]
Kick <nick> on the specified <#channel>, with <reason>.

!ban nick | mask [#channel]
Ban <nick> or <mask> on the specified <#channel>.

!unban nick | mask [#channel]
Unban <nick> or <mask> on the specified <#channel>.

!bk nick [#channel]
Deop, ban and kick <nick> on the specified <#channel>.

!invite nick [#channel]
Invite <nick> to <#channel>.

!voice nick [#channel]
Give voice to <nick> on the specified <#channel>.

!devoice nick [#channel]
Take voice from <nick> on the specified <#channel>.

!say [#channel | $nick] text
Send <text> to a <#channel> or <nick>.

!help
Show the command list, up to the level of who executed it.

*/

#include "mbot.h"

#define SOURCE s->script->source
#define DEST s->script->dest
#define BUF s->script->buf
#define SEND_TEXT s->script->send_text

///////////////
// prototypes
///////////////

static void main_cmd_away (CServer *);
static void main_cmd_nick (CServer *);
static void main_cmd_unbind (CServer *);
static void main_cmd_join (CServer *);
static void main_cmd_part (CServer *);
static void main_cmd_status (CServer *);
static void main_cmd_topic (CServer *);
static void main_cmd_opme (CServer *);
static void main_cmd_deopme (CServer *);
static void main_cmd_op (CServer *);
static void main_cmd_deop (CServer *);
static void main_cmd_kick (CServer *);
static void main_cmd_ban (CServer *);
static void main_cmd_unban (CServer *);
static void main_cmd_bk (CServer *);
static void main_cmd_invite (CServer *);
static void main_cmd_voice (CServer *);
static void main_cmd_devoice (CServer *);
static void main_cmd_say (CServer *);
static void main_cmd_help (CServer *);
static void main_conf (CServer *, const char *);
static void main_stop (CModule *);

/////////////
// commands
/////////////

// !away [msg]
static void
main_cmd_away (CServer *s)
{
  strsplit (CMD[3] + 6, BUF, 0);
  my_strncpy (s->away, BUF[1], AWAY_SIZE);
  s->irc_away (s->away);
}

// !nick nick
static void
main_cmd_nick (CServer *s)
{
  strsplit (CMD[3] + 6, BUF, 1);
  if (BUF[1][0] != 0 && CMD[3][5] == ' ')
    s->irc_nick (strcut (BUF[1], NICK_SIZE));
  else
    SEND_TEXT(DEST, "!nick == !nick nick");
}

// !unbind !command
static void
main_cmd_unbind (CServer *s)
{
  strsplit (CMD[3] + 8, BUF, 1);
  if (BUF[1][0] != 0 && CMD[3][7] == ' ')
    {
      if (s->script->unbind_cmd (BUF[1]))
        SEND_TEXT (DEST, "Command unbinded.");
      else
        SEND_TEXT (DEST, "That command does not exist.");
    }
  else
    SEND_TEXT (DEST, "!unbind == !unbind !command");
}

// !join #channel
static void
main_cmd_join (CServer *s)
{
  strsplit (CMD[3]+6, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][5] == ' ')
    {
      strcut (BUF[1], CHANNEL_SIZE);
      int chan_num = CHANNEL_INDEX (BUF[1]);
      if (chan_num == -1)		// if not already in the channel
        {
          if (s->channel_add (BUF[1], BUF[2]) == 1)		// add it
            {
              SEND_TEXT (DEST, "Maximum channels reached.");
              return;
            }
          chan_num = CHANNEL_INDEX (BUF[1]);	// search it's position again
        }
      else
        SEND_TEXT (DEST, "I'm already on that channel.");
      CHANNELS[chan_num]->irc_join ();		// join it
    }
  else
    SEND_TEXT (DEST, "!join == !join #channel [key]");
}

// !part #channel
static void
main_cmd_part (CServer *s)
{
  strsplit(CMD[3] + 6, BUF, 1);
  if (BUF[1][0] != 0 && CMD[3][5] == ' ')
    {
      int i = CHANNEL_INDEX (BUF[1]);
      if (i != -1)
        {
          CHANNELS[i]->irc_part ();
          s->channel_del (BUF[1]);
        }
      else
        SEND_TEXT (DEST, "Channel does not exist.");
    }
  else
    SEND_TEXT (DEST, "!part == !part #channel");
}

// !status
static void
main_cmd_status (CServer *s)
{
  if (CMD[3][7] == 0)
    {
      snprintf (BUF[1], MSG_SIZE,
                "Status: %s - %s:%d - %s!%s (%s) - %d/%d users, %d/%d channels, %d bytes read, %d written. Up since %s.",
                VERSION_STRING, s->host_current->host, s->host_current->port,
                s->nick, s->user, s->name, (USERS == NULL?0:USERS->user_num),
                USER_MAX, s->channel_num, CHANNEL_MAX, s->bytesin, s->bytesout,
                get_asctime (s->uptime, BUF[2], MSG_SIZE));
      SEND_TEXT (DEST, BUF[1]);
    }
}

// !topic [#channel] text
static void
main_cmd_topic (CServer *s)
{
  strsplit (CMD[3] + 7, BUF, 0);
  if (BUF[1][0] != 0 && CMD[3][6] == ' ')
    {
      int i;
      if (BUF[1][0] != '#')
        {
          i = CHANNEL_INDEX (CMD[2]);
          if (i != -1)
            CHANNELS[i]->irc_topic (BUF[1]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
      else
        {
          strsplit (CMD[3] + 7, BUF, 1);
          i = CHANNEL_INDEX (BUF[1]);
          if (i != -1)
            CHANNELS[i]->irc_topic (BUF[2]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
    }
  else
    SEND_TEXT (DEST, "!topic == !topic [#channel] text");
}

// !opme
static void
main_cmd_opme (CServer *s)
{
  if (CMD[3][5] == 0)
    {
      if (CMD[2][0] == '#')
        {
          mask2nick (CMD[0], BUF[1]);
          int i = CHANNEL_INDEX (CMD[2]);
          if (i != -1)
            CHANNELS[i]->irc_op (BUF[1]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
      else
        SEND_TEXT (DEST, "This command only works inside channels.");
    }
}

// !deopme
static void
main_cmd_deopme (CServer *s)
{
  if (CMD[3][7] == 0)
    {
      if (CMD[2][0] == '#')
        {
          mask2nick (CMD[0], BUF[1]);
          int i = CHANNEL_INDEX (CMD[2]);
          if (i != -1)		// should never fail
            CHANNELS[i]->irc_deop (BUF[1]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
      else
        SEND_TEXT (DEST, "This command only works inside channels.");
    }
}

// !op nick [#channel]
static void
main_cmd_op (CServer *s)
{
  strsplit (CMD[3] + 4, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][3] == ' ')
    {
      if (BUF[2][0] != '#')
        strcpy (BUF[2], CMD[2]);
      int i = CHANNEL_INDEX (BUF[2]);
      if (i != -1)
        CHANNELS[i]->irc_op (BUF[1]);
      else
        SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
    }
  else
    SEND_TEXT (DEST, "!op == !op nick [#channel]");
}

// !deop nick [#channel]
static void
main_cmd_deop (CServer *s)
{
  strsplit (CMD[3] + 6, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][5] == ' ')
    {
      if (BUF[2][0] != '#')
        my_strncpy (BUF[2], CMD[2], MSG_SIZE);
      if (strcasecmp (BUF[1], s->nick) != 0)
        {
          int i = CHANNEL_INDEX (BUF[2]);
          if (i != -1)
            CHANNELS[i]->irc_deop (BUF[1]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
      else
        SEND_TEXT(DEST, SCRIPT_HURT_BOT);
    }
  else
    SEND_TEXT(DEST, "!deop == !deop nick [#channel]");
}

// !kick nick [#channel] [reason]
static void
main_cmd_kick (CServer *s)
{
  strsplit (CMD[3] + 6, BUF, 1);
  if (BUF[1][0] != 0 && CMD[3][5] == ' ')
    {
      if (strcasecmp (BUF[1], s->nick) == 0)	// doesn't kick the bot
        {
          SEND_TEXT (DEST, SCRIPT_HURT_BOT);
          return;
        }
      int i;
      if (BUF[2][0] != '#')
        {
          i = CHANNEL_INDEX(CMD[2]);
          if (i != -1)
            CHANNELS[i]->irc_kick (BUF[1], BUF[2]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
      else
        {
          strsplit (CMD[3] + 6, BUF, 2);
          i = CHANNEL_INDEX (BUF[2]);
          if (i != -1)
            CHANNELS[i]->irc_kick (BUF[1], BUF[3]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
    }
  else
    SEND_TEXT (DEST, "!kick == !kick nick [#channel] [reason]");
}

// !ban nick | mask [#channel]
static void
main_cmd_ban (CServer *s)
{
  strsplit (CMD[3] + 5, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][4] == ' ')
    {
      if (BUF[2][0] != '#')
        strcpy (BUF[2], CMD[2]);
      int i;
      for (i = 0; i < (int)strlen (BUF[1]); i++)
        if (BUF[1][i] == '@')			// check if it's a mask
          {
            i = CHANNEL_INDEX (BUF[2]);
            if (i != -1)
              CHANNELS[i]->irc_ban_mask (BUF[1]);
            else
              SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
            return;
          }
      if (strcasecmp (BUF[1], s->nick) != 0)	// doesn't ban the bot
        {
          i = CHANNEL_INDEX (BUF[2]);
          if (i != -1)
            CHANNELS[i]->irc_ban_nick (BUF[1]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
      else
        SEND_TEXT(DEST, SCRIPT_HURT_BOT);
    }
  else
    SEND_TEXT(DEST, "!ban == !ban nick|mask [#channel]");
}

// !unban nick | mask [#channel]
static void
main_cmd_unban (CServer *s)
{
  strsplit (CMD[3] + 7, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][6] == ' ')
    {
      if (BUF[2][0] != '#')
        strcpy (BUF[2], CMD[2]);
      int i;
      for (i = 0; i < (int)strlen (BUF[1]); i++)
        if (BUF[1][i] == '@')			// check if it's a mask
          {
            i = CHANNEL_INDEX (BUF[2]);
            if (i != -1)
              CHANNELS[i]->irc_unban_mask (BUF[1]);
            else
              SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
            return;
          }
      i = CHANNEL_INDEX (BUF[2]);
      if (i != -1)
        CHANNELS[i]->irc_unban_nick (BUF[1]);
      else
        SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
    }
  else
    SEND_TEXT (DEST, "!unban == !unban nick|mask [#channel]");
}

// !bk nick [#channel]
static void
main_cmd_bk (CServer *s)
{
  strsplit (CMD[3] + 4, BUF, 1);
  if (BUF[1][0] != 0 && CMD[3][3] == ' ')
    {
      if (strcasecmp (BUF[1], s->nick) != 0)		// doesn't bk the bot
        {
          if (BUF[2][0] != '#')		// if the channel was not specified
            {
              strcpy (BUF[3], BUF[2]);				// reason
              strcpy (BUF[2], CMD[2]);				// channel
            }
          else
            strsplit(CMD[3]+4, BUF, 2);
          int i = CHANNEL_INDEX (BUF[2]);
          if (i != -1)			// if the channel exists
            {
              int i2 = CHANNELS[i]->user_index (BUF[1]);
              if (i2 != -1)		// and the user too
                {
                  // make mask
                  make_generic_mask (CHANNELS[i]->users[i2]->mask, BUF[4]);
                  // deop/ban
                  CHANNELS[i]->irc_deopban (BUF[1], BUF[4]);
                  // kick
                  CHANNELS[i]->irc_kick (BUF[1], BUF[3]);
                }
            }
          else
            SEND_TEXT(DEST, SCRIPT_INVALID_CHAN);
      } else
        SEND_TEXT(DEST, SCRIPT_HURT_BOT);
    }
  else
    SEND_TEXT(DEST, "!bk == !bk nick [#channel] [reason]");
}

// !invite nick [#channel]
static void
main_cmd_invite (CServer *s)
{
  strsplit (CMD[3] + 8, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][7] == ' ')
    {
      if (BUF[2][0] != '#')
        strcpy (BUF[2], CMD[2]);
      int i = CHANNEL_INDEX (BUF[2]);
      if (i != -1)
        CHANNELS[i]->irc_invite (BUF[1]);
      else
        SEND_TEXT(DEST, SCRIPT_INVALID_CHAN);
    }
  else
    SEND_TEXT (DEST, "!invite == !invite nick [#channel]");
}

// !voice nick [#channel] 
static void
main_cmd_voice (CServer *s)
{
  strsplit (CMD[3] + 7, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][6] == ' ')
    {
      if (BUF[2][0] != '#')
        my_strncpy (BUF[2], CMD[2], MSG_SIZE);
      int i = CHANNEL_INDEX (BUF[2]);
      if (i != -1)
        CHANNELS[i]->irc_voice (BUF[1]);
      else
        SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
    }
  else
    SEND_TEXT (DEST, "!voice == !voice nick [#channel]");
}

// !devoice
static void
main_cmd_devoice (CServer *s)
{
  strsplit (CMD[3] + 9, BUF, 2);
  if (BUF[1][0] != 0 && CMD[3][8] == ' ')
    {
      if (BUF[2][0] != '#')
        my_strncpy (BUF[2], CMD[2], MSG_SIZE);
      if (strcasecmp (BUF[1], s->nick) != 0)
        {
          int i = CHANNEL_INDEX (BUF[2]);
          if (i != -1)
            CHANNELS[i]->irc_devoice (BUF[1]);
          else
            SEND_TEXT (DEST, SCRIPT_INVALID_CHAN);
        }
      else
        SEND_TEXT (DEST, SCRIPT_HURT_BOT);
    }
  else
    SEND_TEXT(DEST, "!devoice == !devoice nick [#channel]");
}

// !say [#channel | $nick] text
static void
main_cmd_say (CServer *s)
{
  strsplit (CMD[3] + 5, BUF, 0);
  if (BUF[1][0] != 0 && CMD[3][4] == ' ')
    {
      if (BUF[1][0] == '#')
        {
          int i = 0;
          while (BUF[1][i] != ' ' && BUF[1][i] != 0)
            i++;
          my_strncpy (DEST, BUF[1], i);
          my_strncpy (BUF[2], BUF[1] + i + num_spaces (BUF[1]+i), MSG_SIZE);
          my_strncpy (BUF[1], BUF[2], MSG_SIZE);
        }
      if (BUF[1][0] == '$')
        {
          int i = 0;
          while (BUF[1][i] != ' ' && BUF[1][i] != 0)
            i++;
          my_strncpy (DEST, BUF[1], i);
          my_strncpy (DEST, DEST+1, CHANNEL_SIZE);
          if (strcasecmp (NICKSERV, DEST) == 0
              || strcasecmp (CHANSERV, DEST) == 0
              || strcasecmp (MEMOSERV, DEST) == 0
              || strcasecmp (OPERSERV, DEST) == 0)
            {
              SEND_TEXT (SOURCE, "Cannot send to IRC Services.");
              return;
            }
          my_strncpy (BUF[2], BUF[1] + i + num_spaces (BUF[1] + i), MSG_SIZE);
          my_strncpy (BUF[1], BUF[2], MSG_SIZE);
        }
      SEND_TEXT (DEST, BUF[1]);
    }
  else
    SEND_TEXT (DEST, "!say == !say [#channel | $nick] text");
}

// !help
static void
main_cmd_help (CServer *s)
{
  if (CMD[3][5] == 0)
    {
      int level = (USERS == NULL ? 0 : USERS->mask_level (CMD[0]));
      bool has_user;
      SEND_TEXT (SOURCE,"Commands:");
      for (int i = level; i >= 0; i--)
        {
          CScript::cmd_type *c = s->script->cmds[i];
          snprintf (BUF[1], MSG_SIZE, "Level %d:", i);
          has_user = 0;
          while (c != NULL)
            {
              has_user = 1;
              snprintf (BUF[1] + strlen (BUF[1]), MSG_SIZE, " %s", c->cmd);
              c = c->next;
            }
          if (has_user)
            SEND_TEXT (SOURCE, BUF[1]);
        }
    }
}

////////////////////
// module managing
////////////////////

// configuration file's local parser
static void
main_conf (CServer *s, const char *bufread)
{
  char buf[10][MSG_SIZE+1];

  strsplit (bufread, buf, 1);

  // add these commands in each new server
  if (strcasecmp (buf[1], "server") == 0
      || strcasecmp (buf[1], "bot") == 0)
    {
      s->script->bind_cmd (main_cmd_away, 10, "!away");
      s->script->bind_cmd (main_cmd_nick, 10, "!nick");
      s->script->bind_cmd (main_cmd_unbind, 10, "!unbind");
      s->script->bind_cmd (main_cmd_join, 9, "!join");
      s->script->bind_cmd (main_cmd_part, 9, "!part");
      s->script->bind_cmd (main_cmd_status, 7, "!status");
      s->script->bind_cmd (main_cmd_topic, 7, "!topic");
      s->script->bind_cmd (main_cmd_opme, 5, "!opme");
      s->script->bind_cmd (main_cmd_deopme, 5, "!deopme");
      s->script->bind_cmd (main_cmd_op, 5, "!op");
      s->script->bind_cmd (main_cmd_deop, 5, "!deop");
      s->script->bind_cmd (main_cmd_kick, 5, "!kick");
      s->script->bind_cmd (main_cmd_ban, 5, "!ban");
      s->script->bind_cmd (main_cmd_unban, 5, "!unban");
      s->script->bind_cmd (main_cmd_bk, 5, "!bk");
      s->script->bind_cmd (main_cmd_invite, 5, "!invite");
      s->script->bind_cmd (main_cmd_voice, 3, "!voice");
      s->script->bind_cmd (main_cmd_devoice, 3, "!devoice");
      s->script->bind_cmd (main_cmd_say, 2, "!say");
      s->script->bind_cmd (main_cmd_help, 0, "!help");
    }
}

// module termination
static void
main_stop (CModule *m)
{
  // try to remove from all servers, even if it's not there (we don't know)
  for (int i = 0; i < m->b->server_num; i++)
    {
      m->b->servers[i]->script->unbind_cmd ("!away");
      m->b->servers[i]->script->unbind_cmd ("!nick");
      m->b->servers[i]->script->unbind_cmd ("!unbind");
      m->b->servers[i]->script->unbind_cmd ("!join");
      m->b->servers[i]->script->unbind_cmd ("!part");
      m->b->servers[i]->script->unbind_cmd ("!status");
      m->b->servers[i]->script->unbind_cmd ("!topic");
      m->b->servers[i]->script->unbind_cmd ("!opme");
      m->b->servers[i]->script->unbind_cmd ("!deopme");
      m->b->servers[i]->script->unbind_cmd ("!op");
      m->b->servers[i]->script->unbind_cmd ("!deop");
      m->b->servers[i]->script->unbind_cmd ("!kick");
      m->b->servers[i]->script->unbind_cmd ("!ban");
      m->b->servers[i]->script->unbind_cmd ("!unban");
      m->b->servers[i]->script->unbind_cmd ("!bk");
      m->b->servers[i]->script->unbind_cmd ("!invite");
      m->b->servers[i]->script->unbind_cmd ("!voice");
      m->b->servers[i]->script->unbind_cmd ("!devoice");
      m->b->servers[i]->script->unbind_cmd ("!say");
      m->b->servers[i]->script->unbind_cmd ("!help");
    }
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "main",
  NULL,
  main_stop,
  main_conf,
  NULL
};
