/*

  This module logs channels and/or privmsgs from certain nicks to the bot.
Logs are renewed at 6 a.m. everyday by default. Configuration options:

To log a channel:
"log #channel channel.log channel.old.log"
where channel.log is where to log, and channel.old.log is where to put the
last refreshed log.

To log privmsgs from a nick:
"log somenick somenick.log somenick.old.log"

If you want to log ALL privmsgs with the bot:
"log <privmsg> pvt.log pvt.old.log"

This is to send by email the specified log, when it is renewed:
"logmail #channel logs@example.org"
Note that "#channel" can also be "somenick" or "<privmsg>".

With this option you can force mbot to flush the log's file, thus keeping it
always updated, but slowing a bit the bot/machine:
"logflush #channel"
Again, "#channel" can also be "somenick" or "<privmsg>".

*/

#include "mbot.h"

#define LOG_MAX 10			// logs per server
#define LOG_HOUR 6			// renew the logs everyday at 6am
#define LOG_FLUSH 0			// don't flush the logs by default
#define LOG_SIZE 1048576		// each log's maximum size in bytes

#ifdef ECHO_PATH
#  ifdef CAT_PATH
#    ifdef SENDMAIL_PATH
#      define USE_MAIL			// all those are needed to send mail
#    endif
#  endif
#endif

#define SOURCE s->script->source
#define DEST s->script->dest
#define BUF s->script->buf

/////////////////////
// class definition
/////////////////////

class CLog {
public:
  CLog (CServer *, const char *, const char *, const char *);
  ~CLog (void);

  bool start_log (void);
  void stop_log (void);
  void add_log (const char *);

  char filename[FILE_SIZE+1];			// current log file
  char old_filename[FILE_SIZE+1];		// old log file
  char what[CHANNEL_SIZE+1];			// channel or nick to log
  char mail[MAIL_SIZE+1];			// mail where to send the log
  bool log_flush;			// flush the file after each write?

private:

  void change_log (void);
  u_char get_hour (void);

  CServer *s;
  FILE *lfd;					// file descriptor
  u_char hour;					// last made log's time
  char st_time[8];
  char buf[10][MSG_SIZE+1];

};

struct log_type {
  CServer *server;
  CLog *log_pvt;				// logs all pvt
  CLog *logs[LOG_MAX];				// log list
  int log_num;					// number of logs
  log_type *next;
};
struct log_type *log_list;

///////////////
// prototypes
///////////////

static void log_add (CServer *);
static log_type *server2log (CServer *);
static CLog *log_list2log (log_type *, const char *);
static void log_event (CServer *);
static void log_conf (CServer *, const char *);
static void log_stop (CModule *);
static void log_start (CModule *);

////////////////////
// class functions
////////////////////

CLog::CLog (CServer *server, const char *what_to_log, const char *name,
            const char *old_name)
{
  s = server;
  my_strncpy (what, what_to_log, CHANNEL_SIZE);
  my_strncpy (filename, name, FILE_SIZE);
  my_strncpy (old_filename, old_name, FILE_SIZE);
  mail[0] = 0;
  log_flush = LOG_FLUSH;
}

CLog::~CLog (void)
{
  add_log ("Log stopped.");
  stop_log ();
} 

// return 0 if can't open logfile
bool
CLog::start_log (void)
{
  lfd = fopen (filename, "a+");
  if (lfd == NULL)
    return 0;
  hour = get_hour ();
  return 1;
}

void
CLog::stop_log (void)
{
  fclose (lfd);
}

// add time and msg to the log
void
CLog::add_log (const char *msg)
{
  change_log ();
  string_time (s->time_now, st_time);
  fprintf (lfd, "%s %s\n", st_time, msg);
  if (log_flush)
    fflush (lfd);
  hour = get_hour ();
}

// renew log if it's time or if the file is too big
void
CLog::change_log (void)
{
  if ((hour < LOG_HOUR && get_hour () >= LOG_HOUR)
      || lseek (fileno (lfd), 0, SEEK_CUR) >= LOG_SIZE)
    {
      stop_log ();
      rename (filename, old_filename);
#ifdef USE_MAIL
      if (mail[0] != 0)			// send log by mail, if specified
        {
          snprintf (buf[1], MSG_SIZE, 
                    "(%s && %s %s) | %s %s &\n", 
//                    "(%s && %s %s) | %s %s &\n", 
                    ECHO_PATH, CAT_PATH, old_filename, SENDMAIL_PATH, mail);
          system (buf[1]);
        }
#endif
      start_log ();
      fprintf (lfd, "%s\n", get_asctime(s->time_now, buf[1], MSG_SIZE));
    }
}

// return current time
u_char
CLog::get_hour (void)
{
  tm time1;
  time_t time2 = s->time_now;
  time1 = *localtime (&time2);
  return time1.tm_hour;
}

////////////////////
// module managing
////////////////////

// add a server to the logs list
static
void log_add (CServer *s)
{
  log_type *log_buf = log_list;
  log_list = new log_type;
  log_list->server = s;
  log_list->log_pvt = NULL;
  for (int i = 0; i < LOG_MAX; i++)
    log_list->logs[i] = NULL;
  log_list->log_num = 0;
  log_list->next = log_buf;
}

// return the log struct for a given server, NULL if nonexistant
static
log_type *server2log (CServer *s)
{
  log_type *buf = log_list;
  while (buf != NULL)
    {
      if (buf->server == s)
        return buf;
      buf = buf->next;
    }
  return NULL;
}

// return a log object for a given log list, NULL if nonexistant
static CLog *
log_list2log (log_type *list, const char *what)
{
  for (int i = 0; i < list->log_num; i++)
    if (strcasecmp (list->logs[i]->what, what) == 0)
      return list->logs[i];
  return NULL;
}

// events watcher, to log most of them
static void
log_event (CServer *s)
{
  char buf[10][MSG_SIZE+1];
  log_type *log = server2log (s);
  if (log == NULL)
    return;

  int i;
  CLog *l;

  if (strcmp (CMD[1], "JOIN") == 0)
    {
      l = log_list2log (log, CMD[2]);
      if (l != NULL)
        {
          snprintf (buf[1], MSG_SIZE, "%s JOIN", CMD[0]);
          l->add_log (buf[1]);
        }
    }
  else if (strcmp (CMD[1], "QUIT") == 0)
    {
      mask2nick (CMD[0], buf[0]);
      for (i = 0; i < s->channel_num; i++)	// search in all channels
        // if it was here
        if (strcasecmp (CHANNELS[i]->last_deleted_nick, buf[0]) == 0)
          {
            l = log_list2log (log, CHANNELS[i]->name);
            if (l != NULL)
              {
                snprintf (buf[1], MSG_SIZE, "%s QUIT %s", buf[0], CMD[2]);
                l->add_log (buf[1]);
              }
          }
    }
  else if (strcmp (CMD[1], "KICK") == 0)
    {
      l = log_list2log (log, CMD[2]);
      if (l != NULL)
        {
          mask2nick (CMD[0], buf[0]);
          snprintf (buf[1], MSG_SIZE, "%s KICK %s - %s", buf[0], CMD[3],
                    CMD[4]);
          l->add_log (buf[1]);
        }
    }
  else if (strcmp (CMD[1], "PART") == 0)
    {
      l = log_list2log (log, CMD[2]);
      if (l != NULL)
        {
          mask2nick (CMD[0], buf[0]);
          snprintf (buf[1], MSG_SIZE, "%s PART", buf[0]);
          l->add_log (buf[1]);
        }
    }
  else if (strcmp (CMD[1], "NICK") == 0)
    {
      mask2nick (CMD[0], buf[0]);
      for (i = 0; i < s->channel_num; i++)	// search in all channels
        {
          // if it was here
          if (strcasecmp (CHANNELS[i]->last_deleted_nick, buf[0]) == 0)
            {
              l = log_list2log (log, CHANNELS[i]->name);
              if (l != NULL)
                {
                  snprintf (buf[1], MSG_SIZE, "%s NICK %s", buf[0], CMD[2]);
                  l->add_log (buf[1]);
                }
            }
        }
    }
  else if (strcmp (CMD[1], "MODE") == 0)
    {
      l = log_list2log (log, CMD[2]);
      if (l != NULL)
        {
          int num_modes;
          mask2nick (CMD[0], buf[0]);
          snprintf (buf[1], MSG_SIZE, "%s MODE %s", buf[0], CMD[3]);
          for (i = num_modes = 0; CMD[3][i] != 0; i++)
            if (CMD[3][i] != '+' && CMD[3][i] != '-')
              {
                num_modes++;
                snprintf (buf[2], MSG_SIZE, "%s %s", buf[1], 
                          CMD[3+num_modes]);
                my_strncpy (buf[1], buf[2], MSG_SIZE);
              }
          l->add_log (buf[1]);
        }
    }
  else if (strcmp (CMD[1], "PRIVMSG") == 0)
    {
      if (CMD[2][0] == '#')			// if it's a channel
        {
          l = log_list2log (log, CMD[2]);
          if (l != NULL)
            {
              // log the nick
              snprintf (buf[0], MSG_SIZE, "<%s> %s", SOURCE, CMD[3]);
              l->add_log (buf[0]);
            }
        }
      else					// pvt with the bot
        {
          if (log->log_pvt != NULL)		// logs them all?
            {
              snprintf (buf[0], MSG_SIZE, "<%s> %s", SOURCE, CMD[3]);
              log->log_pvt->add_log (buf[0]);
            }
          l = log_list2log (log, SOURCE);	// specific pvt
          if (l != NULL)
            l->add_log (CMD[3]);
        }
    }
}

// configuration file's local parser
static void
log_conf (CServer *s, const char *bufread)
{
  char buf[10][MSG_SIZE+1];

  strsplit (bufread, buf, 1);

  if (strcasecmp (buf[1], "log") == 0)
    {
      log_type *log = server2log (s);
      if (log == NULL)		// if it's the first log for this server
        {
          log_add (s);
          s->script->add_event (log_event);
          log = server2log (s);
        }
      if (log->log_num == LOG_MAX)
        s->bot->conf_error ("maximum log number exceeded.");
      strsplit (bufread, buf, 4);
      log->logs[log->log_num] = new CLog (s, buf[2], buf[3], buf[4]);
      if (!log->logs[log->log_num]->start_log ())
        s->bot->conf_error ("can't open log file.");
      if (strcasecmp (buf[2], "<privmsg>") == 0)	//log all pvts
        log->log_pvt = log->logs[log->log_num];
      log->log_num++;
    }
  else if (strcasecmp (buf[1], "logmail") == 0)
    {
#ifdef USE_MAIL
      log_type *log = server2log (s);
      if (log != NULL)
        {
          strsplit (bufread, buf, 3);
          for (int i = 0; i < log->log_num; i++)
            if (strcasecmp (buf[2], log->logs[i]->what) == 0)
              {
                my_strncpy (log->logs[i]->mail, buf[3], MAIL_SIZE);
                return;
              }
        }
      s->bot->conf_error ("can't find specified log");
#else
      s->bot->conf_error ("can't send mails - either 'cat', 'echo' or 'sendmail' wasn't found by configure");
#endif
    }
  else if (strcasecmp(buf[1], "logflush") == 0)
    {
      log_type *log = server2log (s);
      if (log != NULL)
        {
          strsplit (bufread, buf, 3);
          for (int i = 0; i < log->log_num; i++)
            if (strcasecmp (buf[2], log->logs[i]->what) == 0)
              {
                log->logs[i]->log_flush = 1;
                return;
              }
        }
      s->bot->conf_error ("can't find specified log");
    }
}

// module termination
static void
log_stop (CModule *m)
{
  log_type *buf = log_list, *buf2;
  while (buf != NULL)
    {
      for (int i = 0; i < buf->log_num; i++)
        delete buf->logs[i];
      buf->server->script->del_event (log_event);
      buf2 = buf->next;
      delete buf;
      buf = buf2;
    }
}

// module initialization
static void
log_start (CModule *m)
{
  log_list = NULL;
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "log",
  log_start,
  log_stop,
  log_conf,
  NULL
};
