/*

  This module adds the command "invite", which when executed in pvt with the
bot, makes it invite you to a specified channel. Activate this with:

set invite_channel <#channel>

*/

#include "mbot.h"

struct invite_type {
  CServer *server;
  char name[CHANNEL_SIZE+1];
  invite_type *next;
};
struct invite_type *invite_list;

///////////////
// prototypes
///////////////

static void invite_cmd_invite (CServer *);
static void invite_add (CServer *);
static invite_type *server2invite (CServer *);
static void invite_conf (CServer *, const char *);
static void invite_stop (CModule *);
static void invite_start (CModule *);
static void invite_var (CServer *, const char *, char *, size_t);

/////////////
// commands
/////////////

// invite
static void
invite_cmd_invite (CServer *s)
{
  invite_type *invite = server2invite (s);
  if (invite == NULL)
    return;
  // doesn't work inside #'s
  if (invite->name[0] != 0 && CMD[3][6] == 0 && strcasecmp (CMD[2], s->nick) == 0)
    {
      int i = CHANNEL_INDEX (invite->name);	// if the bot is there..
      if (i != -1)
        CHANNELS[i]->irc_invite (s->script->source);	// ..invite
      return;
    }
}

////////////////////
// module managing
////////////////////

// adds a server/channel pair to the list
static void
invite_add (CServer *s)
{
  invite_type *buf=invite_list;
  invite_list = new invite_type;
  invite_list->server = s;
  invite_list->name[0] = 0;
  invite_list->next = buf;
}

// returns the invite channel for a given server, NULL if nonexistant
static invite_type *
server2invite (CServer *s)
{
  invite_type *buf = invite_list;
  while (buf != NULL)
    {
      if (buf->server == s)
        return buf;
      buf = buf->next;
    }
  return NULL;
}

// configuration file's local parser
static void
invite_conf (CServer *s, const char *bufread)
{
  char buf[10][MSG_SIZE+1];

  strsplit (bufread, buf, 1);

  if (strcasecmp (buf[1], "bot") == 0)
    {
      if (server2invite (s) == NULL)
        {
          invite_add (s);
          s->script->bind_cmd (invite_cmd_invite, 0, "invite");
          s->vars->var_add ("invite_channel", invite_var);
        }
      else
        s->bot->conf_error ("invite already defined in this server.");
    }
}

// module termination
static void
invite_stop (CModule *m)
{
  invite_type *buf = invite_list, *buf2;
  while (buf != NULL)
    {
      buf->server->script->unbind_cmd ("invite");
      buf2 = buf->next;
      delete buf;
      buf = buf2;
    }
  // try to remove from all servers, even if it's not there (we don't know)
  for (int i=0; i<m->b->server_num; i++)
    m->b->servers[i]->vars->var_del ("invite_channel");
}

static void
invite_var (CServer *s, const char *name, char *data, size_t n)
{
  invite_type *buf = server2invite (s);
  if (buf == NULL)
    return;
  if (strcasecmp (name, "invite_channel") == 0)
    {
      if (n != 0)
        my_strncpy (data, buf->name, n);
      else
        my_strncpy (buf->name, data, CHANNEL_SIZE);
    }
}

// module initialization
static void
invite_start (CModule *m)
{
  invite_list = NULL;
}

struct CModule::module_type module = {
  MODULE_VERSION,
  "invite",
  invite_start,
  invite_stop,
  invite_conf,
  invite_var
};
