/*
 *
 *
 *  Copyright (c) 2000 Barak Weichselbaum <barak@komodia.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *
 * Contact info:
 * Site: http://www.komodia.com
 * Email: barak@komodia.com
 */

#include "stdafx.h"
#include "SocketThreadManager.h"
#include "AsyncSocket.h"
#include "OSManager.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

#define CSocketThreadManager_Class "CSocketThreadManager"

CSocketThreadManager::CSocketThreadManager(int iThreadCount,HINSTANCE hInstance) : CSpoofBase(),
																				   m_iThreadCount(iThreadCount),
																				   m_hInstance(hInstance),
																				   m_pThreadData(NULL),
																				   m_pCSection(NULL)
{
	try
	{
		//Set the class name
		SetName(CSocketThreadManager_Class);

		//Start spawning threads
		SpawnThreads();

		//Create the critical section
		m_pCSection=COSManager::CreateCriticalSection();
	}
	ERROR_HANDLER("CSocketThreadManager")
}

CSocketThreadManager::~CSocketThreadManager()
{
	try
	{
		delete m_pCSection;

		//Delete the thread data
		CleanThreads();
	}
	ERROR_HANDLER("~CSocketThreadManager")
}

void CSocketThreadManager::SpawnThreads()
{
	try
	{
		//Start creating threads
		//Allocate the thread structure
		m_pThreadData=new ThreadData[m_iThreadCount];

		//And initialize it
		memset(m_pThreadData,0,sizeof(ThreadData)*m_iThreadCount);

		//Wait for all threads
		HANDLE* pHandle;
		pHandle=new HANDLE[m_iThreadCount];

		//Reset them
		memset(pHandle,0,sizeof(HANDLE)*m_iThreadCount);

		//Start spawning
		for (int iCounter=0;iCounter<m_iThreadCount;++iCounter)
		{
			//Create an event
			m_pThreadData[iCounter].hEvent=CreateEvent(NULL,FALSE,FALSE,NULL);

			//Save it to our array
			pHandle[iCounter]=m_pThreadData[iCounter].hEvent;

			//Set our instance
			m_pThreadData[iCounter].hInstance=m_hInstance;

			//And create it
			m_pThreadData[iCounter].pThread=COSManager::CreateThread(SocketThread);
			
			//Check the thread has been created
			if (!m_pThreadData[iCounter].pThread->GetThreadID())
			{
				//Report the error
				ReportError("SpawnThreads","Failed to create thread!");

				//Delete the handle array
				CleanThreads();

				//Quit
				return;
			}
			else
				//Start the thread
				m_pThreadData[iCounter].pThread->Start((LPVOID)&m_pThreadData[iCounter]);
		}

		//Wait for all the handles to finish
		if (WaitForMultipleObjectsEx(m_iThreadCount,pHandle,TRUE,10000,FALSE)==WAIT_TIMEOUT)
			//Report the error
			ReportError("SpawnThreads","Timeout waiting for threads!");

		//Release all the events
		for (iCounter=0;iCounter<m_iThreadCount;++iCounter)
			CloseHandle(pHandle[iCounter]);

		//Delete all the handles
		delete [] pHandle;
	}
	ERROR_HANDLER("SpawnThreads")
}

DWORD CSocketThreadManager::SocketThread(LPVOID lpParameter)
{
	try
	{
		//Get the address of our data
		ThreadData* pData;
		pData=(ThreadData*)lpParameter;

		//Create the window
		pData->hWindowHandle=CreateWindowEx(0,CAsyncSocket_Class,SOCKET_WINDOW_NAME,
											WS_OVERLAPPED,0,0,0,0,0,NULL,pData->hInstance,NULL);

		//Alert we are done
		SetEvent(pData->hEvent);

		//Check we have this window 
		if (pData->hWindowHandle)
		{
			//Run a message map
			MSG msg;

			while (GetMessage(&msg,NULL,0,0))
			{
				//Translate and dispatch
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}

		return FALSE;
	}
	ERROR_HANDLER_STATIC_RETURN(CSocketThreadManager_Class,"SocketThread",TRUE)
}

int CSocketThreadManager::GetMostAvailableThread() const
{
	try
	{
		//Start position
		int iStartPosition;
		iStartPosition=1;

		int iIndex;
		iIndex=0;

		//Check for freezing
		while (m_pThreadData[iIndex].bFreeze && iIndex<m_iThreadCount)
		{
			//Release the freeze
			m_pThreadData[iIndex].bFreeze=FALSE;

			++iIndex;
			++iStartPosition;
		}

		//Check we aren't in the end
		if (iIndex==m_iThreadCount)
			return GetMostAvailableThread();

		//Start searching the threads
		for (int iCounter=iStartPosition;iCounter<m_iThreadCount;++iCounter)
			//Check is it larger
			if (m_pThreadData[iCounter].iSocketCount<m_pThreadData[iIndex].iSocketCount && m_pThreadData[iCounter].pThread->GetThreadID())
				//Check if is freezed
				if (m_pThreadData[iCounter].bFreeze)
					//Release the freeze
					m_pThreadData[iCounter].bFreeze=FALSE;
				else
					//Set the new index
					iIndex=iCounter;

		//Return the value
		return iIndex+1;
	}
	ERROR_HANDLER_RETURN("GetMostAvailableThread",0)
}

HWND CSocketThreadManager::GetWindowHandle()
{	
	try
	{
		//Shared resource
		m_pCSection->Enter();
	
			//Get the freeiest index
			int iIndex;
			iIndex=GetMostAvailableThread();

			//Check it's valid
			if (!iIndex)
			{
				//Leave the critical section
				m_pCSection->Exit();

				//Quit
				return 0;
			}

			//Increase the socket count
			++m_pThreadData[iIndex-1].iSocketCount;

		//Leave the critical section
		m_pCSection->Exit();
		
		return m_pThreadData[iIndex-1].hWindowHandle;
	}
	ERROR_HANDLER("GetWindowHandle")

	//Quit from the critical section
	m_pCSection->Exit();

	return 0;
}

void CSocketThreadManager::DecreaseSocketCount(HWND hWindowHandle,BOOL bFreeze)
{
	try
	{
		//First find the window handle
		int iIndex;
		iIndex=GetIndexByHWND(hWindowHandle);

		//Check it's valid
		if (!iIndex)
			return;

		//Enter the critical section
		m_pCSection->Enter();

			//Decrement the socket count
			if (m_pThreadData[iIndex-1].iSocketCount>0)
			{
				--m_pThreadData[iIndex-1].iSocketCount;
				m_pThreadData[iIndex-1].bFreeze=bFreeze;
			}

		//Leave the critical section
		m_pCSection->Exit();

		return;
	}
	ERROR_HANDLER("DecreaseSocketCount")

	//Error, release the critical section
	m_pCSection->Exit();
}

int CSocketThreadManager::GetIndexByHWND(HWND hHandle) const
{
	try
	{
		for (int iCounter=0;iCounter<m_iThreadCount;++iCounter)
			if (m_pThreadData[iCounter].hWindowHandle==hHandle)
				//Return it
				return iCounter+1;

		//Nothing
		return 0;
	}
	ERROR_HANDLER_RETURN("GetIndexByHWND",0)
}

void CSocketThreadManager::CleanThreads()
{
	try
	{
		if (m_pThreadData)
		{
			for (int iCounter=0;iCounter<=m_iThreadCount;++iCounter)
				//Post a stop message
				if (m_pThreadData[iCounter].pThread->GetThreadID())
				{
					PostThreadMessage(m_pThreadData[iCounter].pThread->GetThreadID(),WM_QUIT,0,0);

					//Delete the window
					DestroyWindow(m_pThreadData[iCounter].hWindowHandle);
				}

			//Delete the thread classes
			for (iCounter=0;iCounter<=m_iThreadCount;++iCounter)
				//Post a stop message
				delete m_pThreadData[iCounter].pThread;

			//Delete the structure
			delete [] m_pThreadData;
		}
	}
	ERROR_HANDLER("CleanThreads")
}
