/* 
 * $smu-mark$ 
 * $name: waitpacket.c$ 
 * $author: Salvatore Sanfilippo <antirez@invece.org>$ 
 * $copyright: Copyright (C) 1999 by Salvatore Sanfilippo$ 
 * $license: This software is under GPL version 2 of license$ 
 * $date: Fri Nov  5 11:55:50 MET 1999$ 
 * $rev: 8$ 
 */ 

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>

#include "hping2.h"
#include "globals.h"

/*	FIXME:	wait_packet don't support ip and tcp options,
	it waits for 20 bytes IP hdr + 20 bytes TCP hdr.
	--> ip options fixed 31/08/99 (MK)
*/
void	wait_packet(void)
{
	char	packet	[LINK_PACKETSIZE],
		recvflags[1024],
		src_addr[1024],
		dst_addr[1024],
		*p;
	int	recv_sport,
		recv_dport,
		size,
		rstseq,
		winsize,
		status,
		match = FALSE,
		ip_id,
		rel_id;
	float	ms_delay;
	struct	in_addr	src, dst;
	struct  myiphdr *ip;
	struct  mytcphdr *tcp;
	struct  myudphdr *udp;
	struct  myicmphdr *icmp;
	struct  myiphdr *icmp_iph;
	char	hlen;

#if (!defined OSTYPE_LINUX) || (defined FORCE_LIBPCAP)
	size = pcap_recv(packet, LINK_PACKETSIZE);
	if (size == -1) {
		perror("[wait_packet] pcap_recv()");
		return;
	}
#else
	size = recv(sockpacket, packet, LINK_PACKETSIZE, 0);
	if (size == -1) {
		if (errno != EINTR)
		{
			perror("[wait_packet] recv");
			print_statistics(-1);
		}
		else
			return;
	}
#endif

	ip	 = (struct myiphdr*)(packet+ABS_OFFSETIP);

	/* FIXME: compute hlen without sanity checking, but seems it isn't a
	   problem since our buffer is always bigger than interfaces MTU size.
	   If the buffer is resized this may be a problem, even if it is,
	   still now, accessed in read-only. It will be fixed when a general
	   incoming packets sanity checking will be implemented. This sanity
	   checking will include at least IP, TCP, UDP and ICMP morphology
	   and checksum. AZ */
	hlen	 = ip->ihl << 2;
	tcp	 = (struct mytcphdr*)(packet+ABS_OFFSETIP+hlen);
	udp	 = (struct myudphdr*)(packet+ABS_OFFSETIP+hlen);
	icmp	 = (struct myicmphdr*)(packet+ABS_OFFSETIP+hlen);
	icmp_iph = (struct myiphdr*)(packet+ABS_OFFSETIP+hlen+8);

	/* --------- HCMP --------- */
	if (opt_sign && (p = memstr(packet, rsign, size)))
	{
		struct hcmphdr *hcmph;
		__u16 seqnum;
		__u32 usec;

		if (opt_debug)
			fprintf(stderr, "HCMP received\n");

		p+=strlen(rsign);
		if ((size-(packet-p)) < sizeof(struct hcmphdr))
		{
			if (opt_verbose || opt_debug)
				fprintf(stderr, "bad HCMP len received\n");
			goto hcmp_exit;
		}

		hcmph = (struct hcmphdr*) p;

		switch(hcmph->type)
		{
		case HCMP_RESTART:
			seqnum = ntohs(hcmph->typedep.seqnum);
			break;
		case HCMP_SOURCE_QUENCH:
		case HCMP_SOURCE_STIRUP:
			usec = ntohl(hcmph->typedep.usec);
		default:
			if (opt_verbose || opt_debug)
				fprintf(stderr, "bad HCMP type received\n");
			goto hcmp_exit;
		}

		switch(hcmph->type)
		{
		case HCMP_RESTART:
			src_id = seqnum;		/* set id */
			datafiller(NULL, seqnum);	/* data seek */
			goto hcmp_exit;
		case HCMP_SOURCE_QUENCH:
		case HCMP_SOURCE_STIRUP:
			printf("Source quench / stirpu"
				" still not implemented!\n");
			goto hcmp_exit;
		/* default handled in the first switch */
		}
	}

hcmp_exit:

	/* get source and target address */
	bcopy( &(ip->saddr), &src, sizeof(struct in_addr) );
	bcopy( &(ip->daddr), &dst, sizeof(struct in_addr) );
	strncpy(src_addr, inet_ntoa(src), 1024);
	strncpy(dst_addr, inet_ntoa(dst), 1024);

	/* get ip->id */
	if (opt_winid_order)
		ip_id = ip->id;
	else
		ip_id = htons(ip->id);

	/* get source and target port */
	if (opt_udpmode)	/* UDP */
	{
		recv_sport = ntohs(udp->uh_sport);
		recv_dport = ntohs(udp->uh_dport);
	}
	else			/* TCP */
	{
		recv_sport = ntohs(tcp->th_sport);
		recv_dport = ntohs(tcp->th_dport);
	}

	/* --------- TCP --------- */
	if ( ip->protocol == IPPROTO_TCP &&
	     !memcmp(&ip->saddr, &remote.sin_addr, sizeof(ip->saddr)) &&
	     !memcmp(&ip->daddr, &local.sin_addr, sizeof(ip->daddr)) &&
	     (recv_sport == dst_port || opt_force_incdport)
	)
	{
		match = TRUE;

		rstseq    = recv_dport - initsport;
		winsize   = ntohs(tcp->th_win);
		tcp_exitcode = tcp->th_flags;

		recvflags[0] = '\0';
		if (tcp->th_flags & TH_RST)  strcat(recvflags, "R");
		if (tcp->th_flags & TH_SYN)  strcat(recvflags, "S");
		if (tcp->th_flags & TH_ACK)  strcat(recvflags, "A");
		if (tcp->th_flags & TH_FIN)  strcat(recvflags, "F");
		if (tcp->th_flags & TH_PUSH) strcat(recvflags, "P");
		if (tcp->th_flags & TH_URG)  strcat(recvflags, "U");
		if (tcp->th_flags & TH_X)    strcat(recvflags, "X");
		if (tcp->th_flags & TH_Y)    strcat(recvflags, "Y");
		if (recvflags[0] == '\0')    strcat(recvflags, "none");

		/* obtain round trip time */
		status = rtt(rstseq, &ms_delay);

		/* relativize packets ip->id */
		if (opt_relid)
			rel_id = relativize_id(rstseq, &ip_id);
		else
			rel_id = FALSE;

		if (opt_seqnum)
		{
			static unsigned long old_th_seq = 0;
			unsigned long seq_diff;
			if (tcp->th_seq >= old_th_seq)
				seq_diff = tcp->th_seq - old_th_seq;
			else
				seq_diff = (4294967295U - old_th_seq)
					+ tcp->th_seq;
			old_th_seq = tcp->th_seq;
			printf("%10lu +%lu\n",
				(unsigned long) tcp->th_seq,
				(unsigned long) seq_diff);
		}
		if (!opt_quiet && !opt_seqnum) {
			if (status == S_RECV)
				printf("(DUP!) ");
			if (!opt_incdport)
				printf("%d bytes from %s: ",
					size - linkhdr_size, src_addr);
			else
				printf("%d bytes %s,%d: ",
					size - linkhdr_size, src_addr,
					recv_sport);
			printf("flags=%s ", recvflags);
			printf("seq=%d ", rstseq);
			printf("ttl=%d ", ip->ttl);
			printf("id");
			if (rel_id == TRUE)
				printf("=+");
			else
				putchar('=');
			printf("%d ", ip_id);
			printf("win=%d ", winsize);
			printf("rtt=%.1f ms", ms_delay);
			putchar('\n');
		}
		if (opt_verbose && !opt_quiet) {
			printf("              ");
			printf("tos = %10x ", ip->tos);
			printf("len = %10u ", htons(ip->tot_len));
			printf("\n              ");
			printf("seq = %10lu ", (unsigned long) ntohl(tcp->th_seq));
			printf("ack = %10lu ", (unsigned long) ntohl(tcp->th_ack));
			printf("\n              ");
			printf("sum = %10x ", tcp->th_sum);
			printf("urp = %10u ", tcp->th_urp);
			putchar('\n');
		}

		fflush(stdout);
		recv_pkt++;
		if (opt_incdport && !opt_force_incdport)
			dst_port++;
	}

	/* --------- ICMP --------- */
	if ( ip->protocol == IPPROTO_ICMP )
	{
		match = TRUE;

		if ( icmp->type == 0 && icmp->un.echo.id == getpid() )
		{
			recv_pkt++;
			if (!opt_quiet)
			{
				rstseq = icmp->un.echo.sequence;

				/* obtain round trip time */
				status = rtt(rstseq, &ms_delay);

				/* relativize packets ip->id */
				if (opt_relid)
					rel_id = relativize_id(rstseq, &ip_id);
				else
					rel_id = FALSE;

				printf("%d bytes from %s: ",
				size - linkhdr_size, src_addr);
				printf("icmp_seq=%d ", rstseq);
				printf("ttl=%d ", ip->ttl);
				printf("id");
				if (rel_id == TRUE)
					printf("=+");
				else
					putchar('=');
				printf("%d ", ip_id);
				printf("rtt=%.1f ms", ms_delay);
				putchar('\n');
			}
		}
		else if ( icmp->type == 3 && /* Dest. Unreachable */
			!memcmp(&icmp_iph->daddr,
				&remote.sin_addr,
				sizeof(ip->daddr)) &&
			!memcmp(&ip->daddr,
				&local.sin_addr,
				sizeof(ip->daddr))
		)
		{
			if (!opt_quiet)
				log_icmp_unreach(src_addr, icmp->code);
		}
		else if (icmp->type == 11 && /* Time exceeded */
			 !memcmp(&icmp_iph->daddr,
				 &remote.sin_addr,
				 sizeof(ip->daddr)) &&
			 !memcmp(&ip->daddr,
				 &local.sin_addr,
				 sizeof(ip->daddr))
		)
		{
			/* traceroute mode */
			static unsigned char old_src_addr[4]={0,0,0,0};
			if (opt_traceroute)
			{
				if (memcmp(&ip->saddr, old_src_addr, 4))
				{
					memcpy(old_src_addr, &ip->saddr, 4);
					printf("%d->", src_ttl);
					src_ttl++;
					fflush(stdout);
					log_icmp_timeexc(src_addr, icmp->code);
				}
			}
			else
			{
				if (!opt_quiet)
					log_icmp_timeexc(src_addr, icmp->code);
			}
		}
		else
			match = FALSE;
	}

	/* --------- UDP --------- */
	if ( ip->protocol == IPPROTO_UDP &&
	     !memcmp(&ip->saddr, &remote.sin_addr, sizeof(ip->saddr)) &&
	     !memcmp(&ip->daddr, &local.sin_addr, sizeof(ip->daddr)) &&
	     recv_sport == dst_port )

	{
		match = TRUE;
		rstseq    = recv_dport - initsport;

		/* obtain round trip time */
		status = rtt(rstseq, &ms_delay);

		/* relativize ip->id */
		if (opt_relid)
			rel_id = relativize_id(rstseq, &ip_id);
		else
			rel_id = FALSE;

		if (!opt_quiet)
		{
			if (status == S_RECV)
				printf("(DUP!) ");
			if (!opt_incdport)
				printf("%d bytes from %s: ",
					size - linkhdr_size, src_addr);
			else
				printf("%d bytes %s,%d: ",
					size - linkhdr_size, src_addr,
					recv_sport);
			printf("seq=%d ", rstseq);
			printf("ttl=%d ", ip->ttl);
			printf("id");
			if (rel_id == TRUE)
				printf("=+");
			else
				putchar('=');
			printf("%d ", ip_id);
			printf("rtt=%.1f ms", ms_delay);
			putchar('\n');
		}

		fflush(stdout);
		recv_pkt++;
		if (opt_incdport && !opt_force_incdport)
			dst_port++;
	}

	/* dump the received packet in hex */
	if (opt_hexdump && match && !opt_quiet)
	{
		unsigned char *byte;
		int count = 0;

		printf("\t\t");
		for (byte = (unsigned char*) packet; byte < (unsigned char*) (packet+size); byte++)
		{
			count++;
			printf("%02x", *byte);
			if (count % 2 == 0) printf(" ");
			if (count % 16 == 0) printf("\n\t\t");
		}
		printf("\n\n");
	}

	/* dump the received packet printable characters */
	if (opt_contdump && match && !opt_quiet)
	{
		unsigned char *byte;
		int count = 0;

		printf("\t\t");
		for (byte = (unsigned char*) packet; byte < (unsigned char*) (packet+size); byte++)
		{
			count ++;
			if (isprint(*byte))
				printf("%c", *byte);
			else
				printf(".");
			if (count % 32 == 0) printf("\n\t\t");
		}
		printf("\n\n");
	}

	/* display IP options */
	if (opt_rroute && match && !opt_quiet)
	{
		display_ipopt((char*)ip);
	}

	/* exit if 'count' is reached */
	if (count != -1 && count == recv_pkt)
		print_statistics(0);
}
