/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon public domain software
 * originally written at the National Center for Supercomputing Applications,
 * University of Illinois, Urbana-Champaign.
 */

/*
 * mod_protection.c
 * 
 * written by Pierpaolo Giacomin aka yaroze
 * contacts: yaroze@twlc.net
 * thanks: supergate to force me working
 *
 * mod_protection - A module to protect you from malicious request
 *                  and track it faster then parsing logs.
 *
 */

#include "httpd.h"
#include "http_config.h"
#include <syslog.h>

typedef enum{
    PRO_AP_URI,
    PRO_AP_ARG
} pro_ap_entry_type;

typedef struct {
    char *fake;
    regex_t *regexp;
    pro_ap_entry_type type;
} protection_entry;

typedef struct {
    struct hostent *host;
    unsigned short port;
} protection_host;

typedef struct {
    char *menace;
    char *handler;
    array_header *uris;
    array_header *hosts;
} protection_server_conf;

module MODULE_VAR_EXPORT protection_module;

static void *alarm_protection(request_rec *r, int arg)
{
    void *sconf = r->server->module_config;
    protection_server_conf *serverconf =
	(protection_server_conf *) ap_get_module_config(sconf, &protection_module);
    int sd, i;
    char *msg;
    protection_host *hosts = (protection_host *) serverconf->hosts->elts;

    if(r->args) {
	msg = ap_pstrcat(r->pool, r->uri,"?", r->args, NULL);
    }
    else {
	msg = ap_pstrdup(r->pool, r->uri);
    }
    if(r->connection->remote_ip) {
	msg = ap_pstrcat(r->pool, msg, " from ",
			 r->connection->remote_ip, "\n", NULL);
    }
    else {
	msg = ap_pstrcat(r->pool, msg, "\n", NULL);
    }

    if( r->args || !arg  ) {
	for (i = 0; i < serverconf->hosts->nelts; ++i) {
	    protection_host *el = &hosts[i];
	    struct sockaddr_in sin;
	    int j = 0;
	    sd = ap_psocket(r->pool, PF_INET, SOCK_STREAM, IPPROTO_TCP);
	    bzero( (char *) &sin, sizeof(sin) );
	    sin.sin_family = AF_INET;
	    sin.sin_port = htons( el->port );
	    if( !el->host )
		continue;
	    memcpy( &sin.sin_addr, el->host->h_addr, el->host->h_length);
	    connect(sd, (struct sockaddr *)&sin, sizeof(struct sockaddr_in));

	    write(sd, msg, strlen(msg));

	    ap_pclosesocket(r->pool, sd);
	}
    }
    return NULL;
}

static void *create_protection_config(pool *p, server_rec *s)
{
    protection_server_conf *a =
    (protection_server_conf *) ap_pcalloc(p, sizeof(protection_server_conf));

    a->uris = ap_make_array(p, 10, sizeof(protection_entry));
    a->hosts = ap_make_array(p, 10, sizeof(protection_host));
    return a;
}


static void *merge_protection_config(pool *p, void *basev, void *overridesv)
{
    protection_server_conf *a =
    (protection_server_conf *) ap_pcalloc(p, sizeof(protection_server_conf));
    protection_server_conf *base = (protection_server_conf *) basev, *overrides = (protection_server_conf *) overridesv;

    a->menace = overrides->menace;
    a->handler = overrides->handler;
    a->uris = ap_append_arrays(p, overrides->uris, base->uris);
    a->hosts = ap_append_arrays(p, overrides->hosts, base->hosts);
    return a;
}


static const char *add_host(cmd_parms *cmd, void *dummy, char *h, char *p)
{
    unsigned short port;
    server_rec *s = cmd->server;
    protection_server_conf *conf =
	(protection_server_conf *) ap_get_module_config(s->module_config,
							&protection_module);
    protection_host *new = ap_push_array(conf->hosts);

    port = atoi(p);

    new->host = ap_pgethostbyname(cmd->pool, h);
    new->port = port;

    return NULL;
}

static const char *add_menace(cmd_parms *cmd, void *dummy, char *f)
{
    server_rec *s = cmd->server;
    protection_server_conf *conf =
	(protection_server_conf *) ap_get_module_config(s->module_config,
							&protection_module);

#ifndef OS2
    conf->menace = ap_os_canonical_filename(cmd->pool, f);
#else
    conf->menace = f;
#endif
    /*XXX remember the handler */
    conf->handler = cmd->info;

    return NULL;
}

static const char *add_alert_internal(cmd_parms *cmd, void *dummy, char *f, pro_ap_entry_type t, int use_regex)
{
    server_rec *s = cmd->server;
    protection_server_conf *conf =
    (protection_server_conf *) ap_get_module_config(s->module_config, &protection_module);
    protection_entry *new = ap_push_array(conf->uris);

    if (use_regex) {
	new->regexp = ap_pregcomp(cmd->pool, f, REG_EXTENDED);
	if (new->regexp == NULL)
	    return "Regular expression could not be compiled.";
    }
    new->fake = f;
    new->type = t;
    return NULL;
}

static const char *add_arg(cmd_parms *cmd, void *dummy, char *a)
{
    return add_alert_internal(cmd, dummy, a, PRO_AP_ARG, 0);
}

static const char *add_arg_regex(cmd_parms *cmd, void *dummy, char *a)
{
    return add_alert_internal(cmd, dummy, a, PRO_AP_ARG, 1);
}

static const char *add_uri(cmd_parms *cmd, void *dummy, char *u)
{
    return add_alert_internal(cmd, dummy, u, PRO_AP_URI, 0);
}

static const char *add_uri_regex(cmd_parms *cmd, void *dummy, char *u)
{
    return add_alert_internal(cmd, dummy, u, PRO_AP_URI, 1);
}

static const command_rec protection_cmds[] =
{
    {"Menace", add_menace, NULL, RSRC_CONF, TAKE1,
    "where redirect blocked request"},
    {"WarnHost", add_host, NULL, RSRC_CONF, TAKE2,
    "host and port to warn"},
    {"ArgAlertMatch", add_arg_regex, NULL, OR_OPTIONS, TAKE1,
    "a regular expression for protected args"},
    {"ArgAlert", add_arg, NULL, OR_OPTIONS, TAKE1,
    "protected arguments"},
    {"UriAlertMatch", add_uri_regex, NULL, OR_OPTIONS, TAKE1,
    "a regular expression for a protected uri"},
    {"UriAlert", add_uri, NULL, OR_OPTIONS, TAKE1,
    "a protected uri"},
    {NULL}
};


static int protection_matches(const char *uri, const char *protection_fakename)
{
    const char *end_fakename = protection_fakename + strlen(protection_fakename);
    const char *protectionp = protection_fakename, *urip = uri;

    while (protectionp < end_fakename) {
	if (*protectionp == '/') {
	    if (*urip != '/') {
		return 0;
	    }

	    while (*protectionp == '/') {
		++protectionp;
	    }

	    while (*urip == '/') {
		++urip;
	    }
	}
	else {
	    if (*urip++ != *protectionp++) {
		return 0;
	    }
	}
    }


    if (protectionp[-1] != '/' && *urip != '\0' && *urip != '/') {
	return 0;
    }

    return urip - uri;
}

static char *try_protection_list(request_rec *r, protection_server_conf *serverconf, int *arg)
{
    protection_entry *entries = (protection_entry *) serverconf->uris->elts;
    regmatch_t regm[10];
    char *found = NULL;
    char *m = NULL;
    char *comp = NULL;
    int i;
    
    comp = ap_pstrcat(r->pool, r->uri, "?", r->args, NULL);

    if (serverconf->menace) {
	m = serverconf->menace;
    }
    else {
	m = ap_pstrdup(r->pool, "/");
    }

    for (i = 0; i < serverconf->uris->nelts; ++i) {
	protection_entry *p = &entries[i];
	int l;

	switch(p->type) {
	case PRO_AP_URI:
	    if(r->uri) {
		comp = ap_pstrdup(r->pool, r->uri);
		*arg = 0;
	    }
	    break;
	case PRO_AP_ARG:
	    if(r->args) {
		comp = ap_pstrdup(r->pool, r->args);
		*arg = 1;
	    }
	    break;
	default:
	    if(r->uri) {
		comp = ap_pstrdup(r->pool, r->uri);
                *arg = 0;
	    }
	    break;
	}

	if(!comp) {
	    comp = ap_pstrdup(r->pool, "");
	}

	if (p->regexp) {
	    if (!ap_regexec(p->regexp, comp, p->regexp->re_nsub + 1, regm, 0)) {
        	found = m;
	    }
	}
	else {
	    l = protection_matches(comp, p->fake);

	    if (l > 0) {
		found = m;
	    }
	}

	if (found) {
	    return found;
	}
    }

    return NULL;
}

static int trans_protection(request_rec *r)
{
    void *sconf = r->server->module_config;
    protection_server_conf *serverconf =
	(protection_server_conf *) ap_get_module_config(sconf, &protection_module);
    char *ret;
    int arg;

    if ( r->uri[0] != '/' && r->uri[0] != '\0') {
	return DECLINED;
    }

    if ( ( ret = try_protection_list(r, serverconf, &arg)) != NULL) {
	r->filename = ret;
	alarm_protection(r, arg);
	return OK;
    }

    return DECLINED;
}


module MODULE_VAR_EXPORT protection_module =
{
    STANDARD_MODULE_STUFF,
    NULL,			/* initializer */
    NULL,                       /* dir config creater */
    NULL,                       /* dir merger */
    create_protection_config,	/* server config */
    merge_protection_config,	/* merge server configs */
    protection_cmds,    	/* command table */
    NULL,			/* handlers */
    trans_protection,           /* filename translation */
    NULL,			/* check_user_id */
    NULL,			/* check auth */
    NULL,			/* check access */
    NULL,			/* type_checker */
    NULL,       		/* fixups */
    NULL,			/* logger */
    NULL,			/* header parser */
    NULL,			/* child_init */
    NULL,			/* child_exit */
    NULL			/* post read-request */
};
