/* Pilot serial I/O test program
 * Rev 2	Improve serial performance
 * 			Implement chart scrolling
 * (c) 1997 Ingeneering Inc.
 * www.ingeninc.com
 * jeff@ingeninc.com
 * jkh, 12/1/97
 */


/* includes */
#include <Pilot.h>			// system
#include <SerialMgr.h>
#include <SysEvtMgr.h>
#include "HiLo2_res.h"		// app


/* defines */
#define	CHART_RECT_LEFT		30
#define	CHART_RECT_TOP		52
#define	CHART_RECT_WIDTH	100
#define	CHART_RECT_HEIGHT	90
#define MAX_DATA_POINTS		CHART_RECT_WIDTH
#define VERT_SCALE			CHART_RECT_HEIGHT / 255
#define POS_X				CHART_RECT_LEFT + CHART_RECT_WIDTH - 1
#define POS_Y				CHART_RECT_TOP + CHART_RECT_HEIGHT - 1


/* global variables */
static	UInt				SerIOAvg = 0;			// avg (mean) of points received
static	UInt				SerIOCnt = 0;			// points received
static	UInt				SerIOCur = 0;			// value of current point
static	UInt				SerIOSca = 0;			// scaled value of current point
static	UInt				SerIOMax = 0;			// max of all points received
static	UInt				SerIOMin = 257;			// min of all points received
static	UInt				SerIOSum = 0;			// sum of all points received

static	UInt				SerIORef = 0;			// handle to serial port
static	SerSettingsType		SerIOSettings;			// baud rate etc.
static 	Boolean				SerIOConnected = false;	// true when port active
static	RectangleType		ChartRecRect;			// chart window rect
static	Byte				Data[MAX_DATA_POINTS];	// input buffer (ring)
static	UInt				BufHead = 0;			// index to current buffer position
static	ULong				RcvCount = 0;			// number of items read recently


/* messages */
char	*strErrSerSetup = "Error setting up serial port";
char	*strErrSerPortOpen = "Error opening serial port";
char	*strErrRcvChk = "Serial receive check error";
char	*strErrRcvErr = "Serial receive error";
char	*strOKSerClose = "Serial port close OK";
char	*strErrSerClose = "Serial port close error: ";
char	*strErrSerLibOpen = "Error opening serial library";
char	*StrBlank = " ";
		

/* Function prototypes */
static void StartApplication(void);
static void StopApplication(void);
static Boolean MainFormHandleEvent(EventPtr event);
static void EventLoop(void);

static void SerIOOpen(void);
static void SerIOClose(void);
static void SerIOReceive(void);

static void ShowNumResultDlg (char *str, ULong val);
static void ShowNumResultFld (Word FieldID, ULong val);
static void EnableControl (FormPtr Frm, Word ControlID, Boolean State);
static void EnableField (FormPtr Frm, Word FieldID, Boolean State);

static void CreateChartRec(void);
static void AddPoints (UInt pos, ULong cnt, Byte *ib, Byte *ob);
static void DisplayChart(Byte *buf, UInt bufHd, ULong cnt);
static void FocusField(Word fldID);


/*
 * SerIOReceive
 * R:\CodeWarrior\PalmPilot WinSDK\Apps\Working Folder\PalmPilotApp.prc
 */
static void SerIOReceive(void) {
	Err 		error;
	Byte		BufIn[MAX_DATA_POINTS];
	
	if (!SerIORef || !SerIOConnected)
		return;
	
	error = SerReceiveCheck(SerIORef, &RcvCount);
	if (error) {	
		SerClearErr(SerIORef);
		FrmCustomAlert (AlertGenericAlert, strErrRcvChk, StrBlank, StrBlank);
		FocusField(HiLoMainFldScaleField);
		return;
	}
	
	if ((RcvCount < 1))
		return;
		
	RcvCount =  (RcvCount > MAX_DATA_POINTS) ? MAX_DATA_POINTS : RcvCount;

	// Palm OS1, OS2 compatible
	error = SerReceive10(SerIORef, BufIn, RcvCount, 10);
	
	// Palm OS2 compatible only
	//error = 0;
	//RcvCount = SerReceive(SerIORef, BufIn, 1, 0, &error);
	
	if (error) {
		SerClearErr(SerIORef);
		FrmCustomAlert (AlertGenericAlert, strErrRcvErr, StrBlank, StrBlank);
		FocusField(HiLoMainFldScaleField);
		return;
	}

	AddPoints (BufHead, RcvCount, BufIn, Data);
	
	BufHead = (BufHead + RcvCount) % MAX_DATA_POINTS;

	// display stats
	ShowNumResultFld (HiLoMainFldAvgField, SerIOAvg);
	ShowNumResultFld (HiLoMainFldRxCountField, SerIOCnt);
	ShowNumResultFld (HiLoMainFldLoField, SerIOMin);
	ShowNumResultFld (HiLoMainFldHiField, SerIOMax);
	ShowNumResultFld (HiLoMainFldLastField, SerIOCur);
	ShowNumResultFld (HiLoMainFldScaleField, SerIOSca);		
}


/*
 * ShowNumResultDlg
 */
static void ShowNumResultDlg (char *str, ULong val) {
	char		Buf[32];
	
	StrIToA(Buf, val);
	FrmCustomAlert (AlertGenericAlert, str, StrBlank, Buf);
}


/*
 * ShowNumResultFld
 */
static void ShowNumResultFld (Word FieldID, ULong val) {
	FormPtr 	Frm;
	FieldPtr 	FldPtr;
	char		Buf[32];
	
	Frm = FrmGetFormPtr(HiLoMainForm);
	FldPtr = (FieldPtr)(FrmGetObjectPtr(Frm, (FrmGetObjectIndex(Frm, FieldID))));
	StrIToA(Buf, val);
	FldDelete(FldPtr, 0, FldGetTextLength(FldPtr));
	FldInsert(FldPtr, Buf, StrLen(Buf));
}


/*
 * AddPoints
 */
static void AddPoints (UInt pos, ULong cnt, Byte *ib, Byte *ob) {
	SerIOCnt += cnt;
	while (cnt > 0) {
		// get input char, put in buffer
		SerIOCur = *ib;
		ob[pos] = SerIOCur;
		
		// update stats
		SerIOSum += SerIOCur;
		SerIOMin = (SerIOCur < SerIOMin) ? SerIOCur : SerIOMin;
		SerIOMax = (SerIOCur > SerIOMax) ? SerIOCur : SerIOMax;
		
		// update pointers, counters
		cnt--;
		ib++;
		pos = (pos + 1) % MAX_DATA_POINTS;
	}
	
	// update stats
	SerIOAvg = SerIOSum / SerIOCnt;
	SerIOSca = (int) (SerIOCur * VERT_SCALE);
}


/*
 * DisplayChart
 */
static void DisplayChart(Byte *buf, UInt bufHd, ULong cnt) {
	Short			xPos;
	RectangleType	r;
	Short			ScrollSize = 0;
	
	// scroll the window, erase any leavings
	cnt = (cnt > MAX_DATA_POINTS) ? MAX_DATA_POINTS : cnt;
	if (cnt < MAX_DATA_POINTS) {
		WinScrollRectangle (&ChartRecRect, left, cnt, &r);
		WinEraseRectangle(&r, 0);
	}
	else
		WinEraseRectangle(&ChartRecRect, 0);
		
	// draw the last point received, right-most, and work backward
	xPos = POS_X;
	while (cnt > 0) {
		bufHd--;
		bufHd = (bufHd >= MAX_DATA_POINTS) ? MAX_DATA_POINTS - 1 : bufHd;
				
		SerIOCur = buf[bufHd];		
		SerIOSca = (int) (SerIOCur * VERT_SCALE);
		
		WinDrawGrayLine(xPos, POS_Y, xPos, POS_Y - SerIOSca);
		//WinDrawLine(xPos, POS_Y, xPos, POS_Y - SerIOSca);
		
		xPos--;		
		cnt--;		
	}
}


/*
 * EnableControl
 */
static void EnableControl (FormPtr Frm, Word ControlID, Boolean State) {
	ControlPtr 	cptr;
	
	cptr = (ControlPtr)(FrmGetObjectPtr(Frm, (FrmGetObjectIndex(Frm, ControlID))));
	if (State)
		CtlShowControl(cptr);
	else
		CtlHideControl(cptr);
}


/*
 * EnableField
 */
static void EnableField (FormPtr Frm, Word FieldID, Boolean State) {
	FieldPtr	fptr;
	fptr = (FieldPtr)(FrmGetObjectPtr(Frm, (FrmGetObjectIndex(Frm,FieldID))));
	if (State)
		FldDrawField(fptr);
	else
		FldEraseField(fptr);
}


/*
 * SerIOOpen
 */
static void SerIOOpen(void) {
	Err			Error;
	FormPtr 	Frm;
	
	Error = SerOpen(SerIORef, 0, 9600);
	if (Error) {
		FrmCustomAlert (AlertGenericAlert, strErrSerPortOpen, StrBlank, StrBlank);
		SerClearErr(SerIORef);
		return;
	}
	
	SerIOSettings.baudRate = 9600;
	SerIOSettings.ctsTimeout = serDefaultCTSTimeout;
	SerIOSettings.flags =  	serSettingsFlagStopBits1 |
							serSettingsFlagBitsPerChar8	|
							serSettingsFlagCTSAutoM |
							serSettingsFlagRTSAutoM;
	
	Error = SerSetSettings(SerIORef, &SerIOSettings);
	if (Error) {
		FrmCustomAlert (AlertGenericAlert, strErrSerSetup, StrBlank, StrBlank);
		SerClearErr(SerIORef);
		return;		
		}
	
	Frm = FrmGetFormPtr(HiLoMainForm);

	EnableControl(Frm, HiLoMainBtnExitButton, true);
	EnableControl(Frm, HiLoMainBtnGoButton, false);

	EnableField(Frm, HiLoMainFldLoField, true);
	EnableField(Frm, HiLoMainFldHiField, true);
	EnableField(Frm, HiLoMainFldAvgField, true);
	EnableField(Frm, HiLoMainFldRxCountField, true);
	EnableField(Frm, HiLoMainFldLastField, true);
	EnableField(Frm, HiLoMainFldScaleField, true);
	
	FocusField(HiLoMainFldScaleField);
	
	SerIOConnected = true;
}


/*
 * FocusField
 */
static void FocusField(Word fldID) {
	FormPtr 	Frm;
	FieldPtr	fptr;
	
	Frm = FrmGetFormPtr(HiLoMainForm);
	fptr = (FieldPtr)(FrmGetObjectPtr(Frm, (FrmGetObjectIndex(Frm, fldID))));
	FldGrabFocus(fptr);
}


/*
 * SerIOClose
 */
static void SerIOClose(void) {
	char		BufSmall[17];
	char		BufLarge[80];
	Err			Error;
	
	if (!SerIOConnected)
		return;
		
	Error = SerClose (SerIORef);
	if (Error) {
		StrIToA(BufSmall, Error);
		StrCopy(BufLarge, strErrSerClose);
		StrCat(BufLarge, BufSmall);
		FrmCustomAlert (AlertGenericAlert, BufLarge, StrBlank, StrBlank);
	}
}


/*
 * CreateChartRec
 */
static void CreateChartRecRect(void) {
	
	ChartRecRect.topLeft.x = CHART_RECT_LEFT;
	ChartRecRect.topLeft.y = CHART_RECT_TOP;
	ChartRecRect.extent.x = CHART_RECT_WIDTH;
	ChartRecRect.extent.y = CHART_RECT_HEIGHT;
	
	WinDrawRectangleFrame(simpleFrame, &ChartRecRect);
}


/*
 * StopApplication
 */
static void StopApplication(void) {
	SerIOClose();
}


/*
 * MainFormHandleEvent
 */
static Boolean MainFormHandleEvent(EventPtr event) {
	Boolean		handled = false;
	EventType	newEvent;
	
	EvtResetAutoOffTimer();		// ensure serial port stays open
	
	if (event->eType == ctlSelectEvent) {
		switch (event->data.ctlEnter.controlID) {
			case HiLoMainBtnExitButton:
			   	MemSet(&newEvent, sizeof(EventType), 0);
			   	newEvent.eType = appStopEvent;
			   	EvtAddEventToQueue(&newEvent);
   				break;

   			case HiLoMainBtnGoButton:
				SerIOOpen();
   				break;

   		}
		handled = true;
	}

	else if (event->eType == nilEvent) {
		RcvCount  = 0;
		SerIOReceive ();
		if (RcvCount > 0)
			DisplayChart(Data, BufHead, RcvCount);

		handled = true;
	}
		
	return handled;
}


/*
 * EventLoop
 */
static void EventLoop(void) {
	EventType	event;
	//Word		error;
	
	do {
		EvtGetEvent(&event, evtWaitForever);
		if (! SysHandleEvent(&event))
			//if (! MenuHandleEvent(CurrentMenu, &event, &error))
				if (! MainFormHandleEvent(&event))
					FrmHandleEvent(FrmGetActiveForm(), &event);
	}
	while (event.eType != appStopEvent);
}


/*
 * StartApplication
 */
static void StartApplication(void) {
	Err			Error;
	FormPtr		Frm;

	// Initialize and draw the main form.
	Frm = FrmInitForm(HiLoMainForm);	
	FrmSetActiveForm(Frm);
	FrmDrawForm(Frm);

	EnableControl(Frm, HiLoMainBtnExitButton, true);
	EnableField(Frm, HiLoMainFldLoField, false);
	EnableField(Frm, HiLoMainFldHiField, false);
	EnableField(Frm, HiLoMainFldAvgField, false);
	EnableField(Frm, HiLoMainFldRxCountField, false);
	EnableField(Frm, HiLoMainFldLastField, false);
	EnableField(Frm, HiLoMainFldScaleField, false);

	Error = SysLibFind("Serial Library", &SerIORef);
	if (Error) {
		FrmCustomAlert (AlertGenericAlert, strErrSerLibOpen, StrBlank, StrBlank);
		SerClearErr(SerIORef);
		return;		
	}
	
	CreateChartRecRect();
	MemSet(Data, MAX_DATA_POINTS, 0);
}


/*
 * PilotMain
 */
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags) {
	if (cmd == sysAppLaunchCmdNormalLaunch)	{
		StartApplication();
		EventLoop();
		StopApplication();
	}
	return 0;
}
