#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#ifdef X86
#define X86OPT
#endif

#define c_maxwordlen	64
#define c_maxopcount	64
#define c_maxunget	(c_maxwordlen + 4)
#define c_maxstacksize	((c_maxopcount + 4) * 4)

#define c_CODE_OP_B	0
#define c_CODE_OP_L	1
#define c_CODE_OP_R	2
#define c_CODE_OP_MAX	2
#define c_CODE_PUSH_MEM	3
#define c_CODE_PUSH_IMM	4
#define c_CODE_POP_REG	5
#define c_CODE_JZ	6
#define c_CODE_JUMP	7
#ifdef FUTURE
#define c_CODE_CALL	8
#endif
#define c_CODE_ENTER	9
#define c_CODE_LEAVE	10

char *c_ops[] = {
  "0<b=", "!>b[",
  "8>b+", "8>b-", "9>b*", "9>b/", "9>b%",
  "5>b|", "5>b^", "6>b&", "A<l~",
  "7>b<<", "7>b>>",
  "A<l-",
  "0<b+=", "0<b-=", "0<b*=", "0<b/=", "0<b%=",
  "0<b|=", "0<b^=", "0<b&=",
  "0<b<<=", "0<b>>=",
  "4>b==", "4>b!=", "4>b>", "4>b<", "4>b>=", "4>b<=",
  "1>b||", "2>b&&", "3<l!",
  "A>l++", "A>l--", "A>r++", "A>r--",
  NULL
};

int c_pass;

unsigned char *c_code;
int c_ofs, c_codesize;

#ifdef X86
unsigned char *c_codeX86;
int c_codesizeX86;
#endif

long *c_data;
int c_datasize;

long c_stack[c_maxstacksize];
int c_stackptr;

long c_reg;

char c_ungetbuf[c_maxunget];
int c_ungetcount;

#define c_OK		0
#define c_UNKNOWN	1
#define c_UNEXPECTED	2
#define c_COUNT		3
#define c_TOOLONG	4
#define c_TOOCOMPLEX	5
#define c_ARRAYSIZE	6
#define c_DUPE		7
#define c_NOTINFUNC	8
#define c_NESTEDFUNC	9
#define c_EOF		10
#define c_INTERNAL	11
#define c_NOMEM		12
int c_error;

char *c_messages[] = {
  "no error",
  "unknown identifier",
  "unexpected character",
  "error in expression",
  "identifier too long",
  "expression too complex",
  "invalid array size",
  "identifier already defined",
  "not in a function",
  "nested functions not supported",
  "unexpected end of source",
  "internal compiler error",
  "not enough free memory"
};

char c_isident[256];

struct c_VAR {
  struct c_VAR *next;
  char *name;
  int ofs;
};

struct c_FUNC {
  struct c_FUNC *next;
  char *name;
  int ofs;
#ifdef X86
  int ofsX86;
#endif
};

struct c_VAR *c_globalvars = NULL;
struct c_FUNC *c_globalfuncs = NULL;

int (*c_ext_getchar)();
void (*c_ext_rewind)();

void c_init() {
  int c;

  c_datasize = c_ofs = c_ungetcount = c_error = 0;
  for (c = 0; c < 256; c++)
    c_isident[c] = (isalpha(c) || isdigit(c) || c == '_') != 0;

  c_ext_rewind();
}

void c_ungetchar(char c) {
  if (c_ungetcount >= c_maxunget) c_error = c_INTERNAL;
  else c_ungetbuf[c_ungetcount++] = c;
}

char c_getchar(int quote) {
  register int c;
  int space = 0;

  do {
    if (c_ungetcount) c = c_ungetbuf[--c_ungetcount]; else
    if ((c = c_ext_getchar()) <= 0) {
      c = ' '; c_error = c_EOF; break;
    }
    if (quote) break;
    if (c <= ' ') space = -1;
  } while (c <= ' ');
  if (space) {
    c_ungetchar(c); c = ' ';
  }

  return c;
}

char *c_getword() {
  static char word[c_maxwordlen];
  register int pos = 0;

  while (c_isident[word[pos++] = c_getchar(0)])
  if (pos >= c_maxwordlen) {
    c_error = c_TOOLONG; break;
  }

  if (pos != 1) c_ungetchar(word[--pos]);
  word[pos] = 0;

  return word;
}

long c_getint(char *w) {
  long val;

  if (*w == '\'') {
    if ((val = c_getchar(1)) == '\'') c_error = c_UNEXPECTED; else
    if (val == '\\') val = c_getchar(1);
    if (c_getchar(1) != '\'') c_error = c_UNEXPECTED;
  } else
  if (sscanf(w, w[0] == '0' ? w[1] == 'x' ? "%x" : "%o" : "%d", &val) != 1)
    c_error = c_UNEXPECTED;

  return val;
}

void c_putbyte(int ofs, char val) {
  if (c_pass) c_code[ofs] = val;
}

void c_addbyte(char val) {
  if (c_pass) c_code[c_ofs] = val; c_ofs++;
}

void c_putint(int ofs, long val) {
  if (c_pass) {
    c_code[ofs++] = val; c_code[ofs++] = val >> 8;
    c_code[ofs++] = val >> 16; c_code[ofs++] = val >> 24;
  }
}

void c_addint(long val) {
  c_putint(c_ofs, val); c_ofs += 4;
}

long c_readint() {
  c_ofs += 4;
  return
    (long)c_code[c_ofs - 4] + ((long)c_code[c_ofs - 3] << 8) +
    ((long)c_code[c_ofs - 2] << 16) + ((long)c_code[c_ofs - 1] << 24);
}

int c_block(char term, struct c_VAR *globalvars) {
  register char c, *w;
  char *s;
  char savedc;
  int savedo, fixup;
  long val;
  struct c_VAR *var, *localvars = globalvars;
  struct c_FUNC *func;

  while (*(w = c_getword()) != term) {
    if (c_error) {
      if (!term && c_error == c_EOF) c_error = c_OK;
      break;
    }
    if (*w == ' ') continue;

    if (*w == '{')
    if (!term) c_error = c_NOTINFUNC; else
    if (term == ';') term = '}'; else
    if (c_block('}', localvars)) break; else continue; else

/* void, int */
    if (!strcmp(w, "void") || !strcmp(w, "int")) {
      if (c_getchar(0) != ' ' || !c_isident[*(w = c_getword())] || isdigit(*w))
        c_error = c_UNEXPECTED;
      if (c_error) break;
      do if (*w != ' ') {
        if (c_isident[*w] && !isdigit(*w)) {
          if ((c = c_getchar(0)) == ' ') c = c_getchar(0);
          if (c == '(') {
            if (term) c_error = c_NESTEDFUNC; else {
              func = c_globalfuncs;
              if (!(c_globalfuncs = (struct c_FUNC *)malloc(sizeof(struct c_FUNC))) ||
                !(c_globalfuncs->name = (char *)malloc(strlen(w) + 1))) c_error = c_NOMEM;
              if (c_error) break;
              c_globalfuncs->next = func; strcpy(c_globalfuncs->name, w);
              c_globalfuncs->ofs = c_ofs;
#ifdef X86
              c_globalfuncs->ofsX86 = -1;
#endif

              if ((c = c_getchar(0)) == ' ') c = c_getchar(0);
              if (c != ')') c_error = c_UNEXPECTED;
              if ((c = c_getchar(0)) == ' ') c = c_getchar(0);
              if (c != '{') c_error = c_UNEXPECTED; else
              if (!c_error) {
                c_addbyte(c_CODE_ENTER);
                c_block('}', localvars);
                c_addbyte(c_CODE_LEAVE);
              }
            }
            break;
          } else {
            var = localvars;
            while (var != globalvars) {
              if (!strcmp(var->name, w)) c_error = c_DUPE;
              var = var->next;
            }
            var = localvars;
            if (!(localvars = (struct c_VAR *)malloc(sizeof(struct c_VAR))) ||
              !(localvars->name = (char *)malloc(strlen(w) + 1))) c_error = c_NOMEM;
            if (c_error) break;
            localvars->next = var; strcpy(localvars->name, w);
            localvars->ofs = c_datasize++;

            if (c == '[') {
              c_datasize += (val = c_getint(c_getword())) - 1;
              if (c_error) break;
              if (val < 1) c_error = c_ARRAYSIZE;
              if (val > 0x7FFFFFFF / sizeof(long)) c_error = c_ARRAYSIZE;
              if (c_error) break;
              if ((c = c_getchar(0)) == ' ') c = c_getchar(0);
              if (c != ']') c_error = c_UNEXPECTED;
              if (c_error) break;
              if ((c = c_getchar(0)) == ' ') c = c_getchar(0);
            }
            if (c == ';') break;
            if (c != ',') c_error = c_UNEXPECTED;
          }
        } else c_error = c_UNEXPECTED;
        if (c_error) break;
      } while (*(w = c_getword()) != ';');
      if (c_error) break;

/* if, while */
    } else if (!strcmp(w, "if") || !strcmp(w, "while"))
    if (!term) c_error = c_NOTINFUNC; else {
      savedc = *w; savedo = c_ofs;
      while ((c = c_getchar(0)) == ' ');
      if (c != '(') c_error = c_UNEXPECTED;
      if (c_error || c_block(')', localvars)) break;
      c_addbyte(c_CODE_JZ); fixup = c_ofs; c_ofs += 4;
      if (c_error || c_block(';', localvars)) break;
      if (savedc == 'w') {
        c_addbyte(c_CODE_JUMP); c_addint(savedo);
      }
      if (savedc == 'i') {
        while (*(w = c_getword()) == ' ' && !c_error);
        if (!strcmp(w, "else")) {
          c_addbyte(c_CODE_JUMP); c_ofs += 4;
          c_putint(fixup, c_ofs); fixup = c_ofs - 4;
          if (c_error || c_block(';', localvars)) break;
        } else {
          s = w + strlen(w);
          while (s > w) c_ungetchar(*--s);
        }
      }
      c_putint(fixup, c_ofs);
      c_ungetchar(';');

/* expression */
    } else if (*w != ';') {
      unsigned char stack[c_maxopcount];
      int stackptr = 0, balance = -1;
      int op, bestop, wlen, left = 0;

      if (term == ')') stack[stackptr++] = 0xFF;
      do {
        if (*w == ')' || *w == ']' || *w == ';' || *w == term) {
          while (stackptr) {
            if (stack[--stackptr] == 0xFF) break;
            switch (c_ops[stack[stackptr]][2]) {
              case 'l':
                c_addbyte(c_CODE_OP_L); break;
              case 'r':
                c_addbyte(c_CODE_OP_R); break;
              default:
                c_addbyte(c_CODE_OP_B); balance--;
            }
            c_addbyte(stack[stackptr]);
            if (stack[stackptr] == 1) break;
          }
          if (*w == ')' && stack[stackptr] != 0xFF ||
            *w == ']' && stack[stackptr] != 1 ||
            (*w == ';' || (term != ')' && *w == term)) && stackptr)
            c_error = c_COUNT;
          if (c_error || !stackptr && *w == term) break;
          left = -1;
        } else if (isdigit(*w) || *w == '\'') {
          c_addbyte(c_CODE_PUSH_IMM); c_addint(c_getint(w));
          left = -1; balance++;
        } else if (*w == '(' || *w == '[') {
          if (stackptr >= c_maxopcount) c_error = c_TOOCOMPLEX; else
          stack[stackptr++] = *w == '(' ? 0xFF : 1;
          left = 0;
        } else if (*w != ' ') {
          if (var = c_isident[*w] ? localvars : NULL) do
            if (!strcmp(var->name, w)) break;
          while (var = var->next);
          if (var) {
            c_addbyte(c_CODE_PUSH_MEM); c_addint(var->ofs);
            left = -1; balance++;
          } else {
            if ((wlen = strlen(w)) < 2) {
              if (wlen < 2) w[1] = c_getchar(0); w[2] = c_getchar(0);
              w[3] = 0;
            }
            bestop = -1; op = 0;
            do
              if ((left && c_ops[op][2] != 'l' || !left && c_ops[op][2] == 'l') &&
                !memcmp(c_ops[op] + 3, w, strlen(c_ops[op]) - 3))
              if (bestop < 0 || strlen(c_ops[op]) > strlen(c_ops[bestop])) bestop = op;
            while (c_ops[++op]);
            if (wlen < 2 && (bestop < 0 || strlen(c_ops[bestop]) < 6)) {
              c_ungetchar(w[2]);
              if (bestop < 0 || !c_ops[bestop][4]) c_ungetchar(w[1]);
            }
            if (bestop < 0) c_error = c_isident[*w] ? c_UNKNOWN : c_UNEXPECTED;
            else {
              while (stackptr && stack[stackptr - 1] != 0xFF &&
                c_ops[stack[stackptr - 1]][0] >= c_ops[bestop][0] +
                (c_ops[stack[stackptr - 1]][1] == '<' ? 1 : 0))
              {
                switch (c_ops[stack[--stackptr]][2]) {
                  case 'l':
                    c_addbyte(c_CODE_OP_L); break;
                  case 'r':
                    c_addbyte(c_CODE_OP_R); break;
                  default:
                    c_addbyte(c_CODE_OP_B); balance--;
                }
                c_addbyte(stack[stackptr]);
              }
              if (stackptr >= c_maxopcount) c_error = c_TOOCOMPLEX; else
              left = (c_ops[stack[stackptr++] = bestop][2]) == 'r';
            }
          }
          if (c_error) break;
        }
        if (c_error || *w == ';' || *w == term && *w != ')') break;
        w = c_getword();
      } while (1);
      if (c_error) break;
      if (stackptr || balance) c_error = c_COUNT;
      c_addbyte(c_CODE_POP_REG);

      if (!term && !c_error) c_error = c_NOTINFUNC;
      if (*w == term) break;
    }

    if (c_error) break;
  }

  while ((var = localvars) != globalvars) {
    localvars = localvars->next;
    free(var);
  }

  return c_error;
}

#ifdef X86
int c_toX86();
#endif

int c_compile() {
  for (c_pass = 0; c_pass < 2; c_pass++) {
    c_init();
    c_block(0, c_globalvars);
    if (c_error) return c_error;
    if (!c_pass)
    if (!(c_code = (unsigned char *)malloc(c_codesize = c_ofs)))
      return c_error = c_NOMEM;
  }
  if (!(c_data = (long *)malloc(c_datasize * sizeof(long))))
    c_error = c_NOMEM;

#ifdef X86
  if (!c_error) c_toX86();
#endif

  return c_error;
}

#ifdef DEBUG
char *c_disasm() {
  static char cmd[32];

  sprintf(cmd, "%08x: ", c_ofs);
  switch (c_code[c_ofs++]) {
    case c_CODE_OP_B:
      sprintf(cmd + 10, "OPB\t%s", c_ops[c_code[c_ofs++]] + 3); break;
    case c_CODE_OP_L:
      sprintf(cmd + 10, "OPL\t%s", c_ops[c_code[c_ofs++]] + 3); break;
    case c_CODE_OP_R:
      sprintf(cmd + 10, "OPR\t%s", c_ops[c_code[c_ofs++]] + 3); break;
    case c_CODE_PUSH_MEM:
      sprintf(cmd + 10, "PUSH\t[%08x]", c_readint()); break;
    case c_CODE_PUSH_IMM:
      sprintf(cmd + 10, "PUSH\t%08x", c_readint()); break;
    case c_CODE_POP_REG:
      strcpy(cmd + 10, "POP\tREG"); break;
    case c_CODE_JZ:
      sprintf(cmd + 10, "JZ\t%08x", c_readint()); break;
    case c_CODE_JUMP:
      sprintf(cmd + 10, "JUMP\t%08x", c_readint()); break;
#ifdef FUTURE
    case c_CODE_CALL:
      sprintf(cmd + 10, "CALL\t%08x", c_readint()); break;
#endif
    case c_CODE_ENTER:
      strcpy(cmd + 10, "ENTER"); break;
    case c_CODE_LEAVE:
      strcpy(cmd + 10, "LEAVE"); break;
    default:
      sprintf(cmd + 10, "DB\t%02x", c_code[c_ofs - 1]);
  }

  return cmd;
}
#endif

#ifndef X86
void c_func_assign() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] = c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_index() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] = c_data[c_stack[c_stackptr - 1] += c_stack[c_stackptr]];
}

void c_func_add() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] += c_stack[c_stackptr];
}

void c_func_add_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] += c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_sub() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] -= c_stack[c_stackptr];
}

void c_func_sub_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] -= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_mul() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] *= c_stack[c_stackptr];
}

void c_func_mul_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] *= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_div() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] /= c_stack[c_stackptr];
}

void c_func_div_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] /= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_mod() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] %= c_stack[c_stackptr];
}

void c_func_mod_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] %= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_or_i() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] |= c_stack[c_stackptr];
}

void c_func_or_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] |= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_xor_i() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] ^= c_stack[c_stackptr];
}

void c_func_xor_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] ^= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_and_i() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] &= c_stack[c_stackptr];
}

void c_func_and_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] &= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_not_i() {
  c_stack[c_stackptr - 2] = ~c_stack[c_stackptr - 2];
}

void c_func_shl() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] <<= c_stack[c_stackptr];
}

void c_func_shl_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] <<= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_shr() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] >>= c_stack[c_stackptr];
}

void c_func_shr_a() {
  c_stack[c_stackptr - 4] = c_data[c_stack[c_stackptr - 3]] >>= c_stack[c_stackptr - 2];
  c_stackptr -= 2;
}

void c_func_neg() {
  c_stack[c_stackptr - 2] = -c_stack[c_stackptr - 2];
}

void c_func_eq() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] = c_stack[c_stackptr - 2] == c_stack[c_stackptr];
}

void c_func_gt() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] = c_stack[c_stackptr - 2] > c_stack[c_stackptr];
}

void c_func_lt() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] = c_stack[c_stackptr - 2] < c_stack[c_stackptr];
}

void c_func_ge() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] = c_stack[c_stackptr - 2] >= c_stack[c_stackptr];
}

void c_func_le() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] = c_stack[c_stackptr - 2] <= c_stack[c_stackptr];
}

void c_func_and_b() {
  c_stackptr -= 2;
  c_stack[c_stackptr - 2] = c_stack[c_stackptr - 2] && c_stack[c_stackptr];
}

void c_func_not_b() {
  c_stack[c_stackptr - 2] = !c_stack[c_stackptr - 2];
}

void c_func_inc_l() {
  c_data[c_stack[c_stackptr - 1]] = ++c_stack[c_stackptr - 2];
}

void c_func_dec_l() {
  c_data[c_stack[c_stackptr - 1]] = --c_stack[c_stackptr - 2];
}

void c_func_inc_r() {
  c_data[c_stack[c_stackptr - 1]] = c_stack[c_stackptr - 2] + 1;
}

void c_func_dec_r() {
  c_data[c_stack[c_stackptr - 1]] = c_stack[c_stackptr - 2] - 1;
}

void (*c_ops_func[])() = {
  c_func_assign, c_func_index,
  c_func_add, c_func_sub, c_func_mul, c_func_div, c_func_mod,
  c_func_or_i, c_func_xor_i, c_func_and_i, c_func_not_i,
  c_func_shl, c_func_shr,
  c_func_neg,
  c_func_add_a, c_func_sub_a, c_func_mul_a, c_func_div_a, c_func_mod_a,
  c_func_or_a, c_func_xor_a, c_func_and_a,
  c_func_shl_a, c_func_shr_a,
  c_func_eq, c_func_sub, c_func_gt, c_func_lt, c_func_ge, c_func_le,
  c_func_or_i, c_func_and_b, c_func_not_b,
  c_func_inc_l, c_func_dec_l, c_func_inc_r, c_func_dec_r
};

void c_exec_OP() {
  c_ops_func[c_code[c_ofs++]]();
}

void c_exec_PUSH_MEM() {
  register long v;
  c_stack[c_stackptr++] = c_data[v = c_readint()];
  c_stack[c_stackptr++] = v;
}

void c_exec_PUSH_IMM() {
  c_stack[c_stackptr++] = c_readint();
  c_stack[c_stackptr++] = 0x80000000;
}

void c_exec_POP_REG() {
  c_reg = c_stack[c_stackptr -= 2];
}

void c_exec_JZ() {
  if (c_reg) c_ofs += 4; else c_ofs = c_readint();
}

void c_exec_JUMP() {
  c_ofs = c_readint();
}

#ifdef FUTURE
void c_exec_CALL() {
  c_stack[c_stackptr++] = c_ofs + 4;
  c_stack[c_stackptr++] = 0x80000000;
  c_ofs = c_readint();
}
#endif

void c_exec_ENTER() {
}

void c_exec_LEAVE() {
  c_ofs = c_stack[c_stackptr -= 2];
}

void (*c_exec[])() = {
  c_exec_OP,
  c_exec_OP,
  c_exec_OP,
  c_exec_PUSH_MEM,
  c_exec_PUSH_IMM,
  c_exec_POP_REG,
  c_exec_JZ,
  c_exec_JUMP,
#ifdef FUTURE
  c_exec_CALL,
#else
  (void (*)())NULL,
#endif
  c_exec_ENTER,
  c_exec_LEAVE
};

int c_execute(char *name) {
  struct c_FUNC *func;

  c_reg = 0;
  c_stack[0] = 0xDEADBEEF; c_stackptr = 2;

  if (func = c_globalfuncs) do
    if (!strcmp(func->name, name)) break;
  while (func = func->next);
  if (func) c_ofs = func->ofs; else return -1;

  do {
#ifdef DEBUG
    {
      register long val;
      val = c_ofs; puts(c_disasm()); c_ofs = val;
    }
#endif
    c_exec[c_code[c_ofs++]]();
  } while (c_ofs != 0xDEADBEEF);

  return 0;
}

#else

char *c_ops_X86[] = {
  "N\x89\x0B\x8B\xC1",
  "N\x8D\x1C\x8B\x8B\x03",

  "Y\x01\xC8",
  "Y\x2B\xC1",
  "Y\xF7\xE9",
  "Y\x99\xF7\xF9",
  "N\x99\xF7\xF9\x8B\xC2",

  "Y\x09\xC8",
  "Y\x31\xC8",
  "Y\x21\xC8",
  "Y\xF7\xD0",

  "N\xD3\xE0",
  "N\xD3\xE8",

  "Y\xF7\xD8",

  "Y\x01\xC8\x89\x03",
  "Y\x2B\xC1\x89\x03",
  "Y\xF7\xE9\x89\x03",
  "Y\x99\xF7\xF9\x89\x03",
  "N\x99\xF7\xF9\x8B\xC2\x89\x03",

  "Y\x09\xC8\x89\x03",
  "Y\x31\xC8\x89\x03",
  "Y\x21\xC8\x89\x03",

  "N\xD3\xE0\x89\x03",
  "N\xD3\xE8\x89\x03",

  "Y\x2B\xC1\x83\xE8\x01\x19\xC0",
  "Y\x2B\xC1",
  "Y\x8B\xD0\x8B\xC1\x2B\xC2\x99\x23\xC2",
  "Y\x2B\xC1\x99\x23\xC2",
  "Y\x8B\xD0\x8B\xC1\x48\x2B\xC2\x99\x23\xC2",
  "Y\x48\x2B\xC1\x99\x23\xC2",

  "Y\x09\xC8",
  "Y\x83\xC0\xFF\x19\xC0\x83\xC1\xFF\x19\xC9\x21\xC8",
  "Y\x83\xE8\x01\x19\xC0",

  "Y\x40\x89\x03",
  "Y\x48\x89\x03",
  "N\xFF\x02",
  "N\xFF\x0A",
};

struct c_FIXUP {
  struct c_FIXUP *next;
  int src, dst;
};

int c_ofsX86;

void c_putbyteX86(int ofs, char val) {
  if (c_pass) c_codeX86[ofs] = val;
}

void c_addbyteX86(char val) {
  if (c_pass) c_codeX86[c_ofsX86] = val; c_ofsX86++;
}

void c_putintX86(int ofs, long val) {
  if (c_pass) {
    c_codeX86[ofs++] = val; c_codeX86[ofs++] = val >> 8;
    c_codeX86[ofs++] = val >> 16; c_codeX86[ofs++] = val >> 24;
  }
}

void c_addintX86(long val) {
  c_putintX86(c_ofsX86, val); c_ofsX86 += 4;
}

int c_toX86() {
  int val, flags = -1, fullop = -1;
  char *p;
  struct c_FIXUP *list = NULL, *current, *last;
  struct c_FUNC *func;

  for (c_pass = 0; c_pass < 2; c_pass++) {
    c_ofs = 0; c_ofsX86 = 0;

    do {
      if (c_pass) {
        last = NULL;
        if (current = list) do {
          if (current->dst == c_ofs) {
            c_putintX86(current->src, c_ofsX86 - current->src - 4);
            if (last) last->next = current->next; else list = current->next;
            free(current);
            if (!(current = last)) if (current = list) continue; else break;
          }
          last = current; current = current->next;
        } while (current);

        if (func = c_globalfuncs) do
          if (func->ofs == c_ofs) func->ofsX86 = c_ofsX86;
        while (func = func->next);
      }

      switch (c_code[c_ofs++]) {
        case c_CODE_OP_B:
          if (fullop) {
            c_addbyteX86(0x5A); c_addbyteX86(0x59);
          }
          fullop = -1;
          c_addbyteX86(0x5B); c_addbyteX86(0x58);
          flags = *(p = c_ops_X86[c_code[c_ofs++]]) != 'Y';
          while (*++p) c_addbyteX86(*p);
#ifdef X86OPT
          if (c_code[c_ofs] == c_CODE_POP_REG) {
            c_ofs++; break;
          }
          if (c_code[c_ofs] <= c_CODE_OP_MAX) {
            if (c_code[c_ofs] != c_CODE_OP_L) c_addintX86(0xC189DA89);
            fullop = 0; break;
          }
#endif
          c_addbyteX86(0x50); c_addbyteX86(0x53);
          break;
        case c_CODE_OP_R:
          if (fullop) {
            c_addbyteX86(0x5A); c_addbyteX86(0x59);
          }
          fullop = -1;
          flags = *(p = c_ops_X86[c_code[c_ofs++]]) != 'Y';
          while (*++p) c_addbyteX86(*p);
#ifdef X86OPT
          if (c_code[c_ofs] == c_CODE_POP_REG) {
            c_addbyteX86(0x8B); c_addbyteX86(0xC1);
            c_ofs++; break;
          }
          if (c_code[c_ofs] <= c_CODE_OP_MAX) {
            if (c_code[c_ofs] == c_CODE_OP_L) c_addintX86(0xC18BDA8B);
            fullop = 0; break;
          }
#endif
          c_addbyteX86(0x51); c_addbyteX86(0x52);
          break;
        case c_CODE_OP_L:
          if (fullop) {
            c_addbyteX86(0x5B); c_addbyteX86(0x58);
          }
          fullop = -1;
          flags = *(p = c_ops_X86[c_code[c_ofs++]]) != 'Y';
          while (*++p) c_addbyteX86(*p);
#ifdef X86OPT
          if (c_code[c_ofs] == c_CODE_POP_REG) {
            c_ofs++; break;
          }
          if (c_code[c_ofs] <= c_CODE_OP_MAX) {
            if (c_code[c_ofs] != c_CODE_OP_L) c_addintX86(0xC189DA89);
            fullop = 0; break;
          }
#endif
          c_addbyteX86(0x50); c_addbyteX86(0x53);
          break;
        case c_CODE_PUSH_MEM:
          val = (int)&c_data[c_readint()];
#ifdef X86OPT
          if (c_code[c_ofs] <= c_CODE_OP_MAX) {
            c_addbyteX86(c_code[c_ofs] == c_CODE_OP_L ? 0xBB : 0xBA);
            c_addintX86(val);
            c_addbyteX86(0x8B);
            c_addbyteX86(c_code[c_ofs] == c_CODE_OP_L ? 0x05 : 0x0D);
            c_addintX86(val);
            fullop = 0; break;
          }
#endif
          c_addbyteX86(0xFF); c_addbyteX86(0x35); c_addintX86(val);
          c_addbyteX86(0x68); c_addintX86(val);
          break;
        case c_CODE_PUSH_IMM:
          val = c_readint();
#ifdef X86OPT
          if (c_code[c_ofs] == c_CODE_POP_REG) {
            c_addbyteX86(0xB8); c_addintX86(val);
            flags = -1; c_ofs++; break;
          }
          if (c_code[c_ofs] <= c_CODE_OP_MAX) {
            c_addbyteX86(c_code[c_ofs] == c_CODE_OP_L ? 0xB8 : 0xB9);
            c_addintX86(val);
            fullop = 0; break;
          }
#endif
          c_addbyteX86(0x68); c_addintX86(val);
          c_addbyteX86(0x50);
          break;
        case c_CODE_POP_REG:
          c_addbyteX86(0x58); c_addbyteX86(0x58);
          flags = -1;
          break;
        case c_CODE_JZ:
        case c_CODE_JUMP:
#ifdef FUTURE
        case c_CODE_CALL:
#endif
          if (c_code[c_ofs - 1] == c_CODE_JZ) {
#ifdef X86OPT
            if (flags) {
              c_addbyteX86(0x85); c_addbyteX86(0xC0);
            }
#else
            c_addbyteX86(0x85); c_addbyteX86(0xC0);
#endif
            c_addbyteX86(0x0F); c_addbyteX86(0x84);
          } else
#ifdef FUTURE
          c_addbyteX86(c_code[c_ofs - 1] == c_CODE_JUMP ? 0xE9 : 0xE8);
#else
          c_addbyteX86(0xE9);
#endif
          if (!c_pass) {
            current = list;
            if (!(list = (struct c_FIXUP *)malloc(sizeof(struct c_FIXUP))))
              return c_error = c_NOMEM;
            list->next = current; list->src = c_ofsX86; list->dst = c_readint();
          } else c_readint();
          c_ofsX86 += 4;
          break;
        case c_CODE_ENTER:
          c_addbyteX86(0x60); break;
        case c_CODE_LEAVE:
          c_addbyteX86(0x61); c_addbyteX86(0xC3); break;
        default:
          return c_error = c_INTERNAL;
      }
    } while (c_ofs < c_codesize);

    if (c_error) return c_error;
    if (!c_pass)
    if (!(c_codeX86 = (unsigned char *)malloc(c_codesizeX86 = c_ofsX86)))
      return c_error = c_NOMEM;
  }
  if (list) return c_error = c_INTERNAL;

  if (func = c_globalfuncs) do
    if (func->ofsX86 < 0) c_error = c_INTERNAL;
  while (func = func->next);

  return c_error;
}

int c_execute(char *name) {
  struct c_FUNC *func;

  if (func = c_globalfuncs) do
    if (!strcmp(func->name, name)) break;
  while (func = func->next);
  if (!func) return -1;

  ((void (*)())(c_codeX86 + func->ofsX86))();

  return 0;
}
#endif
