/*

#######  #####   #####   #####   ####   ##                TFOS
##      ##   ## ##   ## ##   ##   ##    ##              
##      ##   ## ##      ##        ##    ##
#####   ##   ##  #####   #####    ##    ##		TurboFossil
##      ##   ##      ##      ##   ##    ##
##      ##   ## ##   ## ##   ##   ##    ##                 v0.01
##       #####   #####   #####   ####   #######


######     ##                ##    ##
##   ##                      ##
##   ##   ###   # ####       ##   ###   # ####   #####   ######
######     ##   ##   ##  ######    ##   ##   ## ##   ## ##
##   ##    ##   ##   ## ##   ##    ##   ##   ## ##   ##  #####
##   ##    ##   ##   ## ##   ##    ##   ##   ##  ######      ##
######    ####  ##   ##  #####    ####  ##   ##      ## ######
						 #####

Voor TurboC 2.0, door Michiel Overtoom

*/

#include <dos.h>
#include <stdio.h>
#include <string.h>

/* Functie: Zet fossil hot op een commport, maak DTR hoog
	    Laat de huidige baudrate instelling ongewijzigd

   Inputs:  port bevat het poortnr 0...
	    ctrlcflag bevat pointer naar byte die Ctrl-C's registeert
		      is deze NULL, dan wordt deze optie niet geinstald

   Returns: YES/NO of geslaagd of niet;
	    FosMaxFunc (int) bevat aantal fossilfuncs (excl. Appl. Appendages)
	    FosVersion (int) bevat fossilversienr.
	      (Deze laatste twee alleen geldig indien oproep geslaagd)
*/

int fos_InitDriver(int port,unsigned char *ctrlcflag,
		   int *FosMaxFunc,int *FosVersion)
{
	_DX=port; _BX=0;
	if (ctrlcflag) {
		_AX=_DS; _ES=_AX; _CX=ctrlcflag; _BX=0x4F50;
		}
	_AH=4; geninterrupt(0x14);
	if (_AX!=0x1954) return(0);
	*FosMaxFunc=_BL; *FosVersion=_BH;
	return(1);
}

/* Functie: Init een timer die afloopt na tenths tiende van secondes */
/* Returns: timertick value waarop tijd afloopt */
long fos_StartTimer(int tenths)
{
	unsigned long tick; unsigned int tl,th;
	_AH=0; geninterrupt(0x1a); tl=_DX; th=_CX; tick=th; tick<<=16 ; tick|=tl;
	return(tick+((long)tenths*182)/100);
}

/* Functie: Check of de huidige clocktickvalue groter is dan de opgegeven eindlimiet */
/* Returns: YES/NO alnaargelang timer afgelopen is */
int fos_Timeout(long start)
{
	unsigned long tick; unsigned int tl,th,ovflow;
	_AH=0; geninterrupt(0x1a); ovflow=_AL; tl=_DX; th=_CX; tick=th; tick<<=16 ; tick|=tl;
	if (ovflow) tick+=0x1800B0; /* Middernacht compensatie */
	return(tick>start);
}

/* Functie: Wacht een aantal tiendes van seconden */
fos_Wait(int tenths)
{
	long limit=fos_StartTimer(tenths);
	while(!fos_Timeout(limit));
}

/* Functie: Zet baudrate van een commport op 300,1200 of 2400, allen N,8,1 */
/* Returns: Status bits van fossil */
int fos_SetBaudRate(int port,int baud)
{
	unsigned int ax;
	switch (baud) {
		case  300: ax=0x40; break;
		case 1200: ax=0x80; break;
		case 2400: ax=0xA0; break;
		default: printf("fos_SetBaudRate: weird baudrate %d\n",baud); break;
		}
	ax|=0x03; /* N,8,1 */
	_AX=ax; _DX=port; geninterrupt(0x14);
	return(_AX);
}

/* Functie: Kijk of er een carrier op de betreffende poort is
   Returns: YES/NO alnaargelang de DCD pin */
int fos_Carrier(int port)
{
	_AH=3; _DX=port; geninterrupt(0x14); return(_AX&0x0080);
}

/* Functie: Schrijf string naar FOSSIL output buffer voor een commport
   Inputs:  portnr, string,
	    max. seconden (Geef 0 op om tijdslimietoptie niet te gebruiken)
   Returns: YES/NO alnaargelang gelukt of niet.
*/
int fos_Puts(int port,char *s,int secs)
{
	long limit;
	int writ,chleft=strlen(s);
	if (secs) {
		limit=fos_StartTimer(10*secs);
		}
	while(chleft>0) {
		_DI=s; _DX=port; _CX=chleft;
		_AX=_DS; _ES=_AX; _AH=0x19; geninterrupt(0x14); writ=_AX;
		chleft-=writ; s+=writ;
		if (secs) {
			if (fos_Timeout(limit)) return 0;
			}
		}
	return 1;
}

/* functie: Scan string van commport, geef max.lengte aan
	    Ignore linefeeds, stop op een return of timeout
	    (Niet bedoeld voor user-input, alleen voor modemmeldingen)
   returns: YES/NO gelukt of niet
*/
int fos_Gets(int port,char *s,int maxlen,int secs)
{
	int read=0; long limit;
	int c;
	if (secs) {
		limit=fos_StartTimer(10*secs);
		}
	for(;;) {
		if (secs) if (fos_Timeout(limit)) {s[read]=0; return(0);}
		_AH=0xc; _DX=port; geninterrupt(0x14);
		if (_AX==0xffff) continue;

		_AH=2; _DX=port; geninterrupt(0x14); c=_AX;
		if (c=='\n') continue;
		if (c=='\r') break;

		if (read<maxlen) s[read++]=c;
		}
	s[read]=0;
	return(1);
}


/* Functie: Zet de DTR pin
   Returns: niets
*/
void fos_DTR(int port,int dtr)
{
	_AL=dtr; _AH=6; _DX=port; geninterrupt(0x14);
}

/* Wait for external buffers to clear */
void fos_FlushOut(int port)
{
	_DX=port; _AH=8; geninterrupt(0x14);
}

/* Consumeer modemrespons-strings
   Returns: YES indien de verlangde string binnen de opgegeven tijd gezien werd
	    NO in alle andere gevallen */
int waitresp(int port,char *resp,int secs)
{
	char s[80];
	/* Haal netzolang strings van 't modem, totdat gewenste respons of timeout */
	for(;;) {
		if (!fos_Gets(port,s,79,secs)) return(0);
		if (strnicmp(s,resp,strlen(resp))==0) return(1);
		}
}

/* Functie: Stuur een commando naar het modem en wacht op respons
   Returns: YES indien respons binnen de gestelde tijdslimiet kwam
	    NO in alle andere gevallen
 */
int ModemCommand(int port,char *cmd,char *resp,int secs)
{
	int retries=3;
	while(retries) {
		fos_Puts(port,cmd,0);
		printf(" Modemcommand  %s\n",cmd);
		if (waitresp(port,resp,secs)) {printf(" Modemresponse %s\n",resp); return 1;}
		fos_Puts(port,"\r",0); /* Stuur maar een CR */
		fos_Wait(10);
		retries--;
		}
	printf("\n Modem didn't respond with '%s' to '%s' in %d secs\n",resp,cmd,secs);
	return 0;
}

/* Zoek het telefoonnummer van iemand in DIALBACK.CFG
   Returns: Empty telnr string indien naam niet voorkwam */
getphonenr(char *naam,char *telnr)
{
	FILE *f;
	char s[80],*p;
	int ls;
	strcpy(telnr,"");
	f=fopen("DIALBACK.CFG","rt");
	if (!f) return;
	while(!feof(f)) {
		if (!fgets(s,79,f)) break;
		ls=strlen(s); if (s[ls-1]=='\n') s[ls-1]=0;
		if (strnicmp(s,naam,strlen(naam))==0) {
			p=strchr(s,',');
			if (p) strcpy(telnr,p+1);
			break;
			}
		}
	fclose(f);
}

/* Hang het modem op */
hangup(int port)
{
	fos_DTR(port,0); fos_Wait(10); fos_DTR(port,1);
}

main(int argc, char *argv[])
{
	int vers,maxf=0;
	int port,baud,wait;
	char naam[80],telnr[80],s[160],t[80];
	if (argc<6) {
		printf("Usage:\n\nDialback <port> <baud> <firstnam> <lastnam> <maxwait>\n");
		exit(-1);
		}
	port=atoi(argv[1]);
	baud=atoi(argv[2]);
	if (baud==0) {
		printf(" Local operation not possible\n\n  Press any key...");
		bioskey(0);
		return(-1);
		}
	strcpy(naam,argv[3]); strcat(naam," "); strcat(naam,argv[4]);
	wait=atoi(argv[5]);

	if (!fos_InitDriver(port,0,&maxf,&vers)) {printf(" Fossil driver not found\n"); exit(-1);}
	fos_SetBaudRate(port,baud);
	getphonenr(naam,telnr);
	if (!*telnr) {
		fos_Puts(port," You are not authorized for dialback, returning to BBS...\r\n",0);
		printf(" User %s not authorized to use dialback\n",naam);
		fos_Wait(10);
		return(0);
		}
	sprintf(s," Dialback activated for user \"%s\" at tel.nr %s\r\n",naam,telnr);
	fos_Puts(port,s,0);
	printf("\n\033[1m%s\033[0m\n  DialBack Version 0.01, Software by Michiel Overtoom 1989\n\n",s);

	fos_Puts(port," Hanging up & calling you back. Bye!\r\n",0);
	fos_FlushOut(port);
	hangup(port);

	fos_Wait(20);
	if (!ModemCommand(port,"ATZ\r","OK",4)) exit(-1);
	fos_Wait(10);
	if (!ModemCommand(port,"ATE0V1X4H0\r","OK",4)) exit(-1);

	fos_SetBaudRate(port,baud);
	sprintf(s,"ATDP%s\r",telnr);
	if (baud>300) sprintf(t,"CONNECT %d",baud); else strcpy(t,"CONNECT");

	printf(" Dialling back\n");
	fos_Wait(10);
	if (!ModemCommand(port,s,t,wait)) {
		printf(" Dialback failed\n");
		hangup(port);
		fos_Wait(10);
		exit(-1);
		}
	fos_Puts(port,"\r\n Dialback succesful. Welcome back, returning you to the BBS...\r\n",0);
	printf(" Dialback succesful\n");
	fos_Wait(10);
	exit(0);
}