/* KEYBRD.C this file contain the
 * keyboard processing code, for the
 * MicroEMACS screen editor.
 */
#include        <stdio.h>
#include        "ed.h"
 
#define FLEN    14                      /* Maximum length of function name */
 
typedef struct  {
        short   k_code;                 /* Key code                     */
        int     (*k_fp)();              /* Routine to handle it         */
        char    k_mfunc[FLEN];          /* function name                */
}       KEYTAB;
 
extern  int     clearflag();            /* Clear changed buffer flag    */
extern  int     ctrlg();                /* Abort out of things          */
extern  int     quit();                 /* Quit                         */
extern  int     ctlxlp();               /* Begin macro                  */
extern  int     ctlxrp();               /* End macro                    */
extern  int     ctlxe();                /* Execute macro                */
extern  int     fileinsert();           /* Insert existing file at point*/
extern  int     fileread();             /* Get a file, read only        */
extern  int     filevisit();            /* Get a file, read write       */
extern  int     filewrite();            /* Write a file                 */
extern  int     filesave();             /* Save current file            */
extern  int     filename();             /* Adjust file name             */
extern  int     getccol();              /* Get current column           */
extern  int     gotobol();              /* Move to start of line        */
extern  int     forwchar();             /* Move forward by characters   */
extern  int     gotoeol();              /* Move to end of line          */
extern  int     backchar();             /* Move backward by characters  */
extern  int     forwline();             /* Move forward by lines        */
extern  int     backline();             /* Move backward by lines       */
extern  int     forwpage();             /* Move forward by pages        */
extern  int     backpage();             /* Move backward by pages       */
extern  int     gotobob();              /* Move to start of buffer      */
extern  int     gotoeob();              /* Move to end of buffer        */
extern  int     setfillcol();           /* Set fill column.             */
extern  int     setindcol();            /* Set indent column            */
extern  int     setmark();              /* Set mark                     */
extern  int     swapmark();             /* Swap "." and mark            */
extern  int     forwsearch();           /* Search forward               */
extern  int     backsearch();           /* Search backward              */
extern  int     forwisearch();          /* I-search forward             */
extern  int     backisearch();          /* I-search backward            */
extern  int     replace();              /* Search and replace           */
extern  int     showcpos();             /* Show the cursor position     */
extern  int     nextwind();             /* Move to the next window      */
extern  int     prevwind();             /* Move to the previous window  */
extern  int     onlywind();             /* Make current window only one */
extern  int     splitwind();            /* Split current window         */
extern  int     mvdnwind();             /* Move window down             */
extern  int     mvupwind();             /* Move window up               */
extern  int     enlargewind();          /* Enlarge display window.      */
extern  int     shrinkwind();           /* Shrink window.               */
extern  int     listbuffers();          /* Display list of buffers      */
extern  int     paintbuffer();          /* Set background buffer color  */
extern  int     usebuffer();            /* Switch a window to a buffer  */
extern  int     killbuffer();           /* Make a buffer go away.       */
extern  int     reposition();           /* Reposition window            */
extern  int     refresh();              /* Refresh the screen           */
extern  int     twiddle();              /* Twiddle characters           */
extern  int     tab();                  /* Insert tab                   */
extern  int     newline();              /* Insert CR-LF                 */
extern  int     indent();               /* Insert CR-LF, then indent    */
extern  int     openline();             /* Open up a blank line         */
extern  int     deblank();              /* Delete blank lines           */
extern  int     quote();                /* Insert literal               */
extern  int     backword();             /* Backup by words              */
extern  int     forwword();             /* Advance by words             */
extern  int     forwdel();              /* Forward delete               */
extern  int     backdel();              /* Backward delete              */
extern  int     kill();                 /* Kill forward                 */
extern  int     yank();                 /* Yank back from killbuffer.   */
extern  int     upperword();            /* Upper case word.             */
extern  int     lowerword();            /* Lower case word.             */
extern  int     upperregion();          /* Upper case region.           */
extern  int     lowerregion();          /* Lower case region.           */
extern  int     capword();              /* Initial capitalize word.     */
extern  int     delfword();             /* Delete forward word.         */
extern  int     delbword();             /* Delete backward word.        */
extern  int     killregion();           /* Kill region.                 */
extern  int     copyregion();           /* Copy region to kill buffer.  */
extern  int     writereg();             /* write defined region to file */
extern  int     quickexit();            /* low keystroke style exit.    */
extern  int     goteop();               /* goto end of paragraph */
extern  int     gotbop();               /* goto beginning of paragraph */
extern  int     unkncom();              /* unknown command BEEP */
extern  int     ftopunct();             /* move forward to next punctuation */
extern  int     btopunct();             /* move backward to last punctuation */
extern  int     forwsent();             /* forward to end of sentence */
extern  int     backsent();             /* backward to beginning of sentence */
extern  int     fillpar();              /* fill paragraph to fill column */
extern  int     setpage();              /* set page size from arg */
extern  int     paginate();             /* insert '\f' every page */
extern  int     pageforw();             /* move forward set page length */
extern  int     pageback();             /* move back set page length */
extern  int     print();                /* print buffer with heading */
extern  int     wc();                   /* word and line count of buffer */
extern  int     retversion();           /* version date */
extern  int     kermit();               /* file transfer */
 
/* From here on the commands are from misc.c */
extern  int     mdropln();              /* drop current line and move up*/
extern  int     mindnl();               /* indent subsequent NL same as this */
extern  int     mdeleln();              /* delete entire line from beginning */
extern  int     mdelwln();              /* delete entire line including NL   */
extern  int     mdelind();              /* delete beginning line indentation */
extern  int     markpar();              /* mark paragraph sets mark */
extern  int     tglcase();              /* toggle case of letter at point */
extern  int     killsent();             /* kill sentence forward sets mark */
extern  int     grtw();                 /* remove trailing white space */
extern  int     twaddle();              /* transpaose two words in place */
extern  int     clowsp();               /* close up intervening white space */
extern  int     mrflush();              /* flush right current line */
extern  int     mcenter();              /* center current line */
extern  int     setmode();              /* set a mode for a buffer */
extern  int     sglmode();              /* set global mode      */
extern  int     rettpa();               /* show available memory and usage */
extern  int     showtime();             /* return current time  */
extern  int     goline();               /* goto line in text */
extern  int     enumerate();            /* start or incr. counter variable */
extern  int     gospell();              /* search forward for spell mark */
extern  int     mdoncom();              /* execute a named command */
 
/* functions in macros.c to allow 26 keyboard macros */
extern  int     putmacro();             /* print line macros in text    */
extern  int     getmacro();             /* copy cureent key-mac to key  */
 
/* shell commands */
extern  int     shell();                /* run one command or run cc    */
extern  int     commfil();              /* read alias file              */
extern  int     setpath();              /* get/set path and drive       */
 
/*
 * Command table.
 * This table  is *roughly* in ASCII
 * order, left to right across the characters
 * of the command. This expains the funny
 * location of the control-X commands.
 */
KEYTAB  keytab[] = {
        CTRL|'@',               &setmark,
        "setmark",
        CTRL|'A',               &gotobol,
        "gotobol",
        CTRL|'B',               &backchar,
        "backchar",
        CTRL|'C',               &shell,
        "shell",
        CTRL|'D',               &forwdel,
        "forwdel",
        CTRL|'E',               &gotoeol,
        "gotoeol",
        CTRL|'F',               &forwchar,
        "forwchar",
        CTRL|'G',               &ctrlg,
        "ctrlg",
        CTRL|'H',               &backchar,
        "backchar",
        CTRL|'I',               &tab,
        "tab",
        CTRL|'J',               &indent,
        "indent",
        CTRL|'K',               &mdeleln,
        "mdeleln",
        CTRL|'L',               &refresh,
        "refresh",
        CTRL|'M',               &newline,
        "newline",
        CTRL|'N',               &forwline,
        "forwline",
        CTRL|'O',               &openline,
        "openline",
        CTRL|'P',               &backline,
        "backline",
        CTRL|'Q',               &quote,
        "quote",
        CTRL|'R',               &backsearch,
        "backsearch",
        CTRL|'S',               &forwsearch,
        "forwsearch",
        CTRL|'T',               &twiddle,
        "twiddle",
        CTRL|'V',               &forwpage,
        "forwpage",
        CTRL|'W',               &killregion,
        "killregion",
        CTRL|'Y',               &yank,
        "yank",
        CTRL|'Z',               &quickexit,
        "quickexit",
        CTRL|'\\',              &unkncom,
        "unkncom",
        CTRL|'_',               &kermit,
        "kermit",
        CTRL|'^',               &unkncom,
        "unkncom",
        CTLX|CTRL|'A',          &unkncom,
        "unkncom",
        CTLX|CTRL|'B',          &listbuffers,
        "listbuffers",
        CTLX|CTRL|'C',          &quit,
        "quit",
        CTLX|CTRL|'D',          &unkncom,
        "unkncom",
        CTLX|CTRL|'E',          &commfil,
        "commfil",
        CTLX|CTRL|'F',          &filename,
        "filename",
        CTLX|CTRL|'G',          &ctrlg,
        "ctrlg",
        CTLX|CTRL|'H',          &unkncom,
        "unkncom",
        CTLX|CTRL|'I',          &print,
        "print",
        CTLX|CTRL|'J',          &unkncom,
        "unkncom",
        CTLX|CTRL|'K',          &unkncom,
        "unkncom",
        CTLX|CTRL|'L',          &lowerregion,
        "lowerregion",
        CTLX|CTRL|'M',          &unkncom,
        "unkncom",
        CTLX|CTRL|'N',          &mvdnwind,
        "mvdnwind",
        CTLX|CTRL|'O',          &deblank,
        "deblank",
        CTLX|CTRL|'P',          &mvupwind,
        "mvupwind",
        CTLX|CTRL|'Q',          &unkncom,
        "unkncom",
        CTLX|CTRL|'R',          &fileread,
        "fileread",
        CTLX|CTRL|'S',          &filesave,
        "filesave",
        CTLX|CTRL|'T',          &showtime,
        "showtime",
        CTLX|CTRL|'U',          &upperregion,
        "upperregion",
        CTLX|CTRL|'V',          &filevisit,
        "filevisit",
        CTLX|CTRL|'W',          &filewrite,
        "filewrite",
        CTLX|CTRL|'X',          &swapmark,
        "swapmark",
        CTLX|CTRL|'Y',          &unkncom,
        "unkncom",
        CTLX|CTRL|'Z',          &shrinkwind,
        "shrinkwind",
        CTLX|'!',               &paginate,
        "paginate",
        CTLX|'#',               &setpage,
        "setpage",
        CTLX|'+',               &pageforw,
        "pageforw",
        CTLX|'-',               &pageback,
        "pageback",
        CTLX|'.',               &setindcol,
        "setindcol",
        CTLX|'(',               &ctlxlp,
        "ctlxlp",
        CTLX|')',               &ctlxrp,
        "ctlxrp",
        CTLX|'*',               &retversion,
        "retversion",
        CTLX|'<',               &btopunct,
        "btopunct",
        CTLX|'=',               &showcpos,
        "showcpos",
        CTLX|'>',               &ftopunct,
        "ftopunct",
        CTLX|'0',               &unkncom,
        "unkncom",
        CTLX|'1',               &onlywind,
        "onlywind",
        CTLX|'2',               &splitwind,
        "splitwind",
        CTLX|'3',               &unkncom,
        "unkncom",
        CTLX|'4',               &unkncom,
        "unkncom",
        CTLX|'5',               &unkncom,
        "unkncom",
        CTLX|'6',               &unkncom,
        "unkncom",
        CTLX|'7',               &unkncom,
        "unkncom",
        CTLX|'8',               &unkncom,
        "unkncom",
        CTLX|'9',               &unkncom,
        "unkncom",
        CTLX|'A',               &unkncom,
        "unkncom",
        CTLX|'B',               &usebuffer,
        "usebuffer",
        CTLX|'C',               &paintbuffer,
        "paintbuffer",
        CTLX|'D',               &setpath,
        "setpath",
        CTLX|'E',               &ctlxe,
        "ctlxe",
        CTLX|'F',               &setfillcol,
        "setfillcol",
        CTLX|'H',               &unkncom,
        "unkncom",
        CTLX|'I',               &fileinsert,
        "fileinsert",
        CTLX|'J',               &unkncom,
        "unkncom",
        CTLX|'K',               &killbuffer,
        "killbuffer",
        CTLX|'L',               &unkncom,
        "unkncom",
        CTLX|'M',               &setmode,
        "setmode",
        CTLX|'N',               &nextwind,
        "nextwind",
        CTLX|'O',               &prevwind,
        "prevwind",
        CTLX|'P',               &prevwind,
        "prevwind",
        CTLX|'Q',               &unkncom,
        "unkncom",
        CTLX|'R',               &writereg,
        "writereg",
        CTLX|'S',               &gospell,
        "gospell",
        CTLX|'T',               &unkncom,
        "unkncom",
        CTLX|'U',               &unkncom,
        "unkncom",
        CTLX|'V',               &unkncom,
        "unkncom",
        CTLX|'W',               &wc,
        "wc",
        CTLX|'Y',               &unkncom,
        "unkncom",
        CTLX|'Z',               &enlargewind,
        "enlargewind",
        CTLX|'\\',              &grtw,
        "grtw",
        CTLX|'`',               &getmacro,
        "getmacro",
        CTLX|'~',               &shell,
        "shell",
        CTLX|SPEC|'p',          &swapmark,
        "swapmark",
        META|CTRL|'B',          &backword,
        "backword",
        META|CTRL|'C',          &mcenter,
        "mcenter",
        META|CTRL|'F',          &forwword,
        "forwword",
        META|CTRL|'G',          &ctrlg,
        "ctrlg",
        META|CTRL|'H',          &backword,
        "backword",
        META|CTRL|'I',          &kill,
        "kill",
        META|CTRL|'K',          &mdelwln,
        "mdelwln",
        META|CTRL|'M',          &unkncom,
        "unkncom",
        META|CTRL|'N',          &enumerate,
        "enumerate",
        META|CTRL|'O',          &clowsp,
        "clowsp",
        META|CTRL|'P',          &tglcase,
        "tglcase",
        META|CTRL|'R',          &mrflush,
        "mrflush",
        META|CTRL|'S',          &forwisearch,
        "forwisearch",
        META|CTRL|'T',          &backisearch,
        "backisearch",
        META|'!',               &reposition,
        "reposition",
        META|'.',               &gotoeob,
        "gotoeob",
        META|',',               &gotobob,
        "gotobob",
        META|'>',               &gotoeob,
        "gotoeob",
        META|'<',               &gotobob,
        "gotobob",
        META|' ',               &setmark,
        "setmark",
        META|'@',               &rettpa,
        "rettpa",
        META|'A',               &backsent,
        "backsent",
        META|'B',               &backword,
        "backword",
        META|'C',               &capword,
        "capword",
        META|'D',               &delfword,
        "delfword",
        META|'E',               &forwsent,
        "forwsent",
        META|'F',               &forwword,
        "forwword",
        META|'G',               &goline,
        "goline",
        META|'H',               &markpar,
        "markpar",
        META|'I',               &unkncom,
        "unkncom",
        META|'J',               &mindnl,
        "mindnl",
        META|'K',               &killsent,
        "killsent",
        META|'L',               &lowerword,
        "lowerword",
        META|'M',               &sglmode,
        "sglmode",
        META|'N',               &goteop,
        "goteop",
        META|'O',               &mdropln,
        "mdropln",
        META|'P',               &gotbop,
        "gotbop",
        META|'Q',               &fillpar,
        "fillpar",
        META|'R',               &replace,
        "replace",
        META|'S',               &unkncom,
        "unkncom",
        META|'T',               &twaddle,
        "twaddle",
        META|'U',               &upperword,
        "upperword",
        META|'V',               &backpage,
        "backpage",
        META|'W',               &copyregion,
        "copyregion",
        META|'X',               &mdoncom,
        "mdoncom",
        META|'Y',               &unkncom,
        "unkncom",
        META|'Z',               &unkncom,
        "unkncom",
        META|'\\',              &mdelind,
        "mdelind",
        META|'~',               &clearflag,
        "clearflag",
        META|0x7F,              &delbword,
        "delbword",
#if     ST
        SPEC|'H',               &backline,
        "backline",
        SPEC|'P',               &forwline,
        "forwline",
        SPEC|'K',               &backchar,
        "backchar",
        SPEC|'M',               &forwchar,
        "forwchar",
        SPEC|'b',               &kermit,
        "kermit",
        SPEC|'a',               &yank,
        "yank",
        SPEC|'R',               &openline,
        "openline",
        SPEC|'S',               &backdel,
        "backdel",
        SPEC|'G',               &refresh,
        "refresh",
        SPEC|'q',               &forwdel,
        "forwdel",
        SPEC|'r',               &indent,
        "indent",
        SPEC|'c',               &backword,
        "backword",
        SPEC|'d',               &forwword,
        "forwword",
        SPEC|'e',               &grtw,
        "grtw",
        SPEC|'f',               &retversion,
        "retversion",
        SPEC|'g',               &gotobol,
        "gotobol",
        SPEC|'h',               &gotoeol,
        "gotoeol",
        SPEC|'i',               &unkncom,
        "unkncom",
        SPEC|'J',               &pageback,
        "pageback",
        SPEC|'N',               &pageforw,
        "pageforw",
        SPEC|'j',               &backsent,
        "backsent",
        SPEC|'k',               &forwsent,
        "forwsent",
        SPEC|'l',               &unkncom,
        "unkncom",
        SPEC|'m',               &gotbop,
        "gotbop",
        SPEC|'n',               &goteop,
        "goteop",
        SPEC|'o',               &unkncom,
        "unkncom",
        SPEC|'D',               &quickexit,
        "quickexit",
        SPEC|'C',               &filesave,
        "filesave",
        SPEC|'B',               &filewrite,
        "filewrite",
        SPEC|'A',               &filevisit,
        "filevisit",
        SPEC|'@',               &fileread,
        "fileread",
        SPEC|'?',               &fileinsert,
        "fileinsert",
        SPEC|'>',               &writereg,
        "writereg",
        SPEC|'=',               &filename,
        "filename",
        SPEC|'<',               &listbuffers,
        "listbuffers",
        SPEC|';',               &setmark,
        "setmark",
#endif
        AGRAVE,                 &putmacro,
        "putmacro",
        0x7F,                   &backdel,
        "backdel"
};
 
#define NKEYTAB (sizeof(keytab)/sizeof(keytab[0]))
 
#if     LK201
/*
 * Mapping table for all of the funny
 * keys with the numeric parameters on the LK201.
 * Indexed by the code, which is between 0 (unused) and
 * 34 (F20). An entry of 0 means no mapping. The map
 * goes to command keys. If I had a "special" bit,
 * I could use the code in the escape sequence as a
 * key code, and return (for example) "do" as
 * SPECIAL + 29. Then the dispatch would be
 * done by the default keymap. This is probably a
 * better way to go.
 */
short   lkmap[] = {
        0,
        CTRL|'S',                       /* 1    Find                    */
        CTRL|'Y',                       /* 2    Insert here             */
        CTRL|'W',                       /* 3    Remove                  */
        CTRL|'@',                       /* 4    Select                  */
        META|'V',                       /* 5    Previous screen         */
        CTRL|'V',                       /* 6    Next screen             */
        0,
        0,
        0,
        0,                              /* 10   Compose                 */
        0,
        0,                              /* 12   Print screen            */
        0,
        0,                              /* 14   F4                      */
        0,
        0,
        0,                              /* 17   F6                      */
        0,                              /* 18   F7                      */
        0,                              /* 19   F8                      */
        0,                              /* 20   F9                      */
        0,                              /* 21   F10                     */
        0,
        0,
        0,
        0,
        0,                              /* 26   F14                     */
        0,
        0,                              /* 28   Help                    */
        CTLX|'E',                       /* 29   Do      C-X E           */
        0,
        CTLX|'P',                       /* 31   F17     C-X P           */
        CTLX|'N',                       /* 32   F18     C-X N           */
        CTLX|'Z',                       /* 33   F19     C-X Z           */
        CTLX|CTRL|'Z'                   /* 34   F20     C-X C-Z         */
};
#endif
 
/*
 * This is the general command execution
 * routine. It handles the fake binding of all the
 * keys to "self-insert". It also clears out the "thisflag"
 * word, and arranges to move it to the "lastflag", so that
 * the next command can look at it. Return the status of
 * command.
 */
execute(c, f, n)
register int c, f, n;
{
        register KEYTAB *ktp;
        register int    status;
 
        ktp = &keytab[0];                       /* Look in key table.   */
        while (ktp < &keytab[NKEYTAB]) {
                if (ktp->k_code == c) {
                        thisflag = 0;
                        status   = (*ktp->k_fp)(f, n);
                        lastflag = thisflag;
                        return (status);
                }
                ++ktp;
        }
 
        /*
         * If inwrap a space was typed, fill column is defined, the argument
         * is non-negative, and we are now past fill column, perform word wrap.
         */
        if ((curbp->b_bmode&BMWRAP) != 0 && c == ' ' && fillcol > 0
                && n>=0 && getccol(FALSE) > fillcol)
                {
                linsert(1, c);  /* we want the space */
                wrapword(FALSE, NULL);
                }
        if ((c>=0x20 && c<=0x7E)                /* Self inserting.      */
        ||  (c>=0xA0 && c<=0xFE)) {
                if (n <= 0) {                   /* Fenceposts.          */
                        lastflag = 0;
                        return (n<0 ? FALSE : TRUE);
                }
                thisflag = 0;                   /* For the future.      */
                status   = linsert(n, c);
                lastflag = thisflag;
                return (status);
        }
        lastflag = 0;                           /* Fake last flags.     */
        return (FALSE);
}
 
bindkey(code,name)
register short code;
register char *name;
{
        register KEYTAB *oktp,*nktp;
        int found = FALSE;
 
        nktp = &keytab[0];      /* bind to this key */
        oktp = &keytab[0];      /* the function to bind */
        /* look through to find keytab assoc with "name" */
        while (oktp < &keytab[NKEYTAB]) {
                if (strcmp(oktp->k_mfunc,name)==NULL) {
                        found = TRUE;
                        break;
                }
                ++oktp;
        }
        if (found)
                /* search for key */
                while (nktp < &keytab[NKEYTAB]) {
                        if (nktp->k_code == code) {
                                strcpy(nktp->k_mfunc,oktp->k_mfunc);
                                nktp->k_fp=oktp->k_fp;
                                return(TRUE);
                        }
                        ++nktp;
                }
        return(FALSE);
}
 
/* given a keycode value, return a pointer to the name of that function
 * and get the function's name into `name'.
 */
char *
getfname(kcode,name)
register short kcode;
char name[];
{
        register KEYTAB *ktp;
 
        ktp = &keytab[0];       /* the function to find */
        /* look through to find keytab assoc with "code" */
        while (ktp < &keytab[NKEYTAB]) {
                if (ktp->k_code == kcode)
                        {
                        strcpy(name,ktp->k_mfunc);
                        return(ktp->k_mfunc);
                        }
                ++ktp;
        }
        return((char *)NULL);   /* not found */
}
 
/* return the code associated with `name' */
short
getktpcode(name)
char name[];
{
        register KEYTAB *ktp;
 
        ktp = &keytab[0];       /* the function to find */
        /* look through to find keytab assoc with "name" */
        while (ktp < &keytab[NKEYTAB]) {
                if (strcmp(ktp->k_mfunc,name) == NULL)
                        return(ktp->k_code);
                ++ktp;
        }
        return(NULL);
}
