/* $Id: template.c,v 1.2 1996/10/12 14:13:25 davidn Exp $
 * Functions to support NLMaint's template commands/strings
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <ctype.h>

#include "nlmaint.h"
#include "osdep.h"
#include "log.h"
#include "cmdtpl.h"


/* This is an enumerator for the template array below
 * This makes the array easier to manipulate, since all you
 * have to do is keep the two in sync rather than having to
 * modify code.
 */

enum {
    t_DAYNUM, t_SDAYNUM, t_OLDNUM, t_SOLDNUM,
    t_MASTER, t_OUTPATH, t_MSGBASE, t_NETNAME,
    t_MYADDR, t_SUBMIT,
    t_NLPATH, t_NLNAME, t_NLBASE, t_NLEXT,
    t_NDPATH, t_NDNAME, t_NDBASE, t_NDEXT,
    t_ARCLIST, t_ARCDIFF, t_NLCOMP, t_NDCOMP, t_NLCMPL, t_NDCMPL,
    t_NLSIZE, t_NDSIZE, t_NLDATE,
    t_USER, t_USERNAME, t_USERDIR, t_USERBASE, t_USEREXT,
    t_SFADDR, t_SFNAME, t_SFNUM, t_SFPWD, t_NOTIFY
};
/* Common substitution template for cmds and FILES.BBS updating
 */
static cmdtpl tplsubs[] =
{
    {'y', TC_STD, 0, NULL, NULL},	/* t_DAYNUM   Day number */
    {'Y', TC_STD, 0, NULL, NULL},	/* t_SDAYNUM  Day number left truncated
					 * to 2 chrs */
    {'p', TC_STD, 0, NULL, NULL},	/* t_OLDNUM   Day number for previous
					 * week */
    {'P', TC_STD, 0, NULL, NULL},	/* t_SOLDNUM  Previous weeks number
					 * left truncated */
    {'m', TC_STD, 0, NULL, NULL},	/* t_MASTER   Master directory */
    {'o', TC_STD, 0, NULL, NULL},	/* t_OUTPATH  Output directory */
    {'M', TC_STD, 0, NULL, NULL},	/* t_MSGBASE  Messagebase */
    {'e', TC_STD, 0, NULL, NULL},	/* t_NETNAME  Network name */
    {'O', TC_STD, 0, NULL, NULL},	/* t_MYADDR   My network address */
    {'t', TC_STD, 0, NULL, NULL},	/* t_SUBMIT   Submit address */
    {'n', TC_STD, 0, NULL, NULL},	/* t_NLPATH   Nodelist name (maybe
					 * archived) */
    {'N', TC_NAME, 0, NULL, NULL},	/* t_NLNAME   Name of nodelist */
    {'B', TC_BASE, 0, NULL, NULL},	/* t_NLBASE   Base name of nodelist */
    {'E', TC_EXT, 0, NULL, NULL},	/* t_NLEXT    Nodelist file extension */
    {'d', TC_STD, 0, NULL, NULL},	/* t_NDPATH   Nodediff name */
    {'D', TC_NAME, 0, NULL, NULL},	/* t_NDNAME   Name of nodediff */
    {'b', TC_BASE, 0, NULL, NULL},	/* t_NDBASE   Base name of nodediff */
    {'e', TC_EXT, 0, NULL, NULL},	/* t_NDEXT    Nodediff file extension */
    {'S', TC_STD, 0, NULL, NULL},	/* t_ARCLIST  ARCList sub-directory */
    {'s', TC_STD, 0, NULL, NULL},	/* t_ARCDIFF  ARCDiff sub-directory */
    {'A', TC_STD, 0, NULL, NULL},	/* t_NLCOMP   Nodelist compression
					 * method */
    {'a', TC_STD, 0, NULL, NULL},	/* t_NDCOMP   Nodediff compression
					 * method */
    {'L', TC_STD, 0, NULL, NULL},	/* t_NLCMPL   Nodelist compression
					 * letter */
    {'l', TC_STD, 0, NULL, NULL},	/* t_NDCMPL   Nodediff compression
					 * letter */
    {'I', TC_STD, 0, NULL, NULL},	/* t_NLSIZE   Size of nodelist (bytes) */
    {'i', TC_STD, 0, NULL, NULL},	/* t_NDSIZE   Size of nodediff (bytes) */
    {'c', TC_STD, 0, NULL, NULL},	/* t_NLDATE   Nodelist publish date
					 * DD-Mmm-YYYY */
    {'u', TC_STD, 0, NULL, NULL},	/* t_USER     User specified string */
    {'U', TC_NAME, 0, NULL, NULL},	/* t_USERNAME Same - name only */
    {'Y', TC_DIR, 0, NULL, NULL},	/* t_USERDIR  Same - dir only */
    {'z', TC_BASE, 0, NULL, NULL},	/* t_USERBASE Same - basename only */
    {'Z', TC_EXT, 0, NULL, NULL},	/* t_USEREXT  Same - extension only */
    {'f', TC_STD, 0, NULL, NULL},	/* t_SFADDR   Subfile address */
    {'F', TC_STD, 0, NULL, NULL},	/* t_SFNAME   Subfile name */
    {'g', TC_STD, 0, NULL, NULL},	/* t_SFNUM    Subfile number */
    {'G', TC_STD, 0, NULL, NULL},	/* t_SFPWD    Subfile password */
    {'T', TC_STD, 0, NULL, NULL},	/* t_NOTIFY   Notify priority
					 * parameters */
    {0, TC_LAST, 0, NULL, NULL}
};


/* make_template() - Build a template table from configuration/current data
 */

void
make_template(NLCONFIG * nlc, char const * list, char const * fdif, char const * user, subfile * sf)
{
    time_t lastweek = nlc->publish_time - (SECSPERDAY * 7);
    struct tm *T = localtime(&lastweek);
    struct stat st;
    unsigned i;
    static char daynum[6],	/* current day number */
        oldnum[6],		/* Last week's day number */
        myaddr[32],		/* My address */
        toaddr[32],		/* Submit address */
        sfaddr[32],		/* Subfile address */
        sfnum[16],		/* Subfile number */
        nlsize[16],		/* Nodelist size */
        ndsize[16],		/* Nodediff size */
        nldate[16],		/* Nodelist date */
        notify[64];		/* Notify attributes */

    static char const mns[] = "JanFebMarAprMayJunJulAugSepOctNovDec";
    sprintf(daynum, "%03d", nlc->target);
    sprintf(oldnum, "%03d", T->tm_yday + 1);
    fidofmt(myaddr, &nlc->myaddr, NULL);
    fidofmt(toaddr, &nlc->submitaddr, NULL);
    fidofmt(sfaddr, sf ? &sf->addr : &nlc->myaddr, NULL);
    sprintf(sfnum, "%u", sf ? sf->number : nlc->number);
    if (list && *list && stat(list, &st) == 0)
	sprintf(nlsize, "%ld", (long) st.st_size);
    else
	strcpy(nlsize, "0");
    if (fdif && *fdif && stat(fdif, &st) == 0)
	sprintf(ndsize, "%ld", (long) st.st_size);
    else
	strcpy(ndsize, "0");
    T = localtime(&nlc->publish_time);
    sprintf(nldate, "%02d-%3.3s-%04d", T->tm_mday, mns + (T->tm_mon * 3), T->tm_year + 1900);
    i = (sf ? sf->notify : nlc->notify) & ~(NF_NONE | NF_RECEIPT | NF_ERRORSONLY | NF_WARNONLY | NF_ALL);
    if (~(i & NF_XFLAGS))
	i &= ~(NF_IMM | NF_DIRECT);
    notify_str(notify, i);

    for (i = 0; tplsubs[i].flags != TC_LAST; i++) {
	switch (i) {
	case t_DAYNUM:
	    tplsubs[i].arg = daynum;
	    break;
	case t_SDAYNUM:
	    tplsubs[i].arg = daynum + 1;
	    break;
	case t_OLDNUM:
	    tplsubs[i].arg = oldnum;
	    break;
	case t_SOLDNUM:
	    tplsubs[i].arg = oldnum + 1;
	    break;
	case t_MASTER:
	    tplsubs[i].arg = nlstring(nlc, V_MASTER);
	    break;
	case t_OUTPATH:
	    tplsubs[i].arg = nlstring(nlc, V_OUTPATH);
	    break;
	case t_MSGBASE:
	    tplsubs[i].arg = nlstring(nlc, V_MESSAGES);
	    break;
	case t_NETNAME:
	    tplsubs[i].arg = nlstring(nlc, V_NAME);
	    break;
	case t_MYADDR:
	    tplsubs[i].arg = myaddr;
	    break;
	case t_SUBMIT:
	    tplsubs[i].arg = toaddr;
	    break;
	case t_NLPATH:
	case t_NLNAME:
	case t_NLBASE:
	case t_NLEXT:
	    tplsubs[i].arg = list && *list ? list : ".";
	    break;
	case t_NDPATH:
	case t_NDNAME:
	case t_NDBASE:
	case t_NDEXT:
	    tplsubs[i].arg = fdif && *fdif ? fdif : ".";
	    break;
	case t_ARCLIST:
	    tplsubs[i].arg = nlstring(nlc, V_ARCLIST);
	    break;
	case t_ARCDIFF:
	    tplsubs[i].arg = nlstring(nlc, V_ARCDIFF);
	    break;
	case t_NLCOMP:
	    tplsubs[i].arg = nlstring(nlc, V_ARC);
	    break;
	case t_NDCOMP:
	    tplsubs[i].arg = nlstring(nlc, V_DIFFARC);
	    break;
	case t_NLCMPL:
	    tplsubs[i].arg = nlstring(nlc, V_NLPFX);
	    break;
	case t_NDCMPL:
	    tplsubs[i].arg = nlstring(nlc, V_NDPFX);
	    break;
	case t_NLSIZE:
	    tplsubs[i].arg = nlsize;
	    break;
	case t_NDSIZE:
	    tplsubs[i].arg = ndsize;
	    break;
	case t_NLDATE:
	    tplsubs[i].arg = nldate;
	    break;
	case t_USER:
	case t_USERNAME:
	case t_USERDIR:
	case t_USERBASE:
	case t_USEREXT:
	    tplsubs[i].arg = user;
	    break;
	case t_SFADDR:
	    tplsubs[i].arg = sfaddr;
	    break;
	case t_SFNAME:
	    tplsubs[i].arg = sf ? nlstr(nlc, sf->name) : nlstring(nlc, V_OUTFILE);
	    break;
	case t_SFNUM:
	    tplsubs[i].arg = sfnum;
	    break;
	case t_SFPWD:
	    tplsubs[i].arg = sf ? nlstr(nlc, sf->password) : nlstring(nlc, V_PASSWORD);
	    break;
	case t_NOTIFY:
	    tplsubs[i].arg = notify;
	    break;
	}
    }
}


/* exec_cmd() - execute a pre- or post- processing command template
 */

int
exec_cmd(NLCONFIG * nlc, char const * tpl, char const * list, char const * fdif, char const * user, subfile * sf)
{
    make_template(nlc, list, fdif, user, sf);
    return tpl_exec(tpl, tplsubs);
}


/* update_description() - update a files.bbs file with a description template
 */

int
update_description(NLCONFIG * nlc, char const * filesbbs, char const * tpl, char const * list, char const * fdif)
{
    int rc = 0;
    char *tmp;
    make_template(nlc, list, fdif, filesbbs, NULL);
    if ((tmp = tpl_make(tpl, tplsubs, 2048)) != NULL) {
	FILE *fp = fopen(filesbbs, "a+");
	if (fp == NULL)
	    logit(LOG_WARN, "Can't update description in \"%s\"", filesbbs);
	else {
	    fprintf(fp, "%s\n", tmp);
	    fclose(fp);
	    rc = 1;
	}
	tpl_dealloc(tmp, tplsubs);
    }
    return rc;
}
