/* $Id: strbuf.c,v 1.1.1.1 1996/10/09 11:25:31 davidn Exp $
 * Implements a simple strings buffer
 *
 */

#include <stdlib.h>
#include <string.h>
#include "strbuf.h"
#include "mem.h"
#include "osdep.h"


#define sbptr(sb, ofs)     ((sb)->heap + (ofs))


/* sb_new() - Create/allocate a new strings buffer
 */

strbuf *
sb_new(unsigned sz)
{
    return sb_init(zmalloc("sb_new", sizeof(strbuf)), sz);
}


/* sb_init() - Initialise a new strings buffer
 */

strbuf *
sb_init(strbuf * sb, unsigned sz)
{
    if (sz <= 0)
	sz = STRBUFSZ;
    sb->soSize = (sofs_t) 0;
    sb->soOffset = (sofs_t) 1;
    sb->heap = 0;
    sb_realloc(sb, sz);
    return sb;
}


/* sb_destroy() - Destroy a strings buffer
 */

void
sb_destroy(strbuf * sb)
{
    if (sb)
	zfree("sb_destroy", sb->heap);
}


/* sb_delete() - Destroy/deallocate a strings buffer
 */

void
sb_delete(strbuf * sb)
{
    sb_destroy(sb);
    zfree("sb_delete", sb);
}


/* sb_salloc() - Allocate a string of length in a strings buffer
 */

sofs_t
sb_salloc(strbuf * sb, char const * s, int slen)
{				/* Allocate a substring */
    static char const fn[] = "sb_salloc";
    sofs_t ofs = 0;
    if (s && *s && slen) {
	char *tmp = NULL;
	char *p;
	sofs_t len = slen + 1;	/* Include NUL */
	long top = (long) sb->soOffset + (long) len;
	/*
	 * Need to reallocate the buffer?
	 */
	if (top >= (long) sb->soSize) {

	    /*
	     * Blown the maximum size?
	     */
	    if (top >= (sofs_t) - 1)
		return 0;	/* Can't do much else */

	    /*
	     * We may need to copy it temporarily
	     */
	    if (s > sb->heap && s < (sb->heap + sb->soOffset)) {
		tmp = zmalloc(fn, len);
		memcpy(tmp, s, len);
		s = tmp;
	    }
	    /*
	     * Make the buffer larger
	     */
	    if (!sb_realloc(sb, (unsigned) top))
		return 0;
	}
	/*
	 * Get position of string
	 */
	p = sbptr(sb, ofs = sb->soOffset);
	sb->soOffset += len;

	/*
	 * Copy the new string in
	 */
	memcpy(p, s, len);
	p[len] = '\0';

	/*
	 * Dispose of possible temporary
	 */
	zfree(fn, tmp);
    }
    return ofs;
}


/* sb_alloc() - Allocate an asciiz string in a strings buffer
 */

sofs_t
sb_alloc(strbuf * sb, char const * s)
{

    /*
     * Allocate a standard string
     */
    return (sofs_t) (s == NULL ? 0 : sb_salloc(sb, s, strlen(s)));
}


/* sb_free() - Free a string from a strings buffer
 */

void
sb_free(strbuf * sb, sofs_t ofs)
{
    if (ofs > 0 && ofs < sb->soOffset) {
	char *s = sbptr(sb, ofs);
	if (*(s - 1) == '\0') {
	    sofs_t at, len;
	    sofs_t used = 1;
	    /*
	     * Mark string as deleted
	     */
	    *s = STRDELMK;

	    /*
	     * Now, walk up the buffer and find the last string, adjusting the
	     * next offset downwards if we can
	     */
	    for (at = 1; at < sb->soOffset; at += len) {
		char *p = sbptr(sb, at);
		len = (sofs_t) (strlen(p) + 1);

		/*
		 * Track end of last used string
		 */
		if (*p != STRDELMK)
		    used = at + len;
	    }

	    /*
	     * Set the last offset to one character past end of last string
	     */
	    sb->soOffset = used;
	}
    }
}


/* sb_string() - Return ptr to a string at an offset
 */

char const *
sb_string(strbuf * sb, sofs_t ofs)
{
    if (ofs < 1 || ofs >= sb->soOffset || ofs == STRNONE)
	ofs = 0;
    return sb->heap + ofs;
}


/* sb_reset() - Reset a strings buffer
 */

void
sb_reset(strbuf * sb)
{
    sb->soOffset = 1;
}


/* sb_realloc() - Reallocate a buffer to a larger size
 */

int
sb_realloc(strbuf * sb, int sz)
{

    /*
     * Increment in block size units, rounding up
     */
    sz += (STRBUFSZ - 1);
    sz -= (sz % STRBUFSZ);
    if (sz > sb->soSize) {
	char *newheap;
	if ((newheap = zrealloc("sb_realloc", sb->heap, sz)) == NULL)
	    return 0;
	sb->heap = newheap;
	*sb->heap = '\0';
	sb->soSize = sz;
    }
    return 1;
}


/* sb_pack() - Pack a strings buffer
 */

void
sb_pack(strbuf * sb, int (*reloc) (sofs_t mold, sofs_t mnew))
{
    static char const fn[] = "sb_pack";
    char *newheap = zmalloc(fn, sb->soOffset);
    sofs_t at, len, ofs = 1;
    /*
     * Walk up buffer and discard deleted strings
     */
    *newheap = '\0';
    for (at = 1; at < sb->soOffset; at += len) {
	char *p = sbptr(sb, at);
	len = (sofs_t) (strlen(p) + 1);
	if (*p != STRDELMK && (!reloc || reloc(at, ofs))) {	/* Drop the string if
								 * caller has lost it */
	    memcpy(newheap + ofs, p, len);
	    ofs += len;
	}
    }

    /*
     * Move in the new buffer
     */
    zfree(fn, sb->heap);
    sb->heap = newheap;
    sb->soSize = sb->soOffset = ofs;
}


/* sb_offset() - Find offset of a string in a strings buffer
 */

sofs_t
sb_offset(strbuf * sb, char *s, int icase)
{
    sofs_t at, len;
    for (at = 1; at < sb->soOffset; at += len) {
	char *p = sbptr(sb, at);
	len = (sofs_t) (strlen(p) + 1);
	if ((icase ? stricmp : strcmp) (s, p) == 0)
	    return at;
    }
    return STRNONE;
}
