/* $Id: harray.c,v 1.1.1.1 1996/10/09 11:24:45 davidn Exp $
 * Huge array support for 16-bit intel
 *
 */

#include <string.h>
#include "mem.h"
#include "array.h"

#ifdef __16BITSEGMENTED__

static int
harray_alloc(HArray * A, unsigned tosize)
{

    /*
     * Only allocate if required
     */
    if (tosize > A->asize) {
	lsize_t memsize;
	char huge *newbase;
	/*
	 * If just adding a new element, grow by a delta
	 */
	if (tosize == (A->asize + 1))
	    tosize += (A->asize > 32) ? 16 : (A->asize / 2);
	memsize = (lsize_t) tosize *A->cellsize;	/* Calc required memory */
	newbase = zhrealloc("harray_alloc", A->base, memsize);
	if (newbase == 0L)	/* Disaster */
	    return -1;
	A->base = newbase;
	A->asize = tosize;
	return 1;
    }
    return 0;			/* Nothing allocated */
}


HArray *
harray_new(unsigned elsize, unsigned isize)
{
    return harray_init(zmalloc("harray_new", sizeof(HArray)), elsize, isize);
}

HArray *
harray_init(HArray * A, unsigned elsize, unsigned isize)
{
    A->elsize = elsize;

    /*
     * Round the cell size up to be evenly divisible into 65536 to prevent one
     * cell spanning over segment boundaries
     */
    while ((65536L % elsize) != 0)
	++elsize;
    A->cellsize = elsize;
    A->cellsps = (unsigned) (65536 / elsize);
    A->asize = 0;
    A->aused = 0;
    A->base = 0L;
    harray_alloc(A, isize);
    return A;
}

void
harray_delete(HArray * A)
{
    harray_destroy(A);
    zfree("harray_delete", A);
}

void
harray_destroy(HArray * A)
{
    if (A)
	zhfree("harray_destroy", A->base);
}

unsigned
harray_add(HArray * A, unsigned where, unsigned elems)
{
    unsigned at = A->aused;
    if (elems) {
	A->aused += elems;
	if (harray_alloc(A, A->aused) == -1)
	    A->aused -= elems;
	else {
	    if (where < at)	/* Need to move elements in the array up? */
		hugememmove(harray_ptr(A, where + elems), harray_ptr(A, where),
			    (lsize_t) A->cellsize * (at - where));
	    else
		where = at;
	    for (at = 0; at < elems; at++)
		_fmemset(harray_ptr(A, where + at), 0, A->cellsize);
	    return where;
	}
    }
    return NOELEMENT;
}

unsigned
harray_del(HArray * A, unsigned where, unsigned elems)
{
    if (elems == NOELEMENT || (where + elems) >= A->aused)
	elems = (where > A->aused) ? 0 : A->aused - where;
    else {			/* Need to move some elements down */
	unsigned top = where + elems;
	hugememmove(harray_ptr(A, where), harray_ptr(A, top),
		    (lsize_t) A->elsize * (A->aused - top));
    }
    A->aused -= elems;
    return elems;		/* How many elements deleted */
}

void far *
harray_ptr(HArray * A, unsigned index)
{
    return (void far *) (A->base + ((lsize_t) index * A->cellsize));
}

void
harrayIter_init(HArrayIter * I, HArray * A)
{
    I->A = A;
    I->index = NOELEMENT;
}

void far *
harrayIter_get(HArrayIter * I, int which)
{
    switch (which) {
	case PREV:
	if (I->index != NOELEMENT) {
	    --I->index;
	    break;
	}
	/* Fallthru */
    case LAST:
	I->index = (I->A)->aused - 1;
	break;
    case NEXT:
	if (I->index != NOELEMENT) {
	    ++I->index;
	    break;
	}
	/* Fallthru */
    case FIRST:
	I->index = 0;
	break;
    }
    if (I->index >= (I->A)->aused)	/* Auto reset at either end */
	I->index = NOELEMENT;
    return harray_ptr(I->A, I->index);
}
#endif				/* __16BITSEGMENTED__ */
