/* $Id: aplydiff.c,v 1.1.1.1 1996/10/09 11:25:01 davidn Exp $
 * Apply difference files to nodelists
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "fidoaddr.h"
#include "mem.h"
#include "log.h"
#include "osdep.h"
#include "aplydiff.h"
#include "crc16.h"

#define SECSPERDAY          (60L*60L*24L)


/* diffapp_init() - Initialise a DIFFAPP object
 */

DIFFAPP *
diffapp_init(DIFFAPP * da, time_t publish)
{
    memset(da, 0, sizeof(DIFFAPP));
    da->linebuf = zmalloc("diffapp_init", MAXLINEBUF);
    sb_init(&da->sbuf, 512);
    array_init(&da->diffs, sizeof(DFFILE), 8);
    da->publish = publish;
    da->orig.julian = da->targ.julian = -1;
    da->orig.week = da->targ.week = -1;
    return da;
}


/* diffapp_new() - Create a DIFFAPP object
 */
DIFFAPP *
diffapp_new(time_t publish)
{
    return diffapp_init(zmalloc("diffapp_new", sizeof(DIFFAPP)), publish);
}


/* diffapp_destroy() - Destroy a DIFFAPP objectvoid
 */

void
diffapp_destroy(DIFFAPP * da)
{
    if (da) {
	sb_destroy(&da->sbuf);
	array_destroy(&da->diffs);
	zfree("diffapp_destroy", da->linebuf);
    }
}


/* diffapp_delete() - Destroy and deallocate a DIFFAPP objectvoid
 */

void
diffapp_delete(DIFFAPP * da)
{
    diffapp_destroy(da);
    zfree("diffapp_delete", da);
}


/* diffapp_nlfile() - Add or replace a source nodelist
 */

int
diffapp_nlfile(DIFFAPP * da, char const * nlfile)
{
    int rc = 0;
    short daynumber;
    unsigned short crc16;
    if (nlfiletype(nlfile, &daynumber, &crc16, NULL) == F_LIST) {
	short weekno = weeksold(da->publish, daynumber);
	if (weekno != -1) {	/* What happened to the julian day? */
	    NLFILE *nlf = &da->orig;
	    if (nlf->week == -1 || weekno < nlf->week) {	/* Grab the newest
								 * nodelist presented */
		nlf->file = sb_alloc(&da->sbuf, nlfile);
		nlf->julian = daynumber;
		nlf->week = weekno;
		nlf->crc16 = crc16;
		rc = 1;
	    }
	}
    }
    return rc;
}


/* diffapp_nldiff() - Add nodediff for processing
 */

int
diffapp_nldiff(DIFFAPP * da, char const * nldiff)
{
    int rc = 0;
    short daynumber;
    unsigned short crc16;
    if (nlfiletype(nldiff, &daynumber, &crc16, NULL) == F_DIFF) {
	short weekno = weeksold(da->publish, daynumber);
	if (weekno != -1) {	/* Yow, what happened to the julian day? */

	    /*
	     * Let's add these in the order of weeks with an insertion sort We
	     * won't be doing this often enough to warrant something cute like
	     * a binary search.
	     */
	    DFFILE *diff;
	    unsigned idx = 0;
	    for (idx = 0; (diff = array_ptr(&da->diffs, idx)) != NULL; idx++) {
		if (diff->week == weekno)	/* Two copies of the same
						 * diff??? */
		    return 0;	/* Then we don't want this one! */
		if (weekno < diff->week)	/* Insert BEFORE the next
						 * highest */
		    break;
	    }
	    idx = array_add(&da->diffs, idx, 1);
	    diff = array_ptr(&da->diffs, idx);
	    diff->file = sb_alloc(&da->sbuf, nldiff);
	    diff->julian = daynumber;	/* Day number of old list */
	    diff->week = weekno;/* Week number */
	    diff->state = DA_Init;	/* Ready state */
	    diff->crc16 = 0;
	    diff->counter = 0;
	    diff->fp = NULL;
	    rc = 1;
	}
    }
    return rc;
}


/* diffapp_settarget() - Determine nodelist target and eliminate unused diffs
 */

int
diffapp_settarget(DIFFAPP * da)
{
    NLFILE *nlf = &da->targ;
    if (nlf->week == -1 && da->orig.week != -1) {	/* Not previously done */
	DFFILE *diff;
	int atweek = da->orig.week;	/* We can eliminate older diffs */
	unsigned idx = 0;
	while ((diff = array_ptr(&da->diffs, idx)) && diff->week <= atweek)
	    ++idx;

	/*
	 * Remove the useless diffs
	 */
	array_del(&da->diffs, idx, NOELEMENT);

	/*
	 * Let's see how many consecutive diffs we can combine
	 */
	while ((diff = array_ptr(&da->diffs, --idx)) && diff->week == atweek)
	    atweek--;

	/*
	 * We'll eliminate everything subsequent to one missing one
	 */
	if (++idx)
	    array_del(&da->diffs, 0, idx);
	if (array_items(&da->diffs)) {	/* We have one or more diffs! */
	    char tmp[_MAX_PATH];
	    da->targ.week = atweek;
	    da->targ.julian = nldaynumber((time_t) (da->publish - (atweek * SECSPERDAY * 7)));
	    add_extn(strcpy(tmp, sb_string(&da->sbuf, da->orig.file)), da->targ.julian);
	    da->targ.file = sb_alloc(&da->sbuf, tmp);
	    logit(LOG_MARK, "Starting with nodelist %s (%d)", sb_string(&da->sbuf, da->orig.file), da->orig.julian);
	    logit(LOG_MARK, "Compiling to  nodelist %s (%d)", tmp, da->targ.julian);
	}
    }
    return array_items(&da->diffs);
}


/* diffapp_getline() - Get next line from source nodelist or difference file(s)
 */

static int
diffapp_getline(DIFFAPP * da, FILE *fp, char const * filename, unsigned *lineno)
{
    int len;
    if (fp == NULL || fgets(da->linebuf, MAXLINEBUF - 1, fp) == NULL || *da->linebuf == 0x1a)
	return 0;
    len = strlen(da->linebuf);
    if (da->linebuf[len - 1] != '\n') {	/* No newline? */
	logit(LOG_ERROR, "%s(%u): line too long to process", filename, *lineno);
	return -1;
    }
    (*lineno)++;
    return len;
}


/* diffapp_putline() - Write line to target nodelist, update CRC-16
 */

static int
diffapp_putline(DIFFAPP * da, int len)
{
    NLFILE *nlf = &da->targ;
    if (nlf->lineno)
	nlf->crc16 = update_crc16((unsigned char const *) da->linebuf, len, nlf->crc16);
    else
	nlf->crc16 = 0;		/* First line not included in CRC calculation */
    len = fwrite(da->linebuf, 1, len, nlf->fp);
    nlf->lineno++;
    return len;
}


/* diffapp_execdiff() - This is called to get the next line from a
 * specific diff merge It is used recursively to retrieve a line from
 * each of the previous diffs until the last, where it fetches lines
 * from the original nodelist. This input is then filtered upwards
 * through more frequent diffs until it is either deleted or copied
 * to the output file.
 */

static int
diffapp_execdiff(DIFFAPP * da, unsigned idx)
{
    DFFILE *diff;
    char const *diffname;
    if (idx == array_items(&da->diffs))
	return diffapp_getline(da, da->orig.fp, sb_string(&da->sbuf, da->orig.file), &da->orig.lineno);

    diff = array_ptr(&da->diffs, idx);
    diffname = sb_string(&da->sbuf, diff->file);

    for (;;) {
	int len = 0;
	/*
	 * What are we currently doing with our input?
	 */
	switch (diff->state) {
	default:		/* case DA_EOL: */
	    return 0;

	case DA_Init:		/* In init state - go get a command */
	    if (diff->fp == NULL) {

		if ((diff->fp = fopen(diffname, "rb")) == NULL) {
		    logit(LOG_ERROR, "Unable to open %s: %s", diffname, strerror(errno));
		    return -1;
		}
		/*
		 * Junk the first line, our instructions begin on the next
		 */
		logit(LOG_MARK, "Including difference file %s (%d)", diffname, diff->julian);
		diffapp_getline(da, diff->fp, diffname, &diff->lineno);
		diff->crc16 = 0;
	    }
	    /*
	     * Fetch a new command
	     */
	    if ((len = diffapp_getline(da, diff->fp, diffname, &diff->lineno)) == -1)
		return -1;

	    if (len == 0) {	/* Nothing more to do */
		diff->state = DA_EOL;	/* Legitimate end of input */
		return 0;
	    }
	    switch (*da->linebuf) {
	    case 'A':
		diff->state = DA_Add;
		break;

	    case 'D':
		diff->state = DA_Del;
		break;

	    case 'C':
		diff->state = DA_Copy;
		break;

	    default:
		logit(LOG_ERROR, "%s(%u): unable to execute command '%c'", diffname, &diff->lineno, *da->linebuf);
		return -1;
		break;
	    }
	    diff->counter = (unsigned short) atol(da->linebuf + 1);
	    break;

	case DA_Copy:
	    len = diffapp_execdiff(da, idx + 1);
	    goto nxtline;

	case DA_Add:		/* Adding a line */
	    len = diffapp_getline(da, diff->fp, diffname, &diff->lineno);

    nxtline:

	    if (--diff->counter == 0)
		diff->state = DA_Init;
	    if (len > 0) {

		/*
		 * Accumulate the CRC going to the diff's 'output' file
		 */
		diff->crc16 = update_crc16((unsigned char const *) da->linebuf, len, diff->crc16);

		/*
		 * More diffs to filter, pass this back up to them
		 */
		if (idx)
		    return len;

		/*
		 * If this is our final diff, then we add this to the output
		 * file
		 */
		len = diffapp_putline(da, len);
	    }
	    break;

	case DA_Del:		/* Deleting a line */
	    len = diffapp_execdiff(da, idx + 1);
	    if (--diff->counter == 0)
		diff->state = DA_Init;
	    break;

	}

	if (len == -1)		/* Error */
	    return -1;

    }
}
/* diffapp_close() - Close all nodediff files
 */

void
diffapp_close(DIFFAPP * da)
{
    ArrayIter iter;
    DFFILE *diff;
    arrayIter_init(&iter, &da->diffs);
    while ((diff = arrayIter_get(&iter, NEXT)) != NULL) {
	if (diff->fp) {
	    fclose(diff->fp);
	    diff->fp = NULL;
	}
    }
}
/* diffapp_apply() - Apply nodediffs to product a target nodelist
 */

int
diffapp_apply(DIFFAPP * da)
{
    int rc = -1;
    if (diffapp_settarget(da)) {
	struct stat st;
	char const *src = sb_string(&da->sbuf, da->orig.file);
	char const *dst = sb_string(&da->sbuf, da->targ.file);
	/*
	 * grab the date/time information from the most recent diff file
	 */
	DFFILE *diff = array_ptr(&da->diffs, 0);
	if (diff == NULL || stat(sb_string(&da->sbuf, diff->file), &st) != 0)
	    memset(&st, 0, sizeof st);

	/*
	 * Let's open our source and destination lists
	 */
	if ((da->orig.fp = fopen(src, "rb")) == NULL)
	    logit(LOG_ERROR, "Unable to open %s: %s", src, strerror(errno));
	else {

	    if ((da->targ.fp = fopen(dst, "wb")) == NULL)
		logit(LOG_ERROR, "Unable to create %s: %s", dst, strerror(errno));
	    else {

		/*
		 * Now, the fun starts... we start at the first diff file to
		 * process and snap instructions off the top of the diff file,
		 * and execute each in turn, grabbing the "input" from an
		 * earlier diff file and ultimately the original nodelist.
		 */
		if ((rc = diffapp_execdiff(da, 0)) == 0)
		    fwrite("\x1a", 1, 1, da->targ.fp);	/* ^Z at end */
		diffapp_close(da);
		if (fclose(da->targ.fp) == EOF) {
		    logit(LOG_ERROR, "Write error on %s", dst);
		    rc = -1;
		}
		da->targ.fp = NULL;
	    }

	    fclose(da->orig.fp);
	    da->orig.fp = NULL;
	}

	/*
	 * If everything went ok, then let's check out the serial number
	 */
	if (rc == 0) {
	    short applies_to;
	    unsigned short crc16;
	    rc = -1;		/* Default to error again */
	    if (nlfiletype(dst, &applies_to, &crc16, NULL) != F_LIST)
		logit(LOG_ERROR, "Unexpected format error in %s", dst);
	    else if (crc16 != da->targ.crc16)
		logit(LOG_ERROR, "CRC mismatch (expected %u : got %u)", crc16, da->targ.crc16);
	    else {
		if (st.st_mtime)/* Fix target date/time to same as last diff */
		    setfiletimes(dst, &st);
		rc = 0;		/* Ok! */
	    }
	}
	if (rc != 0) {
	    /*
	     * We need to kill the output if it is incomplete or wrong
	     */
	    remove(dst);
	}
    }
    return rc;
}


/* nlfiletype() - Determine type of file we are looking at.
 * Not strictly part of this module, but it is used a lot here
 * Password checking also provided
 */

int
nlfiletype(char const * path, short *applies_to, unsigned short *crc16, char const * pwd)
{
    int rc = F_UNKNOWN;

    FILE *fp = fopen(path, "r");
    if (fp != NULL) {
	char *p;
	char fline[128];
	static char const *nlstrs[] =
	{
	    ";A ", " -- Day number "
	};
	if (fgets(fline, sizeof fline - 1, fp) != NULL &&
	    strncmp(fline, nlstrs[0], 3) == 0 &&
	    (p = strstr(fline, nlstrs[1])) != NULL) {

	    p += strlen(nlstrs[1]);
	    if (applies_to)
		*applies_to = (short) atoi(p);
	    while (*p && isdigit(*p))	/* day number */
		++p;
	    while (*p && !isdigit(*p))	/* " : " */
		++p;
	    if (*p) {		/* at crc */
		int pwderr = 0;
		if (crc16)
		    *crc16 = (unsigned short) atol(p);

		if (pwd && *pwd) {

		    /*
		     * With nodelists, we need to check this on the first line
		     */
		    pwderr = 1;	/* If this turns out to be a diff, we disregard
				 * it anyway */
		    if (fline[3] == '@' && memcmp(fline + 4, pwd, strlen(pwd)) == 0)
			pwderr = 0;
		}
		if (fgets(fline, sizeof fline - 1, fp) != NULL) {

		    if (*fline != 'D')	/* Not a diff? */
			rc = F_LIST;
		    else {
			rc = F_DIFF;

			/*
			 * With passworded nodediffs, we must look for the
			 * password on the 4th line, as this is the first line
			 * of the new nodelist.
			 */
			if (pwd && *pwd) {	/* Only do this if we are
						 * actually asked */
			    pwderr = 1;	/* Assume error */
			    if (fgets(fline, sizeof fline - 1, fp) != NULL && *fline == 'A' &&	/* Line 3, add */
				fgets(fline, sizeof fline - 1, fp) != NULL && *fline == ';' &&	/* Line 4, first line of
												 * new list */
				fline[3] == '@' && memcmp(fline + 4, pwd, strlen(pwd)) == 0)	/* Must be ;A
												 * @<password> *
												 * nodelist for ... */
				pwderr = 0;	/* Passed muster */
			}
		    }

		    if (pwderr)
			rc |= F_PWDERR;

		}
	    }
	}
	fclose(fp);
    }
    return rc;
}


/* nldaynumber() - Determines the day number of a nodelist from a ctime
 */

short
nldaynumber(time_t date)
{
    struct tm *T = localtime(&date);
    return (short) (T ? (T->tm_yday + 1) : -1);
}
/* weeksold() - Determines how many weeks old a specific day number is based
 *              on some reference date (which determines the day of week).
 */

short
weeksold(time_t epoch, short daynum)
{
    short startat = 0;
    do {
	short atday = nldaynumber(epoch);
	if (atday == daynum)
	    return startat;
	epoch -= SECSPERDAY * 7;
    }				/* More than 1 year, we have to junk it */
    while (++startat < 52);
    return -1;
}
