/* $Id: nledit.c,v 1.1.1.1 1996/10/09 11:26:50 davidn Exp $
 * Nodelist editor (applies diff files)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "nlmaint.h"
#include "aplydiff.h"
#include "posix.h"
#include "log.h"

#include "version.h"

char const progname[] = "NLEdit";
char const progabbr[] = "NLED";
char const version[] = VERSION;

typedef struct _nledit {
    short dow;			/* Publishing day of week */
    short cleanup;		/* Cleanup old lists and diffs? */
    short logformat;		/* Format of log file */
    short julian;		/* Julian day for next publication date */
    time_t publication;		/* Next date of publication */
    char const *orig;		/* Original nodelist name */
    char const *diff;		/* Nodediff name */
    char const *logf;		/* Logfile name */
}   NLECTL;

int
main(int argc, char *argv[])
{
    int rc = 1, fnum = 0, carg = 0, days;
    time_t now;
    struct tm *T;
    FFIND *ff;
    NLECTL nle;
    DIFFAPP da;
    char tmp[32];
    memset(&nle, 0, sizeof nle);
    nle.orig = "nodelist";
    nle.diff = "nodediff";
    nle.dow = V_FRIDAY;
    nle.logf = NULL;

    printf("%s; FidoNet Nodelist File Updater; Ver %s\n"
	   "by " AUTHOR "\n"
	   "Compiled for " HOSTOS " on " __DATE__ " at " __TIME__ "\n\n",
	   progname, version);

    tzset();
    while (++carg < argc) {
	char *argp = argv[carg];
#ifdef DOSISH
	if (*argp == '-' || *argp == '/')	/* command switch */
#else
	if (*argp == '-')	/* command switch */
#endif
	{
	    char ch = *++argp;
#ifdef DOSISH
	    ch = (char) toupper(ch);
#endif
	    switch (ch) {
	    case 'L':		/* Log file name */
		if (!*++argp)
		    argp = argv[++carg];
		nle.logf = argp;
		break;

	    case 'F':		/* Format of log file */
		if (!*++argp)
		    argp = argv[++carg];
		if (argp)
		    nle.logformat = (short) atoi(argp);
		break;

	    case 'D':		/* Specify a publishing day */
		if (!*++argp)
		    argp = argv[++carg];
		if (argp) {
		    static char const dows[] = "SunMonTueWedThuFriSat";
		    if (isdigit(*argp))
			nle.dow = (short) atoi(argp);
		    else
			for (nle.dow = 0; nle.dow < 7 && strnicmp(dows + (nle.dow * 3), argp, 3) != 0; ++nle.dow);
		    if (nle.dow < 0 || nle.dow >= 7)
			fatal(EL_CMDERROR, "Invalid day of week %s", argp);
		}
		break;

	    case 'C':		/* Cleanup after done */
		nle.cleanup = 1;
		break;

	    case 'H':
	    case '?':		/* Help! */
		printf("usage: %s [options] [nodelist_name] [nodediff_name]\n"
		       "Unless provided nodelist_name=\"nodelist\" & nodediff_name=\"nodediff\"\n"
		       "Available options:\n"
		       "  -c        Cleanup old files if build is successful\n"
		       "  -d <day>  Specify day of week nodelist is published (default=Friday)\n"
		       "            Provide day name or number, where 0=Sunday, 1=Monday etc.\n"
		   "  -l <file> Set the name of the log file (default=none)\n"
		       "  -f #      Set log file format; 0=default Bink/Opus, 1=FrontDoor\n",
		       progname);
		return 0;

	    default:
		fatal(EL_CMDERROR, "Unknown command line switch %s (try -?)", --argp);
	    }
	} else
	    switch (fnum++) {
	    case 0:		/* Nodelist name */
		nle.orig = argp;
		break;
	    case 1:
		nle.diff = argp;
		break;
	    default:
		fatal(EL_CMDERROR, "unexpected word on command line '%s'", argp);
	    }
    }

    openlog(nle.logf, nle.logformat, -1);

    now = time(NULL);		/* Get current data to determine targets */
    T = localtime(&now);
    /* Calculate number of days forward that next processing date falls */
    days = (T->tm_wday > nle.dow) ? (7 - (T->tm_wday - nle.dow))
	    : nle.dow - T->tm_wday;
    /*
     * Then, let's find out the date, and therefore the julian day of
     * publication
     */
    nle.publication = now + (days * SECSPERDAY);
    nle.julian = nldaynumber(nle.publication);

    diffapp_init(&da, nle.publication);
    if (access(nle.orig, F_OK) == 0 && diffapp_nlfile(&da, nle.orig))
	fnum = 1;
    else {
	fnum = 0;
	strncpy(tmp, last_component(nle.orig), sizeof tmp - 5);
	tmp[sizeof tmp - 5] = '\0';
	strcpy(path_ext(tmp), ".###");
	ff = file_findfirst(get_path(NULL, nle.orig), tmp);
	if (ff != NULL) {
	    do
		if (file_isreg(ff) && diffapp_nlfile(&da, file_path(ff, NULL)))
		    ++fnum;
	    while (file_findnext(ff));
	    file_findclose(ff);
	}
    }

    if (!fnum)
	logit(LOG_RESULT, "No nodelists '%s' to process", nle.orig);
    else {
	/* Now look for diffs */
	fnum = 0;
	strncpy(tmp, last_component(nle.diff), sizeof tmp - 5);
	tmp[sizeof tmp - 5] = '\0';
	strcpy(path_ext(tmp), ".?##");
	ff = file_findfirst(get_path(NULL, nle.diff), tmp);
	if (ff != NULL) {
	    do
		if (file_isreg(ff) && diffapp_nldiff(&da, file_path(ff, NULL)))
		    ++fnum;
	    while (file_findnext(ff));
	    file_findclose(ff);
	}
	if (!fnum)
	    logit(LOG_RESULT, "No nodediffs '%s' to process (-? for help)", nle.diff);
	else {
	    char const *src = sb_string(&da.sbuf, da.orig.file);
	    fnum = diffapp_settarget(&da);
	    if (!fnum) {
		logit(LOG_WARN, "/ Either nodelist '%s' is up to date", src);
		logit(LOG_WARN, "\\ or one or more difference files are missing");
	    } else if (diffapp_apply(&da) == -1)
		logit(LOG_ERROR, "Error during apply, no cleanup done");
	    else {
		if (!nle.cleanup)
		    logit(LOG_RESULT, "Application successful, cleanup not enabled");
		else {
		    unsigned idx;
		    DFFILE *fdif;
		    logit(LOG_RESULT, "Application successful, cleaning up leftover files");
		    logit(LOG_PROGRESS, remove(src) == 0 ? "Deleted %s" : "Error deleting %s (%d)", src, errno);
		    for (idx = 0; (fdif = array_ptr(&da.diffs, idx)) != NULL; ++idx) {
			src = sb_string(&da.sbuf, fdif->file);
			logit(LOG_PROGRESS, remove(src) == 0 ? "Deleted %s" : "Error deleting %s (%d)", src, errno);
		    }
		}
		rc = 0;
	    }
	}
    }

    openlog(NULL, -1, 0);
    return rc;
}
