/* $Id: filefind.c,v 1.1.1.1 1996/10/09 11:25:20 davidn Exp $
 * File search functions
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "osdep.h"
#include "mem.h"
#include "parse.h"

#ifdef DOSISH
#include <process.h>
#include <io.h>
#if !defined(__EMX__)
#include <direct.h>
#endif

/* "portable" directory functions
 */

#if defined(__OS2__) || defined(__EMX__)
#define INCL_DOS
#include "os2.h"
#undef TRUE
#undef FALSE
#undef  _SYSINT
#if defined(__FLAT__) || defined(__EMX__)
#undef  _FILEFINDBUF
#define _FILEFINDBUF _FILEFINDBUF3
#define _SYSINT ULONG
#define FINDLVL FIL_STANDARD
#else
#define _SYSINT USHORT
#define FINDLVL 0L
#endif
#define ALLFILES  "*"

struct ffind {
    HDIR dh;
    char *directory;
    char *pattern;
    struct _FILEFINDBUF f;
};
#define ff_name(x)     (x)->f.achName
#define ff_size(x)     (x)->f.cbFile
#define ff_attr(x)     (x)->f.attrFile
#define ff_date(x)     *((DOSFileDate *)(&(x)->f.fdateLastWrite))
#define ff_yr(x)       *((DOSFileDate *)(&(x)->f.fdateLastWrite)).year
#define ff_mo(x)       *((DOSFileDate *)(&(x)->f.fdateLastWrite)).month
#define ff_day(x)      *((DOSFileDate *)(&(x)->f.fdateLastWrite)).day
#define ff_time(x)     *((DOSFileTime *)(&(x)->f.ftimeLastWrite))
#define ff_hr(x)       *((DOSFileTime *)(&(x)->f.ftimeLastWrite)).hours
#define ff_min(x)      *((DOSFileTime *)(&(x)->f.ftimeLastWrite)).mins
#define ff_tsec(x)     *((DOSFileTime *)(&(x)->f.ftimeLastWrite)).tsecs

#elif defined(_WIN32) || defined(WIN32)

#include <windows.h>
#define ALLFILES  "*.*"

typedef struct ffind {
    HANDLE handle;
    char *directory;
    char *pattern;
    WIN32_FIND_DATA file;
};
#define ff_name(x)     (x)->file.cFileName
#define ff_size(x)     (x)->file.nFileSizeLow
#define ff_attr(x)     (x)->file.dwFileAttributes

WORD ff_date(DOSFileData * f);
WORD ff_yr(DOSFileData * f);
WORD ff_mo(DOSFileData * f);
WORD ff_day(DOSFileData * f);
WORD ff_time(DOSFileData * f);
WORD ff_hr(DOSFileData * f);
WORD ff_min(DOSFileData * f);
WORD ff_tsec(DOSFileData * f);
#elif defined(__MSDOS__) || defined(MSDOS)

/* Force byte alignment for this braindead struct */

#if defined(_MSC_VER) || defined(_QC) || defined(__WATCOMC__)
#pragma pack(1)
#elif defined(__ZTC__)
#pragma ZTC align 1
#elif defined(__TURBOC__) && (__TURBOC__ > 0x202)
#pragma option -a-
#endif
#ifdef __GNUC__
#define PAK      __attribute__((packed))
#else
#define PAK
#endif
#define ALLFILES  "*.*"

#ifdef __TURBOC__
#include "dir.h"
#endif
#include "dos.h"

 /*
  * Structures to access dates and times
  */

typedef struct {
    unsigned tsecs:5;
    unsigned mins:6;
    unsigned hours:5;
}   DOSFileTime;

typedef struct {
    unsigned day:5;
    unsigned month:4;
    unsigned year:7;
}   DOSFileDate;

typedef struct ffind {
    char reserved[21];
    char attrib;
    DOSFileTime time;
    DOSFileDate date;
    long d_size;
    char d_name[13];
    char *directory;
    char *pattern;
};
#define ff_name(x)     (x)->d_name
#define ff_size(x)     (x)->d_size
#define ff_attr(x)     (x)->attrib
#define ff_date(x)     (x)->date
#define ff_yr(x)       (x)->date.year
#define ff_mo(x)       (x)->date.month
#define ff_day(x)      (x)->date.day
#define ff_time(x)     (x)->time
#define ff_hr(x)       (x)->time.hours
#define ff_min(x)      (x)->time.mins
#define ff_sec(x)      (x)->time.secs

 /* set structure alignment back to default */

#if defined (_MSC_VER) || defined(_QC) || defined(__WATCOMC__)
#pragma pack()
#elif defined (__ZTC__)
#pragma ZTC align
#elif defined(__TURBOC__) && (__TURBOC__ > 0x202)
#pragma option -a.
#endif
#endif


/* file_findqualify() - see if file matches pattern
 */

static int
file_findqualify(FFIND * ff)
{
    return wildmatch(ff_name(ff), ff->pattern, 1);
}


/* file_findfirst() - initiate a directory scan with optional pattern
 */

FFIND *
file_findfirst(char const * dir, char const * patt)
{
    static char const fn[] = "file_findfirst";
#if defined(__MSDOS__)
#elif defined(__OS2__)
    _SYSINT cnt = 1;
#endif				/* defined(_WIN32) || defined(WIN32) */
    char spec[_MAX_PATH];
    FFIND *ff = zmalloc(fn, sizeof(FFIND));
    build_path(spec, dir, ALLFILES, NULL);
    memset(ff, 0, sizeof(FFIND));
#if defined(__MSDOS__)
#if defined (_MSC_VER) || defined(_QC) || defined(__WATCOMC__)
    if (_dos_findfirst(spec, _A_HIDDEN | _A_SYSTEM | _A_SUBDIR | _A_ARCH, (struct find_t *) ff) != 0)
#else
    if (findfirst(spec, (struct ffblk *) ff, _A_HIDDEN | _A_SYSTEM | _A_SUBDIR | _A_ARCH) != 0)
#endif
    {
	zfree(fn, ff);
	return NULL;
    }
#elif defined(__OS2__)
    ff->dh = (HDIR) - 1;
    if (DosFindFirst((PBYTE) spec, &ff->dh, (_SYSINT) FILE_HIDDEN | FILE_SYSTEM | FILE_DIRECTORY | FILE_ARCHIVED,
	 &ff->f, (_SYSINT) sizeof(struct _FILEFINDBUF), &cnt, FINDLVL) != 0) {
	zfree(fn, ff);
	return NULL;
    }
#else				/* defined(_WIN32) || defined(WIN32) */
    if ((ff->handle = FindFirstFile(spec, &(ff->file))) == INVALID_HANDLE_VALUE) {
	zfree(fn, ff);
	return NULL;
    }
#endif
    /* We must allocate these now */
    ff->directory = zstrdup(fn, dir ? dir : "");
    ff->pattern = zstrdup(fn, patt ? patt : "*");
    if (!file_findqualify(ff) && !file_findnext(ff)) {
	file_findclose(ff);
	return NULL;
    }
    return ff;
}


/* file_findnext() - scan for next file
 */

int
file_findnext(FFIND * ff)
{
    do {
#ifdef __MSDOS__
#if defined (_MSC_VER) || defined(_QC) || defined(__WATCOMC__)
	if (_dos_findnext((struct find_t *) ff) != 0)
#else
	if (findnext((struct ffblk *) ff) != 0)
#endif
	    return 0;
#elif defined(__OS2__)
	_SYSINT cnt = 1;
	if (DosFindNext(ff->dh, &ff->f, sizeof(struct _FILEFINDBUF), &cnt))
	    return 0;
#else				/* defined(_WIN32) || defined(WIN32) */
	if  (FindNextFile(ff->handle, &(ff->file)) == 0)
	        return 0;
#endif
    }
    while (!file_findqualify(ff));
    return 1;
}


/* file_findclose() - end file scan
 */

void
file_findclose(FFIND * ff)
{
    if (ff) {
	static char const fn[] = "file_findclose";
#ifdef __MSDOS__

	/*
	 * This might seem weird, but this helps free up directory handles on a
	 * Netware server, which otherwise leaves them allocated (and may
	 * eventually run out), so we keep doing a findnext() until we get to
	 * the end of directory
	 */
	while (file_findnext(ff));
#if defined (_MSC_VER) || defined(_QC) || defined(__WATCOMC__)
	_dos_findclose((struct find_t *) ff);
#endif
#elif defined(__OS2__)
	DosFindClose(ff->dh);
#else				/* defined(_WIN32) || defined(WIN32) */
	FindClose(ff->handle);
#endif
	zfree(fn, ff->directory);
	zfree(fn, ff->pattern);
	zfree(fn, ff);
    }
}


/* file_dir() - return directory portion of a file search object
 */

char const *
file_dir(FFIND * ff)
{
    return ff->directory;
}


/* file_name() - return current filename
 */

char const *
file_name(FFIND * ff)
{
    return ff_name(ff);
}


/* file_path() - return full path name of current file
 */

char const *
file_path(FFIND * ff, char *buf)
{
    return build_path(buf, ff->directory, ff_name(ff), NULL);
}


/* file_attr() - return attributes of current file
 */

unsigned
file_attr(FFIND * ff)
{
    return ff_attr(ff);
}


/* file_size() - return size of current file
 */

long
file_size(FFIND * ff)
{
    return ff_size(ff);
}


/* file_date() - return date of current file
 */

time_t
file_date(FFIND * ff)
{
    ff = ff;
    return -1;
}


/* file_isdir() - see if current entry is a directory
 */

int
file_isdir(FFIND * ff)
{				/* Is file a directory? */
#if defined(__MSDOS__) || defined(_WIN32) || defined(WIN32)
    return !!(file_attr(ff) & _A_SUBDIR);
#elif defined(__OS2__)
    return !!(file_attr(ff) & FILE_DIRECTORY);
#elif				/* */
    return !!(0);
#endif
}


/* file_isexe() - see if current entry is an executable
 */

int
file_isexe(FFIND * ff)
{				/* Is file an executable? */
#if defined(__MSDOS__) || defined(_WIN32) || defined(WIN32) || defined(__OS2__)
    /* We have to do this the hard way, from the name */
    char *p = strrchr(ff_name(ff), '.');
    if (p != NULL && (stricmp(p, ".com") == 0 || stricmp(p, ".exe") == 0))
	return 1;
#elif				/* */
#endif
    return 0;
}


/* file_isreg() - see if current entry is a regular file
 */

int
file_isreg(FFIND * ff)
{				/* Is file a regular file? */
    return !(file_isdir(ff));	/* We may need to do something more compelx
				 * than this */
}


/* file_isdev() - see if current entry is a device
 */

int
file_isdev(FFIND * ff)
{				/* Is flie a device? 1=chr 2=block */
    ff = ff;
    return 0;			/* Cheat, for the time being (until we need it
				 * :-)) */
}
#else

#include <unistd.h>
#include <dirent.h>

#if defined(POSIXDIR)

struct ffind {
    DIR *dir;
    struct dirent *ent;
    char *directory;
    char *pattern;
    int gotstat;
    struct stat st;
};
#define ff_name(x)     (x)->ent->d_name
#define ff_size(x)     (x)->st.st_size


/* file_findqualify() - see if file matches pattern
 */

static int
file_findqualify(FFIND * ff)
{
    return wildmatch(ff_name(ff), ff->pattern, 1);
}


/* file_findfirst() - initiate a directory scan with optional pattern
 */

FFIND *
file_findfirst(char const * dir, char const * patt)
{
    static char const fn[] = "file_findfirst";
    FFIND *ff = zmalloc(fn, sizeof(FFIND));
    memset(ff, 0, sizeof(FFIND));
    if ((ff->dir = opendir(dir)) == NULL || (ff->ent = readdir(ff->dir)) == NULL) {
	if (ff->dir)
	    closedir(ff->dir);
	zfree(fn, ff);
	return NULL;
    }
    /* We must allocate these now */
    ff->directory = zstrdup(fn, dir ? dir : "");
    ff->pattern = zstrdup(fn, patt ? patt : "*");
    if (!file_findqualify(ff) && !file_findnext(ff)) {
	file_findclose(ff);
	return NULL;
    }
    return ff;
}
/* file_findnext() - scan for next file
 */

int
file_findnext(FFIND * ff)
{
    do {
	if ((ff->ent = readdir(ff->dir)) == NULL)
	    return 0;
	memset(&ff->st, 0, sizeof(ff->st));
    }
    while (!file_findqualify(ff));
    return 1;
}


/* file_findclose() - end file scan
 */

void
file_findclose(FFIND * ff)
{
    if (ff) {
	static char const fn[] = "file_findclose";
	if (ff->dir)
	    closedir(ff->dir);
	zfree(fn, ff->directory);
	zfree(fn, ff->pattern);
	zfree(fn, ff);
    }
}


/* file_dir() - return directory portion of a file search object
 */

char const *
file_dir(FFIND * ff)
{
    return ff->directory;
}


/* file_name() - return current filename
 */

char const *
file_name(FFIND * ff)
{
    return ff_name(ff);
}


/* file_path() - return full path name of current file
 */

char const *
file_path(FFIND * ff, char *buf)
{
    return ff->ent ? build_path(buf, ff->directory, ff_name(ff), NULL) : NULL;
}


/* getfileinfo() - retrieve stat info for current file
 */

static int
getfileinfo(FFIND * ff)
{
    if (ff->st.st_dev == 0) {
	char buf[_MAX_PATH];
	if (file_path(ff, buf) == NULL || stat(buf, &ff->st) == -1)
	    return 0;
    }
    return 1;
}


/* file_attr() - return attributes of current file
 */

unsigned
file_attr(FFIND * ff)
{
    return getfileinfo(ff) ? ff->st.st_mode : 0;
}


/* file_size() - return size of current file
 */

long
file_size(FFIND * ff)
{
    return getfileinfo(ff) ? ff_size(ff) : -1L;
}


/* file_date() - return date of current file
 */

time_t
file_date(FFIND * ff)
{
    return getfileinfo(ff) ? ff->st.st_mtime : 0L;
}


/* file_isdir() - see if current entry is a directory
 */

int
file_isdir(FFIND * ff)
{				/* Is file a directory? */
    return getfileinfo(ff) ? S_ISDIR(ff->st.st_mode) : 0;
}



/* file_isexe() - see if current entry is an executable
 */

int
file_isexe(FFIND * ff)
{				/* Is file an executable? */
    return getfileinfo(ff) ? !!(ff->st.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH)) : 0;
}


/* file_isreg() - see if current entry is a regular file
 */

int
file_isreg(FFIND * ff)
{				/* Is file a regular file? */
    return getfileinfo(ff) ? S_ISREG(ff->st.st_mode) : 0;
}


/* file_isdev() - see if current entry is a device
 */

int
file_isdev(FFIND * ff)
{				/* Is flie a device? 1=chr 2=block */
    return getfileinfo(ff) ? (S_ISCHR(ff->st.st_mode) || S_ISBLK(ff->st.st_mode)) : 0;
}
#else
#error No directory search implementation!
#endif

#endif
