/* $Id: cmdtpl.c,v 1.1.1.1 1996/10/09 11:25:14 davidn Exp $
 * Template command support
 *
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "posix.h"
#include "cmdtpl.h"
#include "mem.h"
#include "log.h"


#define BUF_SIZE   2048

/* tpl_make() - create command from a command template
 */

char *
tpl_make(char const * tpl, cmdtpl * substinfo, int maxbuf)
{
    static char const fn[] = "tpl_make";
    char *cmdbuf = NULL;
    if (tpl != NULL) {
	static char const ws[] = " \t";

	int sl = strlen(tpl) + 1;
	/*
	 * Let's make sure this is reasonable
	 */
	if (maxbuf < (sl * 2))
	    maxbuf = sl * 2;

	if ((cmdbuf = zmalloc(fn, maxbuf)) != NULL) {
	    char *ptr = cmdbuf;
	    int cmdlen = strlen(strcpy(cmdbuf, tpl));
	    cmdtpl *t = substinfo;
	    /*
	     * Prepare the command template array
	     */
	    while (t->flags != TC_LAST) {
		t->work = NULL;
		t->len = (short) (t->arg ? strlen(t->arg) : 0);
		++t;
	    }

	    /*
	     * Handle all % translations
	     */

	    while ((ptr = strchr(ptr, TPLPFX)) != NULL) {
		int inslen, skip = 2;
		char *topos = ptr;
		char const *arg = NULL;
		/*
		 * Let's find out if we have a matching substitution
		 */

		for (t = substinfo; t->key != ptr[1] && t->flags != TC_LAST; ++t);

		if (t->len) {
		    switch (t->flags) {
		    case TC_STD:	/* Direct substitution */
			topos = ptr + t->len;
			arg = t->arg;
			break;

		    case TC_WORD:	/* Arg broken into words, used
					 * individually */
			if (t->work != NULL)	/* No work buffer yet */
			    arg = strtok(NULL, ws);
			else if ((t->work = zstrdup(fn, t->arg)) == NULL)
			    arg = NULL;
			else
			    arg = strtok(t->work, ws);
			if (arg == NULL)
			    t->len = 0;
			else
			    topos = ptr + strlen(arg);
			break;

		    case TC_TEMP:	/* Words into temporary file */
			if ((t->work = zstrdup(fn, file_tempname(NULL, NULL))) == NULL)
			    t->len = 0;
			else {
			    char *p = zstrdup(fn, t->arg);
			    if (p != NULL) {
				FILE *fp = NULL;
				if ((fp = fopen(t->work, "w")) == NULL)
				    logit(LOG_ERROR, "Failed to create temp file \"%s\": %s (%d)", t->work, strerror(errno), errno);
				else {
				    char *tmp;
				    for (tmp = strtok(p, ws); tmp != NULL; tmp = strtok(NULL, ws))
					fprintf(fp, "%s\n", tmp);
				    fclose(fp);
				    arg = t->work;
				    topos = ptr + strlen(t->work);
				}
				zfree(fn, p);
			    }
			}
			break;

		    case TC_DIR:	/* Directory part of filename */
			if (t->work == NULL)
			    t->len = strlen(t->work = zstrdup(fn, get_path(NULL, t->arg)));
			topos = ptr + t->len;
			arg = t->work;
			break;

		    case TC_NAME:	/* Name part of filename */
			topos = ptr + strlen(arg = last_component(t->arg));
			break;

		    case TC_BASE:	/* Base part of filename */
			if (t->work == NULL) {
			    t->work = zstrdup(fn, last_component(t->arg));
			    *path_ext(t->work) = '\0';
			    t->len = strlen(arg);
			}
			arg = t->work;
			topos = ptr + t->len;
			break;

		    case TC_EXT:	/* File extension */
			arg = path_ext((char *) t->arg);
			if (*arg)
			    ++arg;
			topos = ptr + strlen(arg);
			break;

		    default:
			if (ptr[1] == TPLPFX) {
			    ++ptr;
			    memmove(topos, ptr, cmdlen - (ptr - cmdbuf) + 1);
			    ++ptr;
			    --cmdlen;
			    continue;
			}
		    }
		}
		inslen = topos - ptr;

		if (inslen + cmdlen - skip >= maxbuf) {
		    logit(LOG_ERROR, "Template substitution exceeded buffer size of %u bytes", maxbuf);
		    tpl_dealloc(cmdbuf, substinfo);
		    return NULL;
		}
		if (topos != ptr + skip)
		    memmove(topos, ptr + skip, cmdlen - (ptr - cmdbuf) + 1);

		if (arg)
		    memmove(ptr, arg, inslen);

		ptr += inslen;
		cmdlen += (inslen - skip);
	    }

	    /*
	     * Let's size it down to be conservative
	     */
	    return zrealloc(fn, cmdbuf, strlen(cmdbuf) + 1);
	}
    }
    return NULL;
}


/* tpl_dealloc() - Deallocate memory used in template subst
 */

void
tpl_dealloc(char *buf, cmdtpl * substinfo)
{
    static char const fn[] = "tpl_dealloc";
    cmdtpl *t = substinfo;
    /*
     * Free any memory used during the process
     */
    while (t->flags != TC_LAST) {
	if (t->work != NULL) {
	    if (t->flags == TC_TEMP)
		remove(t->work);
	    zfree(fn, t->work);
	}
	t->work = NULL;
	++t;
    }
    zfree(fn, buf);
}


/* tpl_exec() - execute a command template
 */

int
tpl_exec(char const * tpl, cmdtpl * substinfo)
{
    int rc = -1;
    char *cmdbuf;
    /*
     * Reset this so it won't confuse the caller
     */
    errno = 0;
    if ((cmdbuf = tpl_make(tpl, substinfo, BUF_SIZE)) != NULL) {
	int waserrno;
	logit(LOG_MARK, "Executing \"%s\"", cmdbuf);

	/*
	 * Command is ready, we're ok to execute the command in 'cmdbuf'
	 */
	rc = file_execute(cmdbuf);

	/*
	 * Preserve errno in case execute failed
	 */
	waserrno = errno;

	if (rc == -1)
	    logit(LOG_ERROR, "Exec error \"%s\": %s (%d)", cmdbuf, strerror(errno), errno);
	else if (rc != 0)
	    logit(LOG_RESULT, "Returned with non-zero status (%d)", rc);
#if 0
	else
	    logit(LOG_PROGRESS, "Return with zero status", rc);
#endif

	/*
	 * Restore errno
	 */
	errno = waserrno;
    }
    return rc;
}
