/* $Id: array.c,v 1.1.1.1 1996/10/09 11:25:03 davidn Exp $
 * Generic/dynamic array handler
 *
 */

#include <string.h>
#include "array.h"
#include "mem.h"


/* array_alloc() - handle array (re)allocation
 */

static int
array_alloc(Array * A, arrindex tosize)
{

    /*
     * Only allocate if required
     */
    if (tosize > A->asize) {
	unsigned long memsize;
	char *newbase;
	/*
	 * If just adding a new element, grow by a delta
	 */
	tosize = (tosize == A->asize + 1) ? (tosize + (A->asize / 2)) : tosize;
	memsize = (unsigned long) tosize *A->elsize;	/* Calc required memory */
	newbase = zrealloc("array_alloc", A->base, (size_t) memsize);
	if (newbase == NULL)	/* Disaster */
	    return -1;
	A->base = newbase;
	A->asize = tosize;
	return 1;
    }
    return 0;			/* Nothing allocated */
}


/* array_setsize() - API to pre-set an array size for effiency
 */

void
array_setsize(Array * A, arrindex isize)
{
    array_alloc(A, isize);
}


/* array_init() - Initialise an array object
 */

Array *
array_init(Array * A, unsigned elsize, arrindex isize)
{
    A->elsize = elsize;		/* Size of individual elements */
    A->asize = 0;		/* Current size of array */
    A->aused = 0;		/* Zero number of elements used */
    A->base = 0;
    array_alloc(A, isize);	/* Allocate the initial array */
    return A;
}


/* array_new() - allocate a new array
 */

Array *
array_new(unsigned elsize, arrindex isize)
{
    return array_init(zmalloc("array_new", sizeof(Array)), elsize, isize);
}


/* array-destroy() - Destroy an array object
 */

void
array_destroy(Array * A)
{
    if (A)
	zfree("array_destroy", A->base);
}


/* array_delete() - free an array
 */

void
array_delete(Array * A)
{
    array_destroy(A);
    zfree("array_delete", A);
}


/* array_add() - add new element to array
 */

arrindex
array_add(Array * A, arrindex where, unsigned elems)
{
    arrindex at = A->aused;
    if (elems) {
	A->aused += elems;
	if (array_alloc(A, A->aused) == -1)
	    A->aused -= elems;
	else {
	    if (where < at)	/* Need to move elements in the array up? */
		memmove(array_ptr(A, where + elems), array_ptr(A, where),
			(size_t) ((at - where) * A->elsize));
	    else
		where = at;
	    memset(array_ptr(A, where), 0, (size_t) (elems * A->elsize));
	    return where;
	}
    }
    return NOELEMENT;
}


/* array_del() - delete elems from array
 */

arrindex
array_del(Array * A, arrindex where, unsigned elems)
{
    if (elems == NOELEMENT || (where + elems) >= A->aused)
	elems = (where > A->aused) ? 0 : (unsigned) (A->aused - where);
    else {			/* Need to move some elements down */
	arrindex top = where + elems;
	memmove(array_ptr(A, where), array_ptr(A, top),
		(size_t) ((A->aused - top) * A->elsize));
    }
    A->aused -= elems;
    return elems;		/* How many elements deleted */
}


/* array_ptr() - get pointer to an element
 */

void *
array_ptr(Array * A, arrindex idx)
{
    if (idx == NOELEMENT || idx >= A->aused)
	return NULL;
    return A->base + (A->elsize * idx);
}


/* array_reset() - reset an array
 */

void
array_reset(Array * A)
{
    A->aused = 0;
}


/* arrayIter_init() - array iterator
 */

void
arrayIter_init(ArrayIter * I, Array * A)
{
    I->A = A;
    I->index = NOELEMENT;
}


/* arrayIter_get() get an element
 */

void *
arrayIter_get(ArrayIter * I, int which)
{
    switch (which) {
	case PREV:
	if (I->index != NOELEMENT) {
	    --I->index;
	    break;
	}
	/* Fallthru */
    case LAST:
	I->index = (I->A)->aused - 1;
	break;

    case NEXT:
	if (I->index != NOELEMENT) {
	    ++I->index;
	    break;
	}
	/* Fallthru */
    case FIRST:
	I->index = 0;
	break;

    }

    if (I->index >= (I->A)->aused)	/* Auto reset at either end */
	I->index = NOELEMENT;

    return array_ptr(I->A, I->index);
}
