package com.oopsconsultancy.classfind;

import java.util.*;
import java.io.*;
import java.util.jar.*;

/**
 * provides the class cahce mechanism
 *
 * @author <a href="mailto:brian@oopsconsultancy.com">Brian Agnew</a>
 * @version $Id: ClassCache.java,v 1.2 2004/03/13 15:52:55 bagnew Exp $
 */
public class ClassCache {

  public class Pair {
    public String element;
    public String fullname;
    public Pair(String p, String name) {
      element = p;
      fullname = name;
    }
    public String toString() {
      return element + " -> " + fullname;
    }
  }

  private final static String CLASS_SUFFIX = ".class";

  private Map components = new HashMap();

  private String classpath = null;

  private boolean expired = false;

  /**
   * Gets the list of packages etc that the given class
   * can be found in
   *
   * @param classname
   * @return a list of matching classes
   */
  public List findClass(String classname) {
    List matches = new ArrayList();
    Set elements = components.keySet();
    Iterator ielements = elements.iterator();
    while (ielements.hasNext()) {
      String pkg = (String)ielements.next();
      List classes = (List)components.get(pkg);
      Iterator iclasses = classes.iterator();
      while (iclasses.hasNext()) {
        String fullname = (String)iclasses.next();
        if (fullname.equals(classname) || fullname.endsWith("." + classname)) {
          matches.add(new Pair(pkg, fullname));
        }
      }
    }
    return matches;
  }

  /**
   * Finds the list of matching packages from the cache
   *
   * @param packageName
   * @return the list of matching packages
   */
  public List findPackages(String packageName) {
    List matches = new ArrayList();
    Set elements = components.keySet();
    Iterator ielements = elements.iterator();
    while (ielements.hasNext()) {
      String pkg = (String)ielements.next();

      List classes = (List)components.get(pkg);
      Iterator iclasses = classes.iterator();
      while (iclasses.hasNext()) {
        String fullname = (String)iclasses.next();
        if (fullname.startsWith(packageName)) {
          // it's a contender
          // com.s -> com.sun.
          // com.sun. -> com.sun.[xyz].
          String remainder = fullname.substring(packageName.length());
          int index = remainder.indexOf(".");
          String res = null;
          if (index == 0) {
            index = remainder.indexOf(".", 0);
          }

          if (index == -1) {
            res = fullname;
          }
          else {
            res = fullname.substring(0, packageName.length() + index);
          }
          if (res != null && !matches.contains(res)) {
            matches.add(res);
          }
        }
      }
    }
    return matches;
  }

  public ClassCache(String cacheName, String classpath) throws IOException {
    load(cacheName);
    // we need to check if it's out of date, here
    if (this.classpath == null || !this.classpath.equals(classpath)) {
      expired = true;
      this.classpath = classpath;
    }
  }

  public void build(String classpath) {

    components.clear();

    StringTokenizer st = new StringTokenizer(classpath, File.pathSeparator);
    while (st.hasMoreTokens()) {
      String element = st.nextToken();

      if (components.get(element) == null && element.length() > 0) {
        File felement = new File(element);
        if (felement.exists()) {
          List classes = getClasses(felement);
          Iterator iclasses = classes.iterator();
          components.put(element, classes);
        }
        else {
          // System.err.println(element + " doesn't exist");
        }
      }
    }
  }

  /**
   * gets the list of classes from a directory (and subdirectories)
   * or the provided jar file
   *
   * @param felement the list of files
   * @return
   */
  private List getClasses(File felement) {
    // System.out.println("Checking " + felement);
    if (felement.isDirectory()) {
      return getClassesFromDir(felement);
    }
    else {
      // we'll assume a jar file here...
      return getClassesFromJar(felement);
    }
  }

  /**
   * opens up the jar file and lists the contents
   *
   * @param jarfile
   * @return the list of classes contained within
   */
  private List getClassesFromJar(File jarfile) {
    List classes = new ArrayList();
    try {
      JarFile jar = new JarFile(jarfile);
      Enumeration entries = jar.entries();
      while (entries.hasMoreElements()) {
        String name = entries.nextElement().toString();
        if (name.endsWith(CLASS_SUFFIX)) {
          classes.add(normalise(name));
        }
      }
    }
    catch (IOException e) {
      // e.printStackTrace();
    }
    return classes;
  }

  /**
   * gets the collection of classes within the
   * directory and its subcomponents
   *
   * @param directory the directory to start from
   * @return the set of classes in this dir and subdirs
   */
  private List getClassesFromDir(File directory) {
    return getClassesFromDir(directory, directory);
  }

  private List getClassesFromDir(File directory, File base) {
    List classes = new ArrayList();
    File[] contents = directory.listFiles();
    for (int c = 0; c < contents.length; c++) {
      if (contents[c].isDirectory()) {
        classes.addAll(getClassesFromDir(contents[c], base));
      }
      else if (contents[c].getName().endsWith(CLASS_SUFFIX)) {
        String name = contents[c].getAbsolutePath();
        name = name.substring(base.getAbsolutePath().length());
        classes.add(normalise(name));
      }
    }
    return classes;
  }

  private String normalise(String name) {
    // it may start with a file separator...
    if (name.startsWith(File.separator)) {
      name = name.substring(1);
    }

    // remove the suffix
    name = name.substring(0, name.length() - CLASS_SUFFIX.length());

    // and convert slashes to periods...
    StringTokenizer st = new StringTokenizer(name, File.separator);
    StringBuffer pkg = new StringBuffer();
    while (st.hasMoreTokens()) {
      pkg.append(st.nextToken());
      if (st.hasMoreTokens()) {
        pkg.append(".");
      }
    }
    return pkg.toString();
  }

  public void save(String name) throws IOException {
    FileWriter fw = new FileWriter(name);
    // System.err.println("Writing " + classpath);
    fw.write(classpath + "\n");

    Set elements = components.keySet();
    Iterator ielements = elements.iterator();
    while (ielements.hasNext()) {
      String pkg = (String)ielements.next();
      List classes = (List)components.get(pkg);
      Iterator iclasses = classes.iterator();
      while (iclasses.hasNext()) {
        String fullname = (String)iclasses.next();
        fw.write(pkg + "\t" + fullname + "\n");
      }
    }
    fw.close();
  }

  public boolean load(String name) throws IOException {
    if (!((new File(name)).exists())) {
      expired = true;
      return false;
    }
    FileReader fr=  new FileReader(name);
    BufferedReader bfr = new BufferedReader(fr);
    int num = 0;
    while (bfr.ready()) {
      String line = bfr.readLine();
      if (num == 0) {
        classpath = line;
      }
      else if (line.startsWith("#")) {
        // ignore comments
      }
      else if (line.trim().equals("")) {
        // ignore blank lines
      }
      else {
        int tabindex = line.indexOf("\t");
        if (tabindex != -1) {
          String element = line.substring(0, tabindex);
          String classname = line.substring(tabindex + 1);
          if (components.get(element) == null) {
            components.put(element, new ArrayList());
          }
          List list = (List)components.get(element);
          list.add(classname);
        }
      }
      num++;
    }
    bfr.close();
    return true;
  }

  public boolean isExpired()
  {
    return expired;
  }
}
