/* $Id: nlscan.c,v 1.2 1996/10/30 19:20:11 davidn Exp $
 * Scans a nodelist, check for parsing errors
 *
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include "wfile.h"
#include "fidoaddr.h"
#include "osdep.h"
#include "crc16.h"
#include "log.h"
#include "checknl.h"

#include "version.h"

char const progname[] = "NLScan";
char const progabbr[] = "NLSL";
char const version[] = VERSION;

enum {
    XIT_OK = 0,
    XIT_ERRFILE = 1,
    XIT_ERRFMT = 2,
    XIT_ERRCRC = 3,
    XIT_UNKNOWN = 4
};

int
main(int argc, char *argv[])
{
    int rc = XIT_ERRFILE;
    printf("%s; FidoNet Nodelist File Checker; Ver %s\n"
	   "by " AUTHOR "\n"
	   "Compiled for " HOSTOS " on " __DATE__ " at " __TIME__ "\n\n",
	   progname, version);

    tzset();
    if (argc < 2)
	fprintf(stderr, "usage: %s <nodelist_filename>\n", argv[0]);
    else {
	char *nodelist = argv[1];

	WFILE *w = wfile_new(nodelist);
	if (!wfile_open(w))
	    logit(LOG_ERROR, "Can't open %s: %s", nodelist, strerror(errno));
	else {
	    clock_t taken, start = clock();
	    unsigned lines = 0;
	    unsigned nodes = 0;
	    unsigned coords = 0;
	    unsigned admins = 0;
	    unsigned unknown = 0;
	    unsigned comments = 0;
	    unsigned short crcval = 0;
	    unsigned short listcrc = (unsigned short) -1;
	    unsigned errcount = 0;
	    enum _nltype type = NL_NONE;
	    char const *q;
	    char const *p;
	    char *s;
	    txtbuf tb;
	    char txtaddr[40] = "";
	    ZNNP ataddr;
	    int l, len;
	    char line[256];

	    memset(&ataddr, 0, sizeof ataddr);
	    /* init with reasonable defaults */
	    init_nlinfo(3,  /* minphone */
			4,  /* maxphone */
			0,  /* pvt node disposition */
			"", /* default baud rates */
			0,  /* unknown zone */
			0,  /* unknown net */
			3,  /* report invalid flags */
			0,  /* ignore invalid uflags */
			1); /* all redundancies checked */
	    rc = XIT_OK;
	    /* Errors buffer */
	    tb_init(&tb, 512);
	    logit(LOG_PROGRESS, "Scanning nodelist: %s", nodelist);
	    while ((p = wfile_ptr(w, &len)) != NULL) {
		wfile_analyse(w, p, len);

		if (lines++ > 0) {
		    q = p + len - 1;
		    while (q > p && (*q == '\r' || *q == '\n'))
			--q;
		    crcval = update_crc16((unsigned char const *) p,
					  q - p + 1, crcval);
		    crcval = update_crc16((unsigned char const *) "\r\n",
			                  2, crcval);
		} else {
		    if ((q = memchr(p, ':', len)) == NULL || (q += 2, !isdigit(*q))) {
			logit(LOG_FATAL, "Invalid file format (no valid CRC in file)");
			rc = XIT_ERRFMT;
			break;
		    }
		    listcrc = (unsigned short) atol(q);
		}
		if (!wfile_isvalid(w)) {
		    logit(LOG_ERROR, "%s(%u): Invalid line format",
			  nodelist, lines);
		    ++unknown;
		} else {

		    if (wfile_iscomment(w))
			++comments;	/* Don't parse */
		    else {
			unsigned long result;

    			if (wfile_isadmin(w)) {
    			    ++admins;
    			    if (wfile_iscoord(w)) {
    				++coords;
    				if (wfile_iszc(w)) {
    				    printf("Zone %3u  Reg %3s  \r",
    					   wfile_zone(w), "-");
				    type = (type == NL_NONE) ? NL_ZONE : NL_COMPOSITE;

				}
    				else if (wfile_isrc(w)) {
    				    printf("Zone %3u  Reg %3u  \r",
    					   wfile_zone(w), wfile_net(w));
				    if (type < NL_REGION)
					type = NL_REGION;
				} else if (wfile_isnc(w) && type < NL_NET)
				    type = NL_NET;
    				fflush(stdout);
    			    } else if (wfile_ishub(w) && type == NL_NONE)
				type = NL_HUB;
    			} else
    			    ++nodes;
    			/*
    			 * At this point, we have an apparently valid node record
    			 * We now need to parse it fully to check the various
    			 * fields.
    			 */
			q = p;
			l = len;
			while (l && (isspace(q[l-1]) || q[l-1] == '\n'))
			    --l;
			result = check_nodeent(&tb,
					       txtaddr,
					       (char **)&q,
					       &l,
					       &ataddr);
			if (result & (NE_SKIPENT|NE_ERRORENT)) {
			    logit(LOG_ERROR, "%s(%u): %s %.*s", nodelist,
				  lines, txtaddr, l, q);
			    ++errcount;
			}
			tb_seek(&tb, 0L, SEEK_SET);
			while (tb_gets(&tb, line, sizeof line - 1)) {
			    q = line + 1;
			    for (l = strlen(q); l && isspace(q[l-1]);)
				--l;
			    if ((s = strstr(q, " removed")) != NULL) {
				strcpy(s, s+8);  /* nothing to remove from! */
				l -= 8;
			    }
			    if (strstr(q, "redundant") != NULL)
				logit(LOG_WARN, "%.*s", l, q);
			    else
				logit(LOG_WARN, "%s(%u): %s %.*s", nodelist,
					lines, txtaddr, l, q);
			}
			tb_reset(&tb);
    		    }
		}
		wfile_advance(w, len);
	    }
	    len = type;
	    if (check_nodeent(&tb, NULL, NULL, &len, NULL) & NE_NOTRED) {
		tb_seek(&tb, 0L, SEEK_SET);
		while (tb_gets(&tb, line, sizeof line - 1)) {
		    q = line + 1;
		    for (l = strlen(q); l && isspace(q[l-1]);)
			--l;
		    logit(LOG_WARN, "%.*s", l, q);
		}
	    }
	    tb_reset(&tb);
	    taken = clock() - start;
	    /* Print out some stats for benchmarking */
	    logit(LOG_MARK, "%u line(s) scanned in %lu.%02lu seconds with %lu errors.",
		  lines, taken / CLOCKS_PER_SEC,
		  ((taken % CLOCKS_PER_SEC) * 100) / CLOCKS_PER_SEC,
		  errcount);
	    if (rc == XIT_OK) {
		if (listcrc == crcval) {
		    logit(LOG_RESULT, "CRC-16 (%u) is OK", listcrc);
		    if (unknown)
			rc = XIT_UNKNOWN;
		} else {
		    logit(LOG_ERROR, "CRC mismatch - got %u, needed %u",
			  crcval, listcrc);
		    rc = XIT_ERRCRC;
		}
		logit(LOG_RESULT, "    Comment lines : %6u", comments);
		logit(LOG_RESULT, " Coorinator nodes : %6u", coords);
		logit(LOG_RESULT, "Other Admin nodes : %6u", admins - coords);
		logit(LOG_RESULT, "  Non-admin nodes : %6u", nodes);
		logit(LOG_RESULT, "      Error lines : %6u", errcount);
		logit(unknown ? LOG_ERROR : LOG_RESULT, "    Invalid lines : %6u",
		      unknown);
	    }
	    wfile_close(w);
	}
    }
    logit(LOG_PROGRESS, "Exiting with status %u", rc);
    return 0;
}
