/* $Id: wfile.c,v 1.2 1996/10/30 19:18:56 davidn Exp $
 * Implements a read-only fast page & line-buffered file
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "osdep.h"
#include "posix.h"
#include "wfile.h"
#include "mem.h"
#include "log.h"

#if defined(__FLAT__) || defined(__386__) || !defined(DOSISH)
#define NLBUFSZ   32768		/* Optimal size */
#else
#define NLBUFSZ   4096		/* Limited memory */
#endif

/* wfile_new() - Create a WFILE object
 */

WFILE *
wfile_new(char const * name)
{
    return wfile_init(zmalloc("wfile_new", sizeof(WFILE)), name);
}
/* wfile_delete() - Destroy/deallocate a WFILE object
 */

void
wfile_delete(WFILE * w)
{
    wfile_destroy(w);
    zfree("wfile_delete", w);
}
/* wfile_init() - Initialise a WFILE object
 */

WFILE *
wfile_init(WFILE * w, char const * name)
{
    memset(w, 0, sizeof(WFILE));
    w->fd = -1;
    w->name = NULL;
    w->flags = 0;
    w->buf = zmalloc("wfile_init", w->bufsize = NLBUFSZ);
    wfile_newfile(w, name);
    return w;
}
/* wfile_newfile() - Reused a WFILE for a different file
 */

void
wfile_newfile(WFILE * w, char const * name)
{
    static char const fn[] = "wfile_newfile";
    wfile_close(w);
    zfree(fn, w->name);
    w->name = zstrdup(fn, name);
}
/* wfile_open() - Open a file for reading
 */

int
wfile_open(WFILE * w)
{
    int rc = 0;
    wfile_close(w);
    if ((w->fd = open(w->name, O_RDONLY | O_BINARY)) != -1) {

	/*
	 * Init page index
	 */
	array_init(&w->pages, sizeof(WPAGE), (unsigned) (lseek(w->fd, 0L, SEEK_END) / (NLBUFSZ - 80)));
	wfile_reset(w);
	lseek(w->fd, 0L, SEEK_SET);
	++rc;
    }
    return rc;
}
/* wfile_reset() - Restarts reading of a WFILE
 */

int
wfile_reset(WFILE * w)
{
    lseek(w->fd, w->filepos = 0L, SEEK_SET);
    w->bufcount = w->bufptr = 0;
    w->linecount = 0;
    w->pagecount = NOELEMENT;
    w->zone = w->net = w->node = 0;
    w->flags = Bf_OPEN;
    return 1;
}
/* wfile_close() - Close a WFILE
 */

void
wfile_close(WFILE * w)
{
    if (w->fd != -1) {
	close(w->fd);
	w->fd = -1;
	w->flags &= ~Bf_OPEN;
	array_destroy(&w->pages);
    }
}
/* wfile_destroy() - Destroy a WFILE object
 */

void
wfile_destroy(WFILE * w)
{
    if (w != NULL) {
	static char const fn[] = "wfile_destroy";
	wfile_close(w);
	zfree(fn, w->buf);
	zfree(fn, w->name);
    }
}
/* wfile_loadbuf() - Load next frame into WFILE buffer
 */

int
wfile_loadbuf(WFILE * w)
{
    unsigned count, avail = w->bufcount - w->bufptr;
    WPAGE *wp;
    if (avail >= NLBUFSZ)
	return avail;

    if (w->bufptr > 0) {
	wp = array_ptr(&w->pages, w->pagecount);
	wp->pagesize = w->bufptr;
	if (w->bufptr < w->bufcount)
	    w->filepos = lseek(w->fd, -(off_t) avail, SEEK_CUR);
	w->bufptr = w->bufcount = 0;
    }
    wp = array_ptr(&w->pages, w->pagecount + 1);
    count = (wp && wp->pagesize) ? wp->pagesize : NLBUFSZ;

    if ((count = read(w->fd, w->buf + w->bufcount, count)) == (unsigned) -1) {
	w->flags |= Bf_IOERR;
	return -1;
    }
    if (count == 0)
	w->flags |= Bf_EOF;
    else {
	if (++w->pagecount >= array_items(&w->pages)) {
	    array_add(&w->pages, w->pagecount, 1);
	    wp = array_ptr(&w->pages, w->pagecount);
	    wp->pageofs = w->filepos;
	    wp->pagesize = 0;
	}
	w->bufcount += count;
	w->filepos += (off_t) count;
    }

    return w->bufcount;
}
/* wfile_getmark() - Save current position as a file mark
 */

WFMARK
wfile_getmark(WFILE * w)
{
    WFMARK m;
    m.filepos = w->filepos - (off_t) (w->bufcount - w->bufptr);
    m.flags = w->flags;
    m.lineno = w->linecount;
    m.zone = w->zone;
    m.net = w->net;
    m.node = w->node;
    return m;
}


/* wfile_setmark() - Reset WFILE to a specific file mark
 */

int
wfile_setmark(WFILE * w, WFMARK * m)
{
    lseek(w->fd, w->filepos = m->filepos, SEEK_SET);
    w->flags = m->flags;
    w->linecount = m->lineno;
    w->zone = m->zone;
    w->net = m->net;
    w->node = m->node;
    w->bufcount = w->bufptr = 0;
    return 0;
}
/* wfile_analyse() - Does quick analysis of a nodelist line to keep
 *                   track of current address and node type
 */

unsigned
wfile_analyse(WFILE * w, char const * ptr, int len)
{
    unsigned newf = 0;
    char *tmp;
    if (*ptr == ';')
	newf |= (ptr[1] == 'A' ? Bf_ACMMNT : (ptr[1] == 'S' ? Bf_SCMMNT : Bf_OCMMNT));
    else if ((tmp = memchr(ptr, ',', len)) == NULL)
	newf |= Bf_UNKNWN;
    else {
	if ((w->node = (node_t) atol(++tmp)) == 0)
	    newf |= Bf_UNKNWN;
	else if (*ptr != ',') {
	    if (*ptr == 'Z' && strncmp(ptr, "Zone", 4) == 0) {
		w->zone = w->net = w->node;
		w->node = 0;
		newf |= Bf_ZC;
	    } else if ((*ptr == 'R' && strncmp(ptr, "Region", 6) == 0) ||
		       (*ptr == 'H' && strncmp(ptr, "Host", 4) == 0)) {
		w->net = w->node;
		w->node = 0;
		newf |= (*ptr == 'R' ? Bf_RC : Bf_NC);
	    } else if (*ptr == 'H' && strncmp(ptr, "Hub", 3) == 0)
		newf |= Bf_HUB;
	    else if (*ptr != 'H' /* Hold */ && *ptr != 'D' /* Down */ && *ptr != 'P' /* Pvt */ )
		newf |= Bf_UNKNWN;
	}
    }
    w->flags &= ~(Bf_ADMIN | Bf_ACMMNT | Bf_SCMMNT | Bf_OCMMNT | Bf_UNKNWN);
    w->flags |= newf;
    return newf;
}
/* wfile_getline() - Get current line, advance and analyse
 */
#if 0
/* Obsolete due to problem/bug: wfile_advance may cause the read buffer
 * to be refreshed, destroying data at 'p'
 */
char const *
wfile_getline(WFILE * w, int *lenptr)
{
    char const *p = wfile_ptr(w, lenptr);
    if (p != NULL) {
	wfile_advance(w, *lenptr);
	wfile_analyse(w, p, *lenptr);
    }
    return p;
}
#endif

/* wfile_advance() - Advance pointer to next line in buffer
 */

int
wfile_advance(WFILE * w, int len)
{
    if (w->flags & Bf_EOF || w->bufptr >= w->bufcount) {
	w->flags |= Bf_EOF;
	return 0;
    }
    w->bufptr += len;
    w->linecount++;
    return 1;
}
/* wfile_ptr() - Return a pointer to the current line
 */

char const *
wfile_ptr(WFILE * w, int *lenptr)
{
    int len;
    char const *tmp, *tmp2;
    int avail = w->bufcount - w->bufptr;
    *lenptr = 0;
    tmp2 = w->buf + w->bufptr;
    if (avail <= 0 || (tmp = memchr(tmp2, '\n', avail)) == NULL) {
	if ((avail = wfile_loadbuf(w)) <= 0)
	    return NULL;
	tmp2 = w->buf + w->bufptr;
	if ((tmp = memchr(tmp2, '\n', avail)) == NULL) {
	    unsigned at = w->bufptr + avail;
	    while (at > w->bufptr && w->buf[--at] == '\x1a')
		--w->bufcount;
	    if (at == w->bufptr)
		return NULL;
	    tmp = w->buf + at;
	}
    }
    len = ++tmp - tmp2;
    *lenptr = len;
    return *tmp2 == '\x1a' ? NULL : tmp2;
}


/* wfile_backup() - Back up one line, paging back if necessary
 */

int
wfile_backup(WFILE * w)
{
    char *p;
    if (w->bufptr == 0) {
	if (w->pagecount == 0)	/* Can't go back any further */
	    return -1;
	else {
	    WPAGE *wp = array_ptr(&w->pages, --w->pagecount);
	    lseek(w->fd, wp->pageofs, SEEK_SET);
	    w->bufptr = w->bufcount = read(w->fd, w->buf, wp->pagesize);
	    w->filepos = wp->pageofs + w->bufcount;
	    w->flags &= ~Bf_EOF;
	}
    }
    p = w->buf + --w->bufptr;
    while (p > w->buf && *--p != '\n');
    if (*p == '\n')
	++p;
    w->bufptr = p - w->buf;
    return w->linecount--;
}
