/* $Id: mem.c,v 1.1.1.1 1996/10/09 11:24:49 davidn Exp $
 * Memory allocation/debugging interface
 * These are centralised for debugging and error handling
 * zmalloc() does not return unless allocation succeeds
 * zfree() does simplistic block checking to ensure we're
 * freeing a valid block
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "mem.h"

/* mymemfatal() - default fatal error handler
 */

static void
mymemfatal(int rc, char const * fmt,...)
{
    va_list argp;
    va_start(argp, fmt);
    vfprintf(stderr, fmt, argp);
    va_end(argp);
    exit(rc);
}

int memfatalcode = 127;
fatalfunc memfatal = mymemfatal;


/* zmalloc() - Allocate a block of memory
 */

void *
zmalloc(char const * fn, size_t sz)
{
    void *p = NULL;
    if (sz) {
	unsigned *i = malloc(sz + sizeof(unsigned));
	if (i == NULL)
	    memfatal(memfatalcode, "allocation error [%s] %u bytes", fn, sz);
	*i++ = MEM_MAGIC_BLOCK;
	p = i;
    }
    return p;
}


/* zrealloc() - Reallocate a block of memory
 */

void *
zrealloc(char const * fn, void *p, size_t sz)
{
    if (p == NULL)
	p = zmalloc(fn, sz);
    else if (sz == 0) {
	zfree(fn, p);
	p = NULL;
    } else {
	unsigned *i = p;
	if (*--i != MEM_MAGIC_BLOCK)
	    memfatal(memfatalcode, "realloc error [%s] corrupted heap?", fn);
	i = realloc(i, sz + sizeof(unsigned));
	if (i == NULL)
	    p = NULL;
	else {
	    *i++ = MEM_MAGIC_BLOCK;
	    p = i;
	}
    }
    return p;
}


/* zfree() - Free a block of memory
 */

void
zfree(char const * fn, void *p)
{
    if (p) {
	unsigned *i = p;
	if (*--i != MEM_MAGIC_BLOCK)
	    memfatal(memfatalcode, "free error [%s] corrupted heap?", fn);
	free(i);
    }
}


/* zstrdup() - Duplicate a string by allocating memory
 */

char *
zstrdup(char const * fn, char const * str)
{
    if (str == NULL)
	return NULL;
    else {
	size_t l = strlen(str) + 1;
	char *p = zmalloc(fn, l);
	memcpy(p, str, l);
	return p;
    }
}
