/* $Id: compress.c,v 1.1.1.1 1996/10/09 11:19:05 davidn Exp $
 * Implements compression interface
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <fcntl.h>

#include "compress.h"
#include "parse.h"
#include "osdep.h"
#include "mem.h"
#include "cmdtpl.h"

#ifdef DOSISH
#include <io.h>
#else
#include <unistd.h>
#endif

#ifndef O_BINARY
#define O_BINARY 0
#endif

ArcInfo *
arcinfo_new(char const * file)
{
    return arcinfo_init(zmalloc("arcinfo_new", sizeof(ArcInfo)), file);
}

static int
hexvalue(char ch)
{
    int val = -1;
    if (ch && isxdigit(ch)) {
	static char const xbuffer[] = "0123456789abcdefABCDEF";
	char const *p;
	if ((p = strchr(xbuffer, ch)) != NULL && (val = p - xbuffer) > 15)
	    val -= 6;
    }
    return val;
}

ArcInfo *
arcinfo_init(ArcInfo * AI, char const * file)
{
    FILE *fp;
    memset(AI, 0, sizeof(ArcInfo));
    sb_init(&AI->strs, 512);
    array_init(&AI->defs, sizeof(ArcDef), 12);

    if ((fp = fopen(file, "r")) != NULL) {
	int lineno = 0;
	ArcDef *def = NULL;
	char tmp[512];

	static struct _verbstr cmpverbs[] =
	{
	    {"ARChiver", 3},
	    {"EXTension", 3},
	    {"ADD", 3},
	    {"EXTRact", 4},
	    {"VIEw", 3},
	    {"IDent", 2},
	    {"END", 3},
	    {"PGPSign", 4},
	    {"PGPCrypt", 4},
#if defined(__MSDOS__)
	    {"DOS", 3},
#elif defined(__OS2__) || defined(__OS2__)
	    {"OS2", 3},
#elif defined(_WIN32) || defined(WIN32)
	    {"WIN", 3},
#elif defined(__NT__) || defined(_NT)
	    {"NT", 2},
#else
	    {"UNX", 3},
#if defined(__linux__) || defined(linux)
	    {"LNX", 3},
#elif defined(__freebsd__) || defined(__FreeBSD__)
	    {"BSD", 3},
#endif
#endif
	    {NULL, 0}
	};
	while (fgets(tmp, sizeof tmp - 1, fp) != NULL) {
	    int include = 0;
	    char const *p;
	    ++lineno;
	    for (p = parseword(tmp, 0); p != NULL; p = parseword(NULL, 0)) {
		unsigned deftype = 0;
		int which = verbno(cmpverbs, p);
		switch (which) {
		case AV_PGPSIGN:
		    ++deftype;
		    /* Fallthru */
		case AV_PGPCRYPT:
		    ++deftype;
		    /* Fallthru */
		case AV_NAME:	/* Archive viewer */
		    def = (include == -1) ? NULL : arcdef_new(AI);
		    /* Fallthru */
		case AV_EXT:	/* Extension */
		case AV_ADD:	/* Add command */
		case AV_EXTRACT:	/* Extract command */
		case AV_VIEW:	/* View command */
		    if ((p = parseline(NULL)) != NULL && def && include != -1) {
			def->os[which] = arcdef_newstring(AI, p);
			def->flags = deftype;
		    }
		    break;
		case AV_END:	/* End Archiver */
		    def = NULL;
		    parseline(NULL);
		    break;
		case AV_MYOS:	/* Operating system sig */
		default:	/* Assume known OS tag */
		    include = 1;
		    continue;
		case -1:	/* Unknown operating system */
		    if (!include)
			include = -1;
		    continue;
		case AV_IDENT:	/* offset,sig */
		    if ((p = parseword(NULL, 0)) != NULL) {
			char const *data = parseline(NULL);
			if (def && include != -1) {	/* Must be in archiver
							 * section */
			    unsigned char mem[128];


			    def->sigofs = atol(p);
			    def->siglen = 0;
			    while (*data && def->siglen < sizeof mem) {
				int val1 = hexvalue(*data++);
				if (val1 == -1) {	/* Must be ascii */
				    if (def->siglen == 0)
					goto ascsig;
				    break;
				} else {
				    int val2 = hexvalue(*data++);
				    if (val2 == -1) {
					if (def->siglen == 0) {
					    --data;
				    ascsig:
					    if ((*--data) == '"') {	/* Handle quoted ascii */
						char *q = strchr(++data, '"');
						if (q == NULL)
						    --data;
						else
						    *q = '\0';
					    }
					    strncpy((char *) mem, data, sizeof mem);
					    mem[sizeof mem - 1] = '\0';
					    def->siglen = strlen((char const *) mem);
					}
					break;
				    }
				    mem[def->siglen++] = (unsigned char) (val1 * 16 + val2);
				}
			    }
			    if (def->siglen) {
				def->sig = zmalloc("arcinfo_init", def->siglen);
				memcpy(def->sig, mem, def->siglen);
			    }
			}
		    }
		    break;
		}
		include = 0;
	    }
	}
	fclose(fp);
    }
    return AI;
}

void
arcinfo_delete(ArcInfo * AI)
{
    arcinfo_destroy(AI);
    zfree("arcinfo_delete", AI);
}

void
arcinfo_destroy(ArcInfo * AI)
{
    if (AI) {
	sb_destroy(&AI->strs);
	array_destroy(&AI->defs);
    }
}

ArcDef *
arcdef_new(ArcInfo * AI)
{
    unsigned idx;
    if (AI == NULL || (idx = array_add(&AI->defs, NOELEMENT, 1)) == NOELEMENT)
	return NULL;
    return array_ptr(&AI->defs, idx);
}

ArcDef *
arcdef_find(ArcInfo * AI, char const * str)
{
    ArcDef *def = NULL;
    if (AI && str && *str) {
	ArrayIter iter;
	arrayIter_init(&iter, &AI->defs);
	while ((def = arrayIter_get(&iter, NEXT)) != NULL)
	    if (strcmp(sb_string(&AI->strs, def->os[AV_NAME]), str) == 0)
		break;
    }
    return def;
}

ArcDef *
arcdef_filetype(ArcInfo * AI, char const * file)
{
    ArcDef *def = NULL;
    if (AI && file) {
	int fd;
	if ((fd = open(file, O_RDONLY | O_BINARY)) != -1) {
	    ArrayIter iter;
	    arrayIter_init(&iter, &AI->defs);
	    while ((def = arrayIter_get(&iter, NEXT)) != NULL)
		if (arcdef_fdtype(def, fd) == 0)
		    break;
	    close(fd);
	}
    }
    return def;
}

int
arcdef_fdtype(ArcDef * def, int fd)
{
    unsigned char buf[128];
    if (def->siglen > sizeof buf)
	def->siglen = sizeof buf;
    if (def->sig == NULL ||
	def->siglen == 0 ||
    lseek(fd, def->sigofs, (def->sigofs < 0L) ? SEEK_END : SEEK_SET) == EOF ||
	read(fd, buf, def->siglen) != def->siglen)
	return -1;
    return memcmp(def->sig, buf, def->siglen);
}

int
arcdef_execute(int whichcmd, ArcInfo * AI, ArcDef * def, char const * arcname, char const * args)
{
    int rc = -1;
    char const *tplstr = arcdef_string(AI, def->os[whichcmd]);
    errno = 0;
    if (tplstr) {
	char curdir[_MAX_PATH];
	static cmdtpl arcprms[] =	/* Archiver template definitions */
	{
	    {'a', TC_STD, 0, NULL, NULL},	/* Archive name */
	    {'n', TC_NAME, 0, NULL, NULL},	/* Archive name, no path */
	    {'b', TC_BASE, 0, NULL, NULL},	/* Archive name, base only */
	    {'f', TC_WORD, 0, NULL, NULL},	/* Individual file names */
	    {'m', TC_STD, 0, NULL, NULL},	/* All filenames */
	    {'r', TC_TEMP, 0, NULL, NULL},	/* Response file name */
	    {'d', TC_STD, 0, NULL, NULL},	/* Current directory */
	    {0, TC_LAST, 0, NULL, NULL},
	};
	/* Fill in the variables */
	arcprms[0].arg = arcprms[1].arg = arcprms[1].arg = arcname;
	arcprms[2].arg = arcprms[3].arg = arcprms[4].arg = args;
	getcurdir(curdir, _MAX_PATH);
	arcprms[5].arg = curdir;
	rc = tpl_exec(tplstr, arcprms);
    }
    return rc;
}
sofs_t
arcdef_newstring(ArcInfo * AI, char const * str)
{
    return sb_alloc(&AI->strs, str);
}

char const *
arcdef_string(ArcInfo * AI, sofs_t ofs)
{
    return sb_string(&AI->strs, ofs);
}
