/* $Id: main.c,v 1.1.1.1 1996/10/09 11:26:37 davidn Exp $
 * NLMaint main module
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <ctype.h>

#include "nlmaint.h"
#include "osdep.h"
#include "filediff.h"
#include "log.h"

#include "version.h"

char const progname[] = "NLMaint";
char const progabbr[] = "NLMT";
char const version[] = VERSION;

static void usage(void);

char const *modes[] =
{
    "TEST", "PROCESS"
};

static void
usage(void)
{
    printf("usage:\n"
      "%s [-p|-t|-s] [-m=<list>] [-n=<name>] [-f] [-h|-?] [<ctlfile[.CTL]>]\n"
	   "  -p[rocess]      Force process mode (make list & submit)\n"
	   "  -t[est]         Force test mode (scan & check submittals only)\n"
	   "  -s[uppress]     Force process mode, but skip archive & submit\n"
	   "  -m[erge]=<list> Merge resulting segment with <list>\n"
	   "  -f[orce]        Force submit, even if segment has not changed\n"
	   "  -n[ame]=<name>  Override name of network to insert into list\n"
	   "  -h[elp] or -?   Print this help message\n"
#ifdef DOSISH
	   "  Use of '/' as switch character is optional, case of switches is ignored\n\n"
#endif
	   ,
	   progname);
}

int
main(int argc, char *argv[])
{
    int rc = EL_CTLERROR, carg = 0, force_process = 0, force_test = 0, no_submit = 0, force_submit = 0;
    sofs_t *so;
    ArrayIter I;
    char const *config_file = NULL;
    char const *merge_file = NULL;
    char const *net_name = NULL;
    struct tm *T;
    NLCONFIG *nlc = 0;
    long days;
    printf("%s; FidoNet Nodelist Maintenenace Tool; Ver %s\n"
	   "by " AUTHOR "\n"
	   "Compiled for " HOSTOS " on " __DATE__ " at " __TIME__ "\n\n",
	   progname, version);

    /* Set our private out-of-memory funcs */
    memfatalcode = EL_MEMERROR;
    memfatal = fatal;
    /* Set timezone */
    tzset();
    /* Eval args */
    while (++carg < argc) {
	char *argp = argv[carg];
#ifdef DOSISH
	if (*argp == '-' || *argp == '/') {    /* Command switch */
#else
	if (*argp == '-') {	/* Command switch */
#endif
	    char const *arg = strtok(++argp, "=:");
	    switch (cmdlineverb(arg)) {
	    case ARG_DEFAULT:	/* Shouldn't happen */
		break;

	    case ARG_PROCESS:	/* Force processing */
		if (force_test || no_submit)
		    goto notargs;
		++force_process;
		break;

	    case ARG_TEST:	/* Force test */
		if (force_process || no_submit)
		    goto notargs;
		++force_test;
		break;

	    case ARG_SUPPRESS:	/* Suppress submissions */
		if (!force_process && !force_test) {
		    force_process = no_submit = 1;
		    break;
		}
	notargs:
		fatal(EL_CMDERROR, "Only one of /T[est] /P[rocess] or /S[ubmit] may be specified");
		break;

	    case ARG_MERGE:	/* Merge list */
		merge_file = strtok(NULL, "");
		if (merge_file == NULL && (merge_file = argv[++carg]) == NULL)
		    fatal(EL_CMDERROR, "/M[erge] specified with no list to merge");
		break;

	    case ARG_NAME:	/* Network name */
		net_name = strtok(NULL, "");
		if (net_name == NULL && (net_name = argv[++carg]) == NULL)
		    fatal(EL_CMDERROR, "/N[ame] specified with no network name");
		break;

	    case ARG_FORCE:	/* Force submit, even if no change */
		force_submit = 1;
		break;

	    case ARG_HELP:
	    case ARG_ALTHELP:
		usage();
		fatal(0, NULL);

	    default:
		usage();
		fatal(EL_CMDERROR, "Unknown command line switch /%s", arg);
		break;
	    }
	} else if (config_file == NULL)
	    config_file = argp;
	else {
	    usage();
	    fatal(EL_CMDERROR, "Unknown word on command line \"%s\"", argp);
	}
    }

    /*
     * Use the default config file if none was specified
     */
    if (config_file == NULL)
	config_file = DFLTCFG;

    /*
     * Read the configuration file
     */

    configure(&nlc, config_file);

    /*
     * Adjust for command line overrides
     */

    if (net_name != NULL)
	nlc->str[V_NAME] = sb_alloc(&nlc->strings, net_name);
    if (merge_file != NULL)
	nlc->str[V_MERGE] = sb_alloc(&nlc->strings, merge_file);

    /*
     * Open our log file
     */

    openlog(nlstring(nlc, V_LOGFILE), nlc->logformat, nlc->loglevel);

    /*
     * Now, determine our target and which operating mode we want
     */

    nlc->now = time(NULL);
    T = localtime(&nlc->now);
    if (force_test)
	nlc->mode = MODE_TEST;
    else if (force_process)
	nlc->mode = MODE_PROCESS;
    /* Test processing mode unless on processing day of week */
    else
	nlc->mode = (T->tm_wday == nlc->process) ? MODE_PROCESS : MODE_TEST;
    if (nlc->process == V_NEVER)
	nlc->process = nlc->publish;

    /* Calculate number of days forward that next processing date falls */
    days = (T->tm_wday > nlc->process) ? (7 - (T->tm_wday - nlc->process))
	    : nlc->process - T->tm_wday;
    /* Now, calculate when the subsequent publishing date is after that */
    days += (nlc->publish < nlc->process) ? (7 - (nlc->process - nlc->publish))
	    : (nlc->publish - nlc->process);
    /* Then, let's find out the date, and therefore the julian day to process */
    nlc->publish_time = nlc->now + (days * SECSPERDAY);
    T = localtime(&nlc->publish_time);
    nlc->target = (T->tm_yday + 1);	/* tm_yday is days after 1st Jan */
    nlc->tyear = T->tm_year + 1900;
    logit(LOG_EVENT, "Running in %s mode for julian day %d", modes[nlc->mode], nlc->target);

    /* First, we search our inbound directories for likely input files */
    arrayIter_init(&I, &nlc->Inbound);
    while ((so = arrayIter_get(&I, NEXT)) != NULL)
	move_inbound(nlc, nlstr(nlc, *so));

    /*
     * Ok, that finishes with things done in both modes Now, let's see if
     * there's anything to process...
     */

    if (nlc->mode != MODE_PROCESS)
	rc = test_segment(nlc) ? EL_TESTERRORS : EL_TESTOK;
    else {
	int dodiff;
	char targ[_MAX_PATH] = "",	/* Target nodelist */
	    mlst[_MAX_PATH] = "",	/* List to merge */
	    fdif[_MAX_PATH] = "",	/* Difference file */
	    ddel[_MAX_PATH] = "";	/* Diff to delete */
	move_updates(nlc);	/* Move list updates, apply difference files */

	if (!nlsnull(nlc, V_BEFORE))
	    exec_cmd(nlc, nlstring(nlc, V_BEFORE), NULL, NULL, NULL, NULL);

	if ((dodiff = make_segment(nlc, targ)) != -1) {
	    char dest[_MAX_PATH];
	    char *submit = targ;/* File to submit */
	    strcpy(mlst, targ);	/* Save this for merge */
	    logit(LOG_EVENT, "New nodelist \"%s\" made successfully", targ);
	    if (dodiff) {
		int d = make_diff(nlc, targ, fdif, force_submit);
		if (d == 0) {
		    logit(LOG_EVENT, "Difference file \"%s\" made successfully", fdif);
		    submit = fdif;	/* Submit the diff file */
		} else if (d == 1) {
		    logit(LOG_EVENT, "Not submitting unchanged file");
		    if (!no_submit)	/* We might be testing here */
			remove(targ);	/* Otherwise we can junk this file */
		    no_submit = 1;	/* Force no submittal either way */
		} else
		    dodiff = 0;	/* if (d == -1) */
	    }
	    if (no_submit)
		logit(LOG_EVENT, "Skipping archive & submit");
	    else {
		int nodearc = 0;
		int diffarc = 0;
		static char const filesbbs[] = "files.bbs";
		if (!nlsnull(nlc, V_ARCLIST) && !nlsnull(nlc, V_ARC)) {
		    if (archive_list(nlc, targ, nlstring(nlc, V_ARCLIST), nlstring(nlc, V_ARC), nlstring(nlc, V_NLPFX), 0) == 0)
			++nodearc;
		    else
			logit(LOG_WARN, "Archiver failed - using raw list instead");

		    if (!nodearc && copylist(targ, nlstring(nlc, V_ARCLIST)))
			++nodearc;	/* Now means succeded, to update
					 * FILES.BBS */
		}
		if (nlsnull(nlc, V_DIFFARC))
		    dodiff = 0;
		else if (dodiff) {	/* Only do this if we have generated a
					 * diff */
		    strcpy(ddel, fdif);
		    if (archive_list(nlc, fdif, nlstring(nlc, V_ARCDIFF), nlstring(nlc, V_DIFFARC), nlstring(nlc, V_NDPFX), 0) == 0)
			++diffarc;
		    else
			logit(LOG_WARN, "Archiver failed - using raw diff instead");

		    if (!diffarc && copylist(fdif, nlstring(nlc, V_ARCDIFF)))
			++diffarc;	/* Now means succeded, to update
					 * FILES.BBS */
		}
		if (nodearc && !nlsnull(nlc, V_LISTDESC))
		    update_description(nlc, build_path(dest, get_path(NULL, targ), filesbbs, NULL), nlstring(nlc, V_LISTDESC), targ, fdif);

		if (diffarc && !nlsnull(nlc, V_DIFFDESC))
		    update_description(nlc, build_path(dest, get_path(NULL, fdif), filesbbs, NULL), nlstring(nlc, V_DIFFDESC), targ, fdif);

		if (nlsnull(nlc, V_EXECSUBMIT)) {
		    if (nlc->submitaddr.net)
			logit(LOG_MARK, "No EXECSUBMIT command, can't submit \"%s\" to %s", submit, fidofmt(NULL, &nlc->submitaddr, NULL));
		} else
		    exec_cmd(nlc, nlstring(nlc, V_EXECSUBMIT), targ, fdif, submit, NULL);

		if (!nlsnull(nlc, V_AFTER))
		    exec_cmd(nlc, nlstring(nlc, V_AFTER), targ, fdif, submit, NULL);

		/* It is safe to delete the diff file now */
		if (*ddel)
		    remove(ddel);

		if (!nlsnull(nlc, V_MERGE)) {
		    if (!find_latest(targ, nlstring(nlc, V_MERGE), F_LIST, nlc->publish_time))
			logit(LOG_WARN, "Can't find any nodelist \"%s\" for merge", nlstring(nlc, V_MERGE));
		    else {
			strcpy(fdif, targ);
			add_extn(fdif, 999);	/* Use <listname>.999 as target */
			logit(LOG_PROGRESS, "Merging \"%s\" with \"%s\" to produce \"%s\"", last_component(targ), last_component(mlst), fdif);
			if (merge_list(fdif, targ, mlst, nlc->myaddr) == 0)
			    logit(LOG_RESULT, "Merge successful. Output file is \"%s\"", fdif);
			else
			    logit(LOG_ERROR, "Merge failed");
		    }
		}
		/*
		 * Finally, let's clean up everything and keep only the latest
		 * copy of input and output files in the MASTER and OUTPUT dirs
		 */

		if (nlc->cleanup) {
		    char const *file;
		    subfile *sf;
		    ArrayIter iter;
		    /* First, segments in the MASTER directory */
		    arrayIter_init(&iter, &nlc->Files);
		    while ((sf = arrayIter_get(&iter, NEXT)) != NULL) {
			file = nlstr(nlc, sf->name);
			build_path(dest, nlstring(nlc, V_MASTER), file, NULL);
			if (find_latest(dest, dest, F_LIST, nlc->publish_time) > 1)
			    cleanup_lists(nlstring(nlc, V_MASTER), file, dest, nlc->publish_time);
		    }

		    /* Then output lists */
		    file = nlstring(nlc, V_OUTFILE);
		    build_path(dest, nlstring(nlc, V_OUTPATH), file, NULL);
		    if (find_latest(dest, dest, F_LIST, nlc->publish_time) > 1)
			cleanup_lists(nlstring(nlc, V_OUTPATH), file, dest, nlc->publish_time);
		}
	    }
	}
	rc = (nlc->errcount) ? EL_PROCESSERRORS : EL_PROCESSOK;
    }

    send_feedback(nlc, no_submit);

    logit(LOG_MARK, "NLMAINT exiting with status %d", rc);
    openlog(NULL, -1, 0);
    return rc;
}
