/* $Id: wfile.h,v 1.1.1.1 1996/10/09 11:18:06 davidn Exp $
 * Buffered nodelist reading
 *
 */

#ifndef _wfile_h
#define _wfile_h

#include <sys/types.h>		/* off_t */
#include "array.h"		/* Array */
#include "fidoaddr.h"		/* node_t */

/* WPAGE: This structure contains a page offset and number of characters
 * that contains a set of complete lines. These are used for seeking
 * around the file when syncing.
 */

typedef struct _wpage {
    off_t pageofs;		/* Offset into file */
    unsigned pagesize;		/* Bytes in this page */
}   WPAGE;


/* WFILE: Super-fast buffered line reader for nodelist files
 * This buffering method is intended to reduce data copying during
 * the read/parsing process. It's "getline" function returns a
 * pointer to the current line (and its length) within the file
 * buffer itself rather than unnecessarily copying it about.
 * Lines are tracked during the read, as is the address context.
 */

typedef struct _wfile {
    int fd;			/* File descriptor */
    unsigned bufsize,		/* Current size of buffer */
        bufcount,		/* How many chrs in buffer */
        bufptr;			/* Current position in buffer */
#define   Bf_OPEN   0x0001	/* File is open */
#define   Bf_EOF    0x0002	/* End of file hit */
#define   Bf_IOERR  0x0004	/* File I/O error */
#define   Bf_UNKNWN 0x0008	/* Input not known */
#define   Bf_ACMMNT 0x0010	/* On A comment */
#define   Bf_SCMMNT 0x0020	/* On S comment */
#define   Bf_OCMMNT 0x0040	/* On other comment */
#define   Bf_CMMNT  0x00f0	/* Any comment */
#define   Bf_ZC     0x0100	/* ZC system */
#define   Bf_RC     0x0200	/* RC system */
#define   Bf_NC     0x0400	/* NC system */
#define   Bf_COORD  0x0700	/* Coordinator system */
#define   Bf_HUB    0x0800	/* Hub system */
#define   Bf_ADMIN  0x0f00	/* On admin/0 node */
    unsigned flags,		/* Buffer flags */
        pagecount,		/* Page index */
        linecount;		/* Line counter */
    off_t filepos;		/* File position at top of buffer */
    node_t zone,		/* Current zone, region net and node */
        net, node;
    char *buf;			/* Buffer */
    char *name;			/* Name of file */
    Array pages;		/* Array of pages */
}   WFILE;
/* File bookmark */

typedef struct _wfmark {
    off_t filepos;		/* Saved position */
    unsigned flags,		/* File flags */
        lineno;			/* Current line number */
    node_t zone,		/* Context */
        net, node;
}   WFMARK;

WFILE *wfile_new(char const * name);
void wfile_delete(WFILE * w);
WFILE *wfile_init(WFILE * w, char const * name);
void wfile_destroy(WFILE * w);
int wfile_open(WFILE * w);
void wfile_newfile(WFILE * w, char const * name);
int wfile_reset(WFILE * w);
void wfile_close(WFILE * w);
int wfile_loadbuf(WFILE * w);
WFMARK wfile_getmark(WFILE * w);
int wfile_setmark(WFILE * w, WFMARK * m);
/* Get current line & advance pointer */
/*  char const * wfile_getline(WFILE * w, int * lenptr); */
/* Get current line, do not advance pointer */
char const *wfile_ptr(WFILE * w, int *lenptr);
/* Advance pointer */
int wfile_advance(WFILE * w, int len);
/* Backup pointer */
int wfile_backup(WFILE * w);
/* Analysis of line data */
unsigned wfile_analyse(WFILE * w, char const * ptr, int len);


#define wfile_eof(w)        ((w)->flags & (Bf_EOF|Bf_IOERR))
#define wfile_isadmin(w)    (!!((w)->flags & Bf_ADMIN))
#define wfile_iscoord(w)    (!!((w)->flags & Bf_COORD))
#define wfile_iszc(w)       (!!((w)->flags & Bf_ZC))
#define wfile_isrc(w)       (!!((w)->flags & Bf_RC))
#define wfile_isnc(w)       (!!((w)->flags & Bf_NC))
#define wfile_ishub(w)      (!!((w)->flags & Bf_HUB))
#define wfile_iszrc(w)      (!!((w)->flags & (Bf_RC|Bf_ZC)))
#define wfile_isvalid(w)    (!((w)->flags & Bf_UNKNWN))
#define wfile_iscomment(w)  (!!((w)->flags & (Bf_CMMNT)))
#define wfile_isacomment(w) (!!((w)->flags & (Bf_ACMMNT|Bf_OCMMNT)))
#define wfile_zone(w)       ((w)->zone)
#define wfile_net(w)        ((w)->net)
#define wfile_node(w)       ((w)->node)
#define wfile_atline(w)     ((w)->linecount)

#endif
