/* $Id: nldiff.c,v 1.1.1.1 1996/10/09 11:26:43 davidn Exp $
 * A simple program to produce a difference file between two nodelists
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "nlmaint.h"
#include "filediff.h"
#include "aplydiff.h"
#include "posix.h"
#include "log.h"

#include "version.h"

char const progname[] = "NLDiff";
char const progabbr[] = "NLDF";
char const version[] = VERSION;


typedef struct _nldiff {
    short dow;			/* Publishing day of week */
    short julian;		/* Julian day for next publication date */
    time_t publication;		/* Next date of publication */
    short logformat;		/* Format of log file */
    char *orig;			/* Original nodelist name */
    char *targ;			/* Destination nodelist name */
    char *diff;			/* Nodediff name */
    char *logf;			/* Logfile name */
}   NLDCTL;

int
main(int argc, char *argv[])
{
    int rc = 1, fnum = 0, carg = 0, days;
    time_t now;
    struct tm *T;
    NLDCTL nld;
    char tmp[_MAX_PATH];
    memset(&nld, 0, sizeof nld);
    nld.dow = V_FRIDAY;
    nld.logf = NULL;

    printf("%s; FidoNet Nodediff Extractor; Ver %s\n"
	   "by " AUTHOR "\n"
	   "Compiled for " HOSTOS " on " __DATE__ " at " __TIME__ "\n\n",
	   progname, version);

    tzset();
    while (++carg < argc) {
	char *argp = argv[carg];
#ifdef DOSISH
	if (*argp == '-' || *argp == '/')	/* command switch */
#else
	if (*argp == '-')	/* command switch */
#endif
	{
	    char ch = *++argp;
#ifdef DOSISH
	    ch = (char) toupper(ch);
#endif
	    switch (ch) {
	    case 'L':		/* Log file name */
		if (!*++argp)
		    argp = argv[++carg];
		nld.logf = argp;
		break;

	    case 'F':		/* Format of log file */
		if (!*++argp)
		    argp = argv[++carg];
		if (argp)
		    nld.logformat = (short) atoi(argp);
		break;

	    case 'D':		/* Specify a publishing day */
		if (!*++argp)
		    argp = argv[++carg];
		if (argp) {
		    static char const dows[] = "SunMonTueWedThuFriSat";
		    if (isdigit(*argp))
			nld.dow = (short) atoi(argp);
		    else
			for (nld.dow = 0; nld.dow < 7 && strnicmp(dows + (nld.dow * 3), argp, 3) != 0; ++nld.dow);
		    if (nld.dow < 0 || nld.dow >= 7)
			fatal(EL_CMDERROR, "Invalid day of week %s", argp);
		}
		break;

	    case 'O':		/* Specify output diff filename */
		if (!*++argp)
		    argp = argv[++carg];
		nld.diff = argp;
		break;

	    case 'H':
	    case '?':		/* Help! */
		printf("usage: %s [options] orig_nodelist targ_nodelist\n"
		       "Unldss provided nodelist_name=\"nodelist\" & nodediff_name=\"nodediff\"\n"
		       "Available options:\n"
		       "  -o <file> Specify root name of nodediff file to extract (default=\"nodediff\")\n"
		       "  -d <day>  Specify day of week nodelist is published (default=Friday)\n"
		       "            Provide day name or number, where 0=Sunday, 1=Monday etc.\n"
		   "  -l <file> Set the name of the log file (default=none)\n"
		       "  -f #      Set log file format; 0=default Bink/Opus, 1=FrontDoor\n",
		       progname);
		return 0;

	    default:
		fatal(EL_CMDERROR, "Unknown command line switch %s (try -?)", --argp);
	    }
	} else
	    switch (fnum++) {
	    case 0:		/* Nodelist name */
		nld.orig = argp;
		break;
	    case 1:
		nld.targ = argp;
		break;
	    default:
		fatal(EL_CMDERROR, "unexpected word on command line '%s'", argp);
	    }
    }

    /* Some sanity checking */
    if (nld.orig == NULL)
	fatal(EL_CMDERROR, "Original nodelist filename is required");
    if (nld.targ == NULL)
	fatal(EL_CMDERROR, "Target nodelist filename is required");

    openlog(nld.logf, nld.logformat, -1);

    now = time(NULL);		/* Get current data to determine targets */
    T = localtime(&now);
    /* Calculate number of days forward that next processing date falls */
    days = (T->tm_wday > nld.dow) ? (7 - (T->tm_wday - nld.dow))
	    : nld.dow - T->tm_wday;
    /*
     * Then, let's find out the date, and therefore the julian day of
     * publication
     */
    nld.publication = now + (days * SECSPERDAY);
    nld.julian = nldaynumber(nld.publication);

    /* This is where the application really starts :-) */

    if ((rc = filediff(diff_name(tmp, nld.targ, nld.diff), nld.orig, nld.targ)) != 0)
	remove(tmp);

    /* .. and subsequently ends */

    openlog(NULL, -1, 0);
    return rc;
}
