/* $Id: txtbuf.c,v 1.1.1.1 1996/10/09 11:25:34 davidn Exp $
 * Implements a generic text buffer with 'gap' movement for
 * optimised insertions and deletions.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include "txtbuf.h"
#include "mem.h"

#define TBBLKSZ 512		/* Allocation unit size */

/* Common private memory allocator
 */

static int
tb_allocate(txtbuf * tb, bofs_t sz)
{
    sz += (TBBLKSZ - 1);	/* Round up to nearest block size multiple */
    sz -= (sz % TBBLKSZ);
    if (sz > tb->bufSize) {	/* Only do this if necessary */
	bufptr_t newbuf = zrealloc("tb_allocate", tb->buffer, sz);
	if (newbuf) {
	    bofs_t larger = sz - tb->bufSize;	/* How much bigger we are */
	    bofs_t tomove = tb->bufSize - (tb->gapOfs + tb->gapSize);
	    tb->buffer = newbuf;
	    if (tomove) {
		newbuf += (tb->gapOfs + tb->gapSize);	/* Old buffer top */
		memmove(newbuf, newbuf + larger, tomove);	/* Move data upwards */
	    }
	    tb->gapSize += larger;	/* And make the gap larger */
	    tb->bufSize = sz;
	    return 1;
	}
	return 0;		/* Nothing allocated */
    }
    return 1;
}


txtbuf *
tb_new(bofs_t sz)
{
    return tb_init(zmalloc("tb_new", sizeof(txtbuf)), sz);
}

void
tb_delete(txtbuf * tb)
{
    tb_destroy(tb);
    zfree("tb_delete", tb);
}
/* Init a text buffer */

txtbuf *
tb_init(txtbuf * tb, bofs_t sz)
{
    tb->bufSize = tb->gapOfs = tb->gapSize = 0;
    tb->buffer = 0;
    tb_allocate(tb, sz);
    return tb;
}
/* Destroy a text buffer */

void
tb_destroy(txtbuf * tb)
{
    if (tb) {
	zfree("tb_destroy", tb->buffer);
	tb->bufSize = tb->gapOfs = tb->gapSize = 0;
	tb->buffer = 0;
    }
}
/* Remove all text */

void
tb_reset(txtbuf * tb)
{
    tb->gapOfs = 0;
    tb->gapSize = tb->bufSize;
}


/* Move gap offset */

bofs_t
tb_seek(txtbuf * tb, bofs_t pos, int whence)
{
    bofs_t cursize = tb_size(tb);
    if (whence == SEEK_CUR)
	pos = tb->gapOfs + pos;
    else if (whence == SEEK_END)
	pos = cursize + pos;
    else if (whence == SEEK_SET)
	pos = pos;
    else
	return (bofs_t) - 1;	/* Error */

    if (pos > cursize)		/* Auto correct */
	pos = cursize;

    if (pos != tb->gapOfs) {
	if (tb->gapSize) {
	    /*
	     * Gap is moving up |    |-----|       |  Current |        |---->|
	     * |  New
	     */
	    if (pos > tb->gapOfs)
		memmove(tb->buffer + tb->gapOfs, tb->buffer + tb->gapOfs + tb->gapSize, pos - tb->gapOfs);
	    /*
	     * Gap is moving down |        |-----|   |  Current |    |<----|
	     * |  New
	     */
	    else
		memmove(tb->buffer + pos + tb->gapSize, tb->buffer + pos, tb->gapOfs - pos);
	}
	tb->gapOfs = pos;	/* We have moved it! */
    }
    return pos;
}
/* This is THE insertion function */

int
tb_write(txtbuf * tb, void const * data, int count)
{
    if ((bofs_t) count > tb->gapSize && !tb_allocate(tb, tb->bufSize + count))
	count = 0;		/* We must accept everything for this to
				 * succeed */
    else if (count) {
	memmove(tb->buffer + tb->gapOfs, data, count);
	tb->gapOfs += count;
	tb->gapSize -= count;
    }
    return count;
}
/* Write string */

int
tb_puts(txtbuf * tb, char const * str)
{
    return tb_write(tb, str, strlen(str));
}
/* Write character */

int
tb_putc(txtbuf * tb, char ch)
{
    return tb_write(tb, &ch, 1);
}
/* Formatted output */
int
tb_printf(txtbuf * tb, char const * fmt,...)
{
    int rc;
    va_list argp;
    char buf[1024];		/* Be conservative with stack */
    va_start(argp, fmt);
    rc = vsprintf(buf, fmt, argp);
    va_end(argp);
    return tb_write(tb, buf, rc);
}
/* Remove/delete text */

int
tb_remove(txtbuf * tb, int count)
{				/* Get size of data above gap */
    bofs_t top = tb->bufSize - (tb->gapOfs + tb->gapSize);
    if (count > (int) top)
	count = top;
    if (count <= 0)
	count = 0;
    else
	tb->gapSize += count;
    return count;
}
/* Read data */

int
tb_read(txtbuf * tb, char *buf, int count)
{
    bofs_t ofs = tb->gapOfs + tb->gapSize;
    bofs_t top = tb->bufSize - ofs;
    if (count > (int) top)
	count = top;
    if (count <= 0)
	count = 0;
    else {			/* Move data at top of buffer to user's buffer */
	memmove(buf, tb->buffer + ofs, count);
	/* And advance bottom of buffer */
	memmove(tb->buffer + tb->gapOfs, tb->buffer + ofs, count);
	tb->gapOfs += count;
    }
    return count;
}
/* Get string */

char *
tb_gets(txtbuf * tb, char *buf, int max)
{
    bofs_t ofs = tb->gapOfs + tb->gapSize;
    bofs_t top = tb->bufSize - ofs;
    if (top == 0)
	buf = NULL;
    else {
	bufptr_t at = tb->buffer + ofs;
	bufptr_t ptr = memchr(at, '\n', top);	/* Look for newline */
	if (ptr == NULL)
	    ptr = tb->buffer + tb->bufSize;
	else
	    ++ptr;		/* Include the newline */
	top = (bofs_t) (ptr - at);
	if (top > (bofs_t) max)	/* Correct count */
	    top = max;
	if (top == 0)
	    buf = NULL;
	else {			/* Move data at top of buffer to user's buffer */
	    memmove(buf, at, top);
	    /* And advance bottom of buffer */
	    memmove(tb->buffer + tb->gapOfs, at, top);
	    tb->gapOfs += top;
	    buf[top] = '\0';	/* NUL terminate */
	}
    }
    return buf;
}
