/* $Id: nodelist.c,v 1.4 1996/10/31 19:59:32 davidn Exp $
 * Nodelist related functions
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "nlmaint.h"
#include "compress.h"
#include "aplydiff.h"
#include "posix.h"
#include "parse.h"
#include "log.h"
#include "wfile.h"
#include "crc16.h"
#include "checknl.h"
#include "cfile.h"
#include "filediff.h"


enum {				/* --- Exit codes for check_segment --- */
    NLX_OK = 0,			/* Segment is fine */
    NLX_ERRFILE = 1,		/* File access problem */
    NLX_ERRFMT = 2,		/* Format problem in 1 or more lines */
    NLX_FATFMT = 3,		/* Serious! format problem */
    NLX_ERRCRC = 4,		/* CRC's do not agree */
    NLX_UNKNOWN = 5		/* Unknown lines */
};

enum {				/* -- Bitmasks for which files to process -- */
    mv_ARC = 0x0001,		/* Archived & PGP signed/encoded files */
    mv_LIST = 0x0002,		/* Nodelists */
    mv_DIFF = 0x0004,		/* Diff files */
    mv_BAD = 0x0008,		/* Anything unrecognisable */
    mv_ALL = 0x000f,
    mv_APPLY = 0x0010,		/* Apply difference files */
    mv_CHECK = 0x0020,		/* Check nodelists */
    mv_NOMOVE = 0x0040,		/* Don't actually copy/move (for diffs) */
    mv_LVL = 0x0080,		/* Set for non first-level processing */
    mv_NONOTIF = 0x0100		/* Don't notify of receipt */
};

static int bad_file(NLCONFIG * nlc, char const * path);
static int process_update(NLCONFIG * nlc, char *path, char const * destdir, subfile * sf, unsigned types);
static int move_set(NLCONFIG * nlc, char const * destdir, char const * dir, char const * pattern, subfile * sf, unsigned types);
static void comment_include(CFILE * outfp, char const * filename, int year);
static int output_entry(CFILE * outfp, txtbuf * errs, char const * p, int len, ZNNP * ataddr, char *txtaddr);
static void write_hdr(NLCONFIG * nlc, CFILE * outfp, unsigned short crc16);
static void *memstr(void const * n, size_t sn, void const * h, size_t sh);
static void init_process(NLCONFIG * nlc);


static int
bad_file(NLCONFIG * nlc, char const * path)
{
    char const *badpath = nlstring(nlc, V_BADFILES);
    char badfilename[_MAX_PATH];
    build_path(badfilename, badpath, last_component(path), NULL);
    logit(LOG_EVENT, "BAD file %s => %s", path, badpath);
    nlc->errcount++;
    return fmove(badfilename, path);
}

static int
process_update(NLCONFIG * nlc, char *path, char const * destdir, subfile * sf, unsigned types)
{
    int rc = -1;
    short applies_to;
    unsigned short crc16;
    int pt = nlfiletype(path, &applies_to, &crc16, nlstr(nlc, sf->password));
    int fnok = stricmp(last_component(path), nlstr(nlc, sf->name)) == 0;
    char const *file;
    char const *msg = "file type unknown";

    static char const *ft[] = {"unknown", "file", "segment", "diff"};
    if (pt == F_LIST || pt == F_DIFF) {
	if ((pt == F_LIST && !(types & mv_LIST)) || (pt == F_DIFF && !(types & mv_DIFF)))
	    return 0;

	if (fnok) {
	    if (pt == F_LIST)
		rc = 0;
	    else
		msg = "requires a generic extension";
	} else {
	    char *ext = path_ext(path);
	    msg = "has no or an invalid extension";
	    if (*ext && isdigit(ext[2]) && isdigit(ext[3])) {
		if (pt == F_DIFF && ext[1] == 'D')
		    rc = 0;
		else if (isdigit(ext[1])) {
		    int day = atoi(++ext);
		    int age = weeksold(nlc->publish_time, day);
		    if (age < 0 || age > 26)	/* ~6 months late check */
			msg = "does not appear to be current";
		    else
			rc = 0;
		}
	    }
	}
	if (rc == 0) {
	    if ((pt == F_LIST) && (types & mv_CHECK)) {
		rc = check_segment(path);
		if (rc == NLX_ERRFMT)	/* This is not too serious to prevent
					 * processing */
		    rc = 0;	/* Process list, clean up as well */
	    }
	}
    } else {
	if (!(types & mv_BAD))
	    return 0;		/* Ignore it */
	if (pt & F_PWDERR) {
	    msg = "contains no or an invalid password";
	    pt &= ~F_PWDERR;
	}
    }

    file = last_component(path);
    if (rc == 0) {		/* Move file to destination directory */
	char dest[_MAX_PATH];
	if (types & mv_NOMOVE)
	    strcpy(dest, path);
	else {
	    if ((rc = fmove(build_path(dest, destdir, file, NULL), path)) != 0)
		logit(LOG_ERROR, "Error moving %s segment \"%s\" to \"%s\": %s (%d)", ft[pt + 1], file, destdir, strerror(errno), errno);
	    else {
		logit(LOG_EVENT, "Moved \"%s\" to \"%s\"", path, destdir);
		if (!(types & mv_NONOTIF))
		    fb_printf(nlc, sf, NM_LOG | NM_RECEIPT | NM_FEEDBACK, "Nodelist segment \"%s\" received without error.\n", file);
	    }
	}
	/* Add diffs for processing, if requested */
	if (rc == 0 && (pt == F_DIFF) && (types & mv_APPLY) && sf->pda)
	    diffapp_nldiff(sf->pda, dest);
    } else {
	logit(LOG_ERROR, "%s \"%s\" %s", ft[pt + 1], file, msg);
	if ((pt == F_LIST || pt == F_DIFF) && !(types & mv_NONOTIF)) {	/* Ignore stuff that we
									 * don't know about */
	    fb_printf(nlc, sf, NM_ERROR | NM_LOG | NM_RECEIPT | NM_FEEDBACK, "Problem accepting segment \"%s\"\n -- %s %s\n", file, ft[pt + 1], msg);
	}
    }

    return rc;
}


static int
move_set(NLCONFIG * nlc, char const * destdir, char const * dir, char const * pattern, subfile * sf, unsigned types)
{
    int rc = 0;
    FFIND *ff;
    if ((ff = file_findfirst(dir, pattern)) != NULL) {
	do {
	    if (file_isreg(ff)) {	/* Ignore subdirs etc. */
		char path[_MAX_PATH];
		ArcDef *def = arcdef_filetype(&nlc->Arcs, file_path(ff, path));
		/*
		 * Determine the file type If we recognise it, queue it for
		 * processing
		 */

		if (!(types & mv_LVL) &&
		    (sf->notify & (NF_PGPSIGN | NF_PGPCRYPT)) &&
		    (!def || !(def->flags & (PGPSIGN | PGPCRYPT)))) {
		    logit(LOG_ERROR, "Missing required PGP signature \"%s\" - processing skipped", path);
		    rc = -1;
		} else if (def) {	/* Unpack into a subdirectory */
		    char curdir[_MAX_PATH], subdir[_MAX_PATH];
		    getcurdir(curdir, _MAX_PATH);
		    build_path(subdir, destdir, pattern, NULL);
		    *path_ext(subdir) = '\0';	/* Shave off extension for
						 * non-generic names */
		    if (makesubdir(subdir) != 0)
			logit(LOG_WARN, "Can't create directory \"%s\" - processing skipped", subdir);
		    else {
			if (cleandir(subdir) != 0)
			    logit(LOG_WARN, "Problem cleaning temporary dir \"%s\"", subdir);

			if (changedir(subdir) != 0)
			    logit(LOG_ERROR, "Can't change to directory \"%s\" - processing skipped", subdir);
			else {
			    if ((rc = arcdef_execute(AV_EXTRACT, &nlc->Arcs, def, path, NULL)) == 0)
				rc = move_set(nlc, destdir, subdir, pattern, sf, types | mv_LVL);
			}
		    }
		    changedir(curdir);
		    cleandir(subdir);
		    removedir(subdir);
		}
		 /* Unknown file type - raw diff or list? */ 
		else
		    rc = process_update(nlc, path, destdir, sf, types);

		if (rc == 0)
		    remove(path);
		else
		    bad_file(nlc, path);
	    }
	}
	while (file_findnext(ff));
	file_findclose(ff);
    }
    return rc;
}
/* move_inbound
 * Scan a directory for incoming nodelist segments
 * Examine the file's packaging and enforce restrictions, encryption & signatures
 */

void
move_inbound(NLCONFIG * nlc, char const * dir)
{
    if (dir && *dir && array_items(&nlc->Files)) {
	ArrayIter iter;
	subfile *sf;
	arrayIter_init(&iter, &nlc->Files);
	while ((sf = arrayIter_get(&iter, NEXT)) != NULL) {
	    char *ext, tmp[_MAX_PATH];
	    strncpy(tmp, nlstr(nlc, sf->name), sizeof tmp - 4);
	    tmp[sizeof tmp - 5] = '\0';
	    if (!*(ext = path_ext(tmp)))
		strcpy(ext, ".??#");
	    /*
	     * Unpack files, move received lists and diffs into the update
	     * directory
	     */
	    move_set(nlc, nlstring(nlc, V_UPDATE), dir, tmp, sf, mv_ALL);
	}
    }
}
/* move_updates
 * Move updates from the update directory to the master directory
 * This checks the CRC on nodelists, and attempts to apply any
 * difference file(s) found in the process
 */

void
move_updates(NLCONFIG * nlc)
{
    if (array_items(&nlc->Files)) {
	ArrayIter iter;
	subfile *sf;
	arrayIter_init(&iter, &nlc->Files);
	while ((sf = arrayIter_get(&iter, NEXT)) != NULL) {
	    char const *segname = nlstr(nlc, sf->name);
	    char *ext, tmp[_MAX_PATH];
	    strncpy(tmp, segname, sizeof tmp - 4);
	    tmp[sizeof tmp - 5] = '\0';
	    ext = path_ext(tmp);

	    /*
	     * Handling for generic lists
	     */

	    if (!*ext) {
		int fnum;
		DIFFAPP da;
		char latest[_MAX_PATH];
		/* Init the diff-apply here, in case we pick up any diffs */
		sf->pda = diffapp_init(&da, nlc->publish_time);

		/*
		 * First, let's just move any and all nodelists into the master
		 * directory where we can find them if we encounter any diffs
		 */
		strcpy(ext, ".###");
		move_set(nlc, nlstring(nlc, V_MASTER), nlstring(nlc, V_UPDATE), tmp, sf, mv_LIST | mv_CHECK | mv_NONOTIF);
		/*
		 * Then we look for and process difference files First,
		 * however, we must select the most recent nodelist...
		 */
		build_path(latest, nlstring(nlc, V_MASTER), tmp, NULL);
		if (find_latest(latest, latest, F_LIST, nlc->publish_time) == 0)
		    logit(LOG_WARN, "No segment \"%s\" available in MASTER dir", segname);
		else
		    diffapp_nlfile(&da, latest);

		strcpy(ext, ".?##");
		move_set(nlc, nlstring(nlc, V_MASTER), nlstring(nlc, V_UPDATE), tmp, sf, mv_DIFF | mv_APPLY | mv_NONOTIF);
		/* Gather diffs in MASTER directory too */
		move_set(nlc, nlstring(nlc, V_MASTER), nlstring(nlc, V_UPDATE), tmp, sf, mv_DIFF | mv_APPLY | mv_NOMOVE | mv_NONOTIF);

		if ((fnum = diffapp_settarget(&da)) == 0) {
		    char const *next = path_ext(latest);
		    /* Only issue this msg if segment is not up to date */
		    if (!*next || atoi(next + 1) != nlc->target)
			logit(LOG_PROGRESS, "No difference file(s) to apply for \"%s\"", segname);
		}
		/* Let's process them! */
		else if (diffapp_apply(&da) == -1)
		    logit(LOG_ERROR, "Difference file application errror for \"%s\" - no cleanup done", segname);
		else {
		    char const *src = sb_string(&da.sbuf, da.orig.file);
		    DFFILE *fdif = NULL;
		    ArrayIter I;
		    /*
		     * Let's clean up diffs to keep things tidy
		     */
		    arrayIter_init(&I, &da.diffs);
		    do {
			if (remove(src) == 0)
			    logit(LOG_PROGRESS, "Deleted \"%s\"", src);
			else
			    logit(LOG_ERROR, "Error deleting \"%s\": %s (%d)", src, strerror(errno), errno);
			if ((fdif = arrayIter_get(&I, NEXT)) != NULL)
			    src = sb_string(&da.sbuf, fdif->file);
		    } while (fdif);

		    logit(LOG_RESULT, "Nodelist segment \"%s\" updated", segname);
		}
		/* Dump the nodediff app data */
		diffapp_destroy(&da);
	    }
	     /* Non-generic updates */ 
	    else
		move_set(nlc, nlstring(nlc, V_MASTER), nlstring(nlc, V_UPDATE), tmp, sf, mv_LIST | mv_CHECK);
	}
    }
}

int
check_segment(char const * listname)
{
    int rc = NLX_OK;
    WFILE wfp;
    wfile_init(&wfp, listname);
    if (!wfile_open(&wfp)) {
	logit(LOG_ERROR, "Can't open segment %s: %s", listname, strerror(errno));
	rc = NLX_ERRFILE;
    } else {
	int len;
	char const *p;
	clock_t taken, start = clock();
	unsigned short crcval = 0;
	unsigned short listcrc = (unsigned short) -1;

	enum {
	    LINES, COMMENTS, NODES, COORDS, ADMINS, UNKNOWN, MAXSTAT
	};

	unsigned stt[MAXSTAT];
	for (len = LINES; len < MAXSTAT; len++)
	    stt[len] = 0;

	logit(LOG_PROGRESS, "Checking nodelist segment: %s", listname);
	while ((p = wfile_ptr(&wfp, &len)) != NULL) {
	    char const *q;
	    wfile_analyse(&wfp, p, len);

	    if (stt[LINES]++ > 0) {	/* Skip first line in CRC calculation */
		q = p + len - 1;
		while (q > p && (*q == '\r' || *q == '\n'))
		    --q;
		crcval = update_crc16((unsigned char const *) p, q - p + 1, crcval);
		crcval = update_crc16((unsigned char const *) "\r\n", 2, crcval);
	    } else {		/* Grab CRC-16 from first line */
		q = memchr(p, ':', len);
		if (q == NULL || (q += 2, !isdigit(*q))) {
		    logit(LOG_FATAL, "Invalid file format (no valid CRC in file)");
		    rc = NLX_ERRFMT;
		    break;
		}
		listcrc = (unsigned short) atol(q);
	    }

	    if (!wfile_isvalid(&wfp)) {
		logit(LOG_ERROR, "%s(%u): Invalid line format", listname, stt[LINES]);
		++stt[UNKNOWN];
	    } else if (wfile_iscomment(&wfp))
		++stt[COMMENTS];/* Don't parse */
	    else {
		if (wfile_isadmin(&wfp)) {
		    ++stt[ADMINS];
		    if (wfile_iscoord(&wfp)) {
			++stt[COORDS];
			if (wfile_iszc(&wfp))
			    printf("Zone %3u  Reg %3s  \r", wfile_zone(&wfp), "-");
			else if (wfile_isrc(&wfp))
			    printf("Zone %3u  Reg %3u  \r", wfile_zone(&wfp), wfile_net(&wfp));
			fflush(stdout);
		    }
		} else
		    ++stt[NODES];

		/*
		 * At this point, we have an apparently valid node record. We
		 * now need to parse it fully to check the validity of various
		 * fields.
		 */

	    }
	    wfile_advance(&wfp, len);
	}

	taken = clock() - start;
	p = last_component(listname);

	logit(LOG_MARK, "%s: %u line(s) scanned in %lu.%02lu seconds.", p, stt[LINES], taken / CLOCKS_PER_SEC, ((taken % CLOCKS_PER_SEC) * 100) / CLOCKS_PER_SEC);
	if (listcrc == crcval) {
	    logit(LOG_RESULT, "%s: CRC-16 (%05u) is OK", p, listcrc);
	    if (stt[UNKNOWN])
		rc = NLX_UNKNOWN;
	} else {
	    logit(LOG_ERROR, "%s: CRC mismatch - got %05u but needed %05u", p, crcval, listcrc);
	    rc = NLX_ERRCRC;
	}

	logit(LOG_RESULT, "%s: %5u comment lines", p, stt[COMMENTS]);
	logit(LOG_RESULT, "%s: %5u coordinator nodes", p, stt[COORDS]);
	logit(LOG_RESULT, "%s: %5u other administrative nodes", p, stt[ADMINS] - stt[COORDS]);
	logit(LOG_RESULT, "%s: %5u non-administrative nodes", p, stt[NODES]);
	if (stt[UNKNOWN])
	    logit(LOG_ERROR, "%s: %5u invalid lines", p, stt[UNKNOWN]);
	wfile_close(&wfp);
    }

    wfile_destroy(&wfp);

    return rc;
}

static void *
memstr(void const * n, size_t sn, void const * h, size_t sh)
{
    register void *p;
    /*
     * This is brute force, but we really don't need anything too clever
     */
    while (sn > sh && (p = strchr(n, *(char const *) h)) != NULL) {
	if (memcmp(p, h, sh) == 0)	/* Found it? */
	    return p;
	p = (char *) p + 1;	/* Move past chr match */
	sn -= ((char const *) p - (char const *) n);	/* Reduce size of mem to
							 * search */
	n = p;
    }
    return NULL;
}


static void
comment_include(CFILE * outfp, char const * filename, int year)
{
    /* Ignore it if it simply does not exist */
    if (access(filename, F_OK) == 0) {
	WFILE wfp;
	wfile_init(&wfp, filename);
	if (!wfile_open(&wfp))
	    logit(LOG_WARN, "Unable to open comment file \"%s\": %s", filename, strerror(errno));
	else {
	    int len;
	    char const *p;
	    while ((p = wfile_ptr(&wfp, &len)) != NULL) {
		int less;
		char *s;
		if (year && (s = memstr(p, len, "####", 4)) != NULL) {
		    char yr[12];
		    sprintf(yr, "%04u", year);
		    memcpy(s, yr, 4);
		}
		/*
		 * Since we want to be portable to non-DOSish ASCII
		 * environments, we can't just assume that line endings in text
		 * files are CRLF. So we need to handle those separately. We
		 * can also trim unnecessary trailing whitespace at the same
		 * time.
		 */
		for (less = 0; less < len && isspace(p[len - less - 1]);)
		    ++less;

		cfprintf(outfp, "%.*s%.*s\r\n", ((*p == ';') ? 0 : (less < len) ? 3 : 2),
			 ";S ", len - less, p);
		/* Advance to next line */
		wfile_advance(&wfp, len);
	    }
	}
	wfile_destroy(&wfp);
    }
}


static int
output_entry(CFILE * outfp, txtbuf * errs, char const * p, int len, ZNNP * ataddr, char *txtaddr)
{
    int errcount = 0;
    unsigned long result;
    char line[MAXLINE];
    while (len && isspace(p[len - 1]))
	len--;
    tb_reset(errs);		/* Reset our errors log */
    /* Let comments through unmolested, but check node lines */
    result = (*p == ';') ? 0L : check_nodeent(errs, txtaddr, (char **) &p, &len, ataddr);
    /*
     * Start each network with a comment line for easier browsing
     */
    if ((result & NE_NET) && outfp != NULL)
	cfputs(";\r\n", outfp);
    /*
     * Skip output of this record (indicates an illegality error)
     */
    if (result & NE_SKIPENT)
	++errcount;
    else {
	char const *pfx = "";
	/*
	 * Flag error entries
	 */
	if (result & NE_ERRORENT) {	/* Flag as error */
	    ++errcount;
	    pfx = ";E ";
	}
	/*
	 * And output it to the list
	 */
	if (outfp != NULL)
	    cfprintf(outfp, "%s%.*s\r\n", pfx, len, p);
    }

    /*
     * Now, let's process the errors buffer
     */
    tb_seek(errs, 0L, SEEK_SET);
    while (tb_gets(errs, line, MAXLINE - 1) != NULL) {
	char *q = line;
	int l = strlen(q);
	while (l && isspace(q[l - 1]))
	    --l;		/* Shave off our newline delimiter */
	q[l] = '\0';
	if (*q) {
	    int ch = *q++;
	    /* Propogate the error msg to the nodelist */
	    if ((nm_flag(ch) & NM_LIST) && outfp != NULL)
		cfprintf(outfp, ";E -- %s\r\n", q);
	}
    }
    return errcount;
}

static char const *nlttypes[] =
{
    "Hub", "Network", "Region", "Zone", "Composite", NULL
};

static char const *nlday[] =
{
    "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"
};

static char const *nlmon[] =
{
    "January", "February", "March", "April", "May", "June", "July",
    "August", "September", "October", "November", "December"
};

static void
write_hdr(NLCONFIG * nlc, CFILE * outfp, unsigned short crc16)
{
    struct tm *T = localtime(&nlc->publish_time);
    char year[6], dnum[6], pwd[64] = "";
    cfseek(outfp, 0L, SEEK_SET);
    sprintf(year, "%04u", T->tm_year + 1900);
    sprintf(dnum, "%03u", nlc->target);
    if (nlc->str[V_PASSWORD])
	sprintf(pwd, "@%s ", nlstring(nlc, V_PASSWORD));
    cfprintf(outfp, ";A %s%s Nodelist for %s, %s %d, %s -- Day number %s : %05u\r\n",
	     pwd, nlttypes[nlc->type], nlday[T->tm_wday], nlmon[T->tm_mon],
	     T->tm_mday, year, dnum, crc16);
    cfreset(outfp);		/* Reset the CRC accumulator */
}

static void
readxflags(char const * filename, int uflags)
{
    if (filename && *filename) {
	int (*addflag) (char const * flag) = uflags ? add_uflag : add_flag;
	FILE *fp = fopen(filename, "r");
	if (fp == NULL)
	    logit(LOG_WARN, "Can't access %sflags file \"%s\" %s (%d)", uflags ? "user" : "", filename, strerror(errno), errno);
	else {
	    int lineno = 0;
	    char tmp[128];
	    while (fgets(tmp, sizeof tmp - 1, fp) != NULL) {
		char const *p = parseword(tmp, 0);
		++lineno;
		while (p != NULL) {
		    char const *msg = NULL;
		    switch (*p) {
		    case 'U':
			msg = uflags ? "'U' prefix redundant here" : "'U' prefix reserved for user flags";
			break;
		    case 'G':
			msg = "'G' prefix reserved for internetwork gateways";
			break;
		    case 'X':
			if (strlen(p) != 2 || !isupper(p[1])) {
			    msg = "'X' prefix reserved file-request capability";
			    break;
			}
		    default:
			if (!addflag(p))
			    msg = "flag is redundant";
			break;
		    }
		    if (msg)
			logit(LOG_WARN, "%s(%d): Ignoring \"%s\" - %s", last_component(filename), lineno, p, msg);
		    p = parseword(NULL, 0);
		}
	    }
	    fclose(fp);
	}
    }
}


static int
doerrors(NLCONFIG * nlc, txtbuf * errs, subfile * sf, char const * base, int lineno, char const * textaddr)
{
    int errcount = 0;
    int pos = 0;
    char pfx[64] = "";
    char line[256];
    if (base)
	pos = (lineno) ? sprintf(pfx, "%s(%d) ", base, lineno)
		: sprintf(pfx, "%s ", base);
    if (!textaddr)
	textaddr = "";

    tb_seek(errs, 0L, SEEK_SET);
    while (tb_gets(errs, line, MAXLINE - 1) != NULL) {
	char ch = *line;
	if (ch) {
	    char const *errtype = nm_flag(ch) & NM_ERROR ? "error: " :
	    nm_flag(ch) & NM_WARNING ? "warning: " :
	    "";
	    ++errcount;
	    /* Tell submitter/operator/log about the problem */
	    if (nm_flag(*line) & NM_NOLINE)
		fb_printf(nlc, sf, nm_flag(*line), "%s %s%s", base, errtype, line + 1);
	    else
		fb_printf(nlc, sf, nm_flag(*line), "%s%s%s %s", pfx, errtype, textaddr, line + 1);
	}
    }
    return errcount;
}

static void
init_process(NLCONFIG * nlc)
{
    /*
     * Initialise the nodelist line parser
     */
    init_nlinfo(nlc->phone_parts_min,
		nlc->phone_parts_max,
		nlc->pvtdisp,
		nlstring(nlc, V_BAUDRATE),
		nlc->myaddr.zone,
		nlc->myaddr.net,
		nlc->flag_handling,
		nlc->uflag_handling,
		nlc->phone_handling);

    /*
     * Add new/additional flags & user flags
     */
    readxflags(nlstring(nlc, V_FLAG), 0);
    readxflags(nlstring(nlc, V_UFLAG), 1);
}
/* Test segment is a test run, but only on the data segment
 * No output is produced
 */

int
test_segment(NLCONFIG * nlc)
{
    int errcount = 0, warncount = 0, lineno = 0;
    sofs_t *so;
    ArrayIter iter;
    txtbuf tb;
    ZNNP ataddr;
    char textaddr[40] = "";	/* check_nodeent() formats an address for us */
    init_process(nlc);
    memset(&ataddr, 0, sizeof ataddr);
    logit(LOG_PROGRESS, "Testing local %s segment", nlttypes[nlc->type]);

    /* Errors buffer */
    tb_init(&tb, 512);

    /* Let's include our data now */
    arrayIter_init(&iter, &nlc->Data);
    while ((so = arrayIter_get(&iter, NEXT)) != NULL) {
	char const *p = nlstr(nlc, *so);
	int count = output_entry(NULL, &tb, p, strlen(p), &ataddr, textaddr);
	if (tb_size(&tb))
	    warncount += doerrors(nlc, &tb, NULL, "DATA", lineno, textaddr);
	errcount += count;
    }

    fb_printf(nlc, NULL, NM_LOG | NM_FEEDBACK | (errcount ? NM_ERROR : warncount ? NM_WARNING : 0),
	      "%d error(s) %d warning(s) in local %s segment\n\n", errcount, warncount, nlttypes[nlc->type]);

    tb_destroy(&tb);
    return errcount;
}
/* Make the current nodelist segment
 */

int
make_segment(NLCONFIG * nlc, char *targ)
{
    int rc = -1, dodiff = 0,	/* Produce a difference file? */
        errcount, warncount, allerrcount = 0, allwarncount = 0, commentno, len;
    unsigned lineno;
    char *ext;
    sofs_t *so;
    subfile *sf;
    CFILE *outfp;
    FILE *comments = NULL;
    char const *netname;
    txtbuf tb;
    ZNNP ataddr;
    ArrayIter iter;
    char textaddr[40] = "";	/* check_nodeent() formats an address for us */


    init_process(nlc);

    netname = nlstring(nlc, V_NAME);
    if (!*(ext = path_ext(build_path(targ, nlstring(nlc, V_OUTPATH), nlstring(nlc, V_OUTFILE), NULL)))) {
	++dodiff;		/* Generic output file, we can do a diff */
	add_extn(targ, nlc->target);
    }
    if ((outfp = cfopen(targ, "w+b")) == NULL)
	fatal(EL_IOERROR, "Unable to open new nodelist \"%s\"\n", targ);

    if (*netname)
	nlttypes[NL_COMPOSITE] = netname;
    else if (nlc->type == NL_COMPOSITE)
	logit(LOG_WARN, "No net NAME given for list header (use /N or NAME verb)");

    logit(LOG_PROGRESS, "Building %s nodelist \"%s\"", nlttypes[nlc->type], targ);

    /* Write nodelist header line */
    write_hdr(nlc, outfp, 0);

    if (!nlsnull(nlc, V_COMMENTS))
	comments = fopen(nlstr(nlc, V_COMMENTS), "w");

    /* Include the two optional comment headers */
    comment_include(outfp, nlstring(nlc, V_COPYRIGHT), nlc->tyear);
    comment_include(outfp, nlstring(nlc, V_PROLOG), 0);

    memset(&ataddr, 0, sizeof ataddr);

    logit(LOG_PROGRESS, "Processing local %s segment", nlttypes[nlc->type]);

    /* Errors buffer */
    tb_init(&tb, 512);

    warncount = errcount = lineno = commentno = 0;
    /* Let's include our data now */
    arrayIter_init(&iter, &nlc->Data);
    while ((so = arrayIter_get(&iter, NEXT)) != NULL) {
	char const *p = nlstr(nlc, *so);
	if (*p == ';') {
	    if (comments != NULL) {
		if (commentno++ == 0)
		    fprintf(comments, "Local segment\n");
		fprintf(comments, "%s\n", p);
	    }
	} else {
	    int count = output_entry(outfp, &tb, p, strlen(p), &ataddr, textaddr);
	    if (tb_size(&tb))
		warncount += doerrors(nlc, &tb, NULL, "DATA", lineno, textaddr);
	    errcount += count;
	}
    }
    allerrcount += errcount;
    allwarncount += warncount;	/* Includes errros+warnings! */

    fb_printf(nlc, NULL, NM_LOG | NM_FEEDBACK | (errcount ? NM_ERROR : warncount ? NM_WARNING : 0),
	      "%d error(s) %d warning(s) in local %s segment\n\n", errcount, warncount, nlttypes[nlc->type]);

    arrayIter_init(&iter, &nlc->Files);
    while ((sf = arrayIter_get(&iter, NEXT)) != NULL) {
	char tmp[_MAX_PATH];
	warncount = errcount = 0;
	build_path(tmp, nlstring(nlc, V_MASTER), nlstr(nlc, sf->name), NULL);
	if (!find_latest(tmp, tmp, F_LIST, nlc->publish_time)) {
	    fb_printf(nlc, sf, nm_error(NM_ERROR | NM_FEEDBACK),
		      "No valid %s %d nodelist segment found.\n"
		      "This means that your segment has not been included in the current %s segment and"
	     "a new complete list should be sent to %s as soon as possible.\n"
		      "== DO NOT IGNORE THIS MESSAGE ==\n",
		      nlttypes[sf->type], sf->number, nlttypes[nlc->type],
		      fidofmt(NULL, &nlc->myaddr, NULL));
	    logit(LOG_ERROR, " No nodelist segment found for %s %d (%s)", nlttypes[sf->type], sf->number, nlstr(nlc, sf->name));
	    errcount++;		/* Count this as an error */
	} else {
	    WFILE wfp;
	    char const *base = last_component(tmp);
	    wfile_init(&wfp, tmp);
	    if (!wfile_open(&wfp))
		logit(LOG_ERROR, "Unable to open input segment \"%s\": %s (%s)", tmp, strerror(errno), errno);
	    else {
		char const *p;
		commentno = lineno = 0;
		logit(LOG_PROGRESS, "Processing %-8s%-5d -- file %s", nlttypes[sf->type], sf->number, tmp);
		while ((p = wfile_ptr(&wfp, &len)) != NULL) {
		    ++lineno;
		    if (*p == ';' && *(p + 1) != 'E') {
			if (comments != NULL) {
			    if (commentno++ == 0)
				fprintf(comments, "%s\n", base);
			    fprintf(comments, "%.*s\n", len, p);
			}
		    } else {	/* Drop non-error comments */
			int count = output_entry(outfp, &tb, p, len, &ataddr, textaddr);
			if (tb_size(&tb))
			    warncount += doerrors(nlc, &tb, sf, base, lineno, textaddr);
			errcount += count;
		    }
		    wfile_advance(&wfp, len);
		}
	    }
	    wfile_destroy(&wfp);

	    /* Final checking of redundant phone numbers */
	    tb_reset(&tb);
	    len = sf->type;
	    if (check_nodeent(&tb, NULL, NULL, &len, NULL) & NE_NOTRED)
		warncount += doerrors(nlc, &tb, sf, base, 0, NULL);

	    fb_printf(nlc, sf, NM_LOG | NM_FEEDBACK | (errcount ? NM_ERROR : warncount ? NM_WARNING : 0),
		      "%s: processing completed with %d error(s) and %d warning(s)\n\n", base, errcount, warncount);
	}

	allerrcount += errcount;
	allwarncount += warncount;
    }

    /* Check for redundancies */
    tb_reset(&tb);
    len = nlc->type;
    if (check_nodeent(&tb, NULL, NULL, &len, NULL) & NE_NOTRED)
	allwarncount += doerrors(nlc, &tb, sf, "DATA", 0, NULL);

    fb_printf(nlc, NULL, NM_LOG | NM_FEEDBACK | (errcount ? NM_ERROR : warncount ? NM_WARNING : 0),
	      "%s: processing completed with %d error(s) and %d warning(s)\n\n", last_component(targ), allerrcount, allwarncount);

    tb_destroy(&tb);

    if (comments != NULL)
	fclose(comments);

    /* Add the epilog */
    comment_include(outfp, nlstring(nlc, V_EPILOG), 0);

    if (cferror(outfp))
	logit(LOG_FATAL, "Write error on \"%s\"", targ);
    else {
	unsigned short mycrc = cfcrc16(outfp);	/* We need to write this */
	cfwrite("\x1a", 1, 1, outfp);	/* Write the EOF marker first */
	write_hdr(nlc, outfp, mycrc);	/* Rewrite the header */
	rc = 0;
    }
    if (cfclose(outfp) == EOF && rc == 0) {
	logit(LOG_FATAL, "Write error on \"%s\"", targ);
	rc = -1;
    }
    nlc->errcount += allerrcount;
    return (rc == -1) ? rc : dodiff;
}


int
make_diff(NLCONFIG * nlc, char const * targ, char *diffname, int force)
{
    int weeks = 0, rc = -1;
    time_t prev = nlc->publish_time;
    struct tm *T;
    char oldlist[_MAX_PATH];
    do {
	prev = nlc->publish_time - (SECSPERDAY * 7);	/* Day number for last
							 * week's list */
	T = localtime(&prev);
	add_extn(strcpy(oldlist, targ), T->tm_yday + 1);
    } while (++weeks < 26 && (rc = access(oldlist, F_OK)) != 0);

    if (rc != 0)
	logit(LOG_WARN, "Old nodelist not found, no difference file can be made");
    else {
	short oldday, newday;
	unsigned short oldcrc16, newcrc16;

	static char const notlist[] = "\"%s\" does not appear to be a valid nodelist";
	rc = -1;
	if (nlfiletype(oldlist, &oldday, &oldcrc16, NULL) != F_LIST)
	    logit(LOG_ERROR, notlist, oldlist);
	else if (nlfiletype(targ, &newday, &newcrc16, NULL) != F_LIST)
	    logit(LOG_ERROR, notlist, targ);
	else if (!force && (oldcrc16 == newcrc16)) {
	    logit(LOG_EVENT, "Old and new lists are identical");
	    rc = 1;		/* Tell caller there's nothing to submit */
	} else {
	    char const *dname = nlstring(nlc, V_OUTDIFF);
	    if (dname == NULL || !*dname)
		dname = nlstring(nlc, V_OUTFILE);
	    if ((rc = filediff(diff_name(diffname, targ, dname), oldlist, targ)) == -1)
		logit(LOG_WARN, "Error(s) during processing, no difference file generated");
	}
    }
    return rc;
}

int
archive_list(NLCONFIG * nlc, char *targ, char const * destdir, char const * method, char const * pfx, int do_rm)
{
    int rc = -1;
    ArcDef *def = arcdef_find(&nlc->Arcs, method);
    if (def == NULL)
	logit(LOG_ERROR, "Unknown compression method \"%s\" compressing \"%s\"", method, targ);
    else {
	char const *file = targ;
	char *ext;
	char curdir[_MAX_PATH], dest[_MAX_PATH];
	getcurdir(curdir, _MAX_PATH);	/* Save current directory */
	if (changedir(get_path(NULL, targ)) != 0)
	    logit(LOG_ERROR, "Can't change to same directory to compress \"%s\"", targ);
	else
	    file = last_component(targ);	/* Effectively junk paths */
	ext = path_ext(build_path(dest, destdir, last_component(targ), NULL));
	/* Do standard substitutions if it is a generic list name */
	if (wildmatch(ext, ".###", 0))	/* All digits! */
	    ext[1] = (pfx && *pfx) ? *pfx : *method;	/* Insert archive
							 * modifier */
	else if (wildmatch(ext, ".D##", 1)) {	/* Diff with generic extension */
	    ext[2] = ext[1];	/* Shift the 'D' across */
	    ext[1] = (pfx && *pfx) ? *pfx : *method;	/* Insert the archive
							 * modifier */
	}
	 /* For non-generic lists, use the standard archive extension instead */ 
	else
	    strcpy(ext + 1, sb_string(&nlc->Arcs.strs, def->os[AV_EXT]));

	logit(LOG_PROGRESS, "Compressing \"%s\" to \"%s\"", targ, dest);

	rc = arcdef_execute(AV_ADD, &nlc->Arcs, def, dest, file);
	if (rc == 0) {		/* Compression succeeded */
	    if (do_rm)		/* Remove original file if we are asked to */
		remove(targ);
	    strcpy(targ, dest);
	}
	changedir(curdir);	/* Change back to original directory */
    }
    return rc;
}

int
find_latest(char *targ, char const * src, int type, time_t current)
{
    int rc = 0;
    int latest = 32767;
    FFIND *ff;
    char *ext;
    char tmp[_MAX_PATH];
    strcpy(tmp, last_component(src));
    if (*(ext = path_ext(tmp)) == '\0')	/* Generic list or diff? */
	strcpy(ext, ".[D0123]##");

    ff = file_findfirst(get_path(NULL, src), tmp);
    if (ff != NULL) {
	/*
	 * Find the most recent nodelist segment or difference file
	 */
	do {
	    if (file_isreg(ff)) {
		short daynum;
		unsigned short crc16;
		file_path(ff, tmp);
		if (nlfiletype(tmp, &daynum, &crc16, NULL) == type) {
		    int age = weeksold(current, daynum);
		    if (age >= 0 && age < latest) {
			latest = age;
			strcpy(targ, tmp);
			++rc;
		    }
		}
	    }
	} while (file_findnext(ff));
	file_findclose(ff);
    }
    return rc;
}

int
cleanup_lists(char const * dir, char const * file, char const * except, time_t current)
{
    int rc = 0;
    FFIND *ff;
    char *ext;
    char tmp[_MAX_PATH];
    strcpy(tmp, file);
    if (*(ext = path_ext(tmp)) == '\0')	/* Generic list or diff? */
	strcpy(ext, ".[0123]##");

    ff = file_findfirst(dir, tmp);
    if (ff != NULL) {
	/*
	 * Find the most recent nodelist segment or difference file
	 */
	do {
	    if (file_isreg(ff)) {
		short daynum;
		unsigned short crc16;
		file_path(ff, tmp);
		if (stricmp(last_component(tmp), last_component(except)) != 0 &&	/* Not exception */
		    nlfiletype(tmp, &daynum, &crc16, NULL) == F_LIST &&	/* Is a nodelist */
		    weeksold(current, daynum) > 0 && remove(tmp) == 0) {	/* And is older than
										 * this week */
		    ++rc;
		    logit(LOG_PROGRESS, "Deleted \"%s\"", tmp);
		}
	    }
	} while (file_findnext(ff));
	file_findclose(ff);
    }
    return rc;
}


/* copylist() - Copy a nodelist into a given directory
 */

int
copylist(char *src, char const * destdir)
{
    char tmp[_MAX_PATH];
    build_path(tmp, destdir, last_component(src), NULL);
    if (stricmp(tmp, src) == 0)
	return 1;		/* Pretend success if we don't need to copy it */
    if (fcopy(tmp, src, 0) == 0) {
	strcpy(src, tmp);	/* Update source buffer */
	return 1;
    }
    logit(LOG_WARN, "File copy \"%s\" => \"%s\" failed", src, destdir);
    return 0;			/* Failure */
}
